<?php
/**
 * Author:      Aspen Grove Studios
 * License: GNU General Public License version 3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.en.html
 */

/*
    Product Sales Report Pro for WooCommerce
    Copyright (C) 2021  Aspen Grove Studios

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

	=====
	
	This file contains code from the Easy Digital Downloads Software Licensing addon.
	Copyright (c) Sandhills Development, LLC; released under the GNU General Public License version 2 or later; used in this project under GNU General Public License version 3 or later (see license/LICENSE.TXT).
*/


class HM_Product_Sales_Report_Pro
{

	private static $customFieldNames, $addonFields, $orderFieldNames, $customerFieldNames;

	/*
		The following function contains code copied from from WooCommerce; see license/woocommerce-license.txt for copyright and licensing information
	*/
    public static function getReportData($wc_report, $product_ids, $startDate = null, $endDate = null, $refundOrders = false)
    {
		global $wpdb, $hm_wc_report_extra_sql;
		$hm_wc_report_extra_sql = array();
		
		$groupByProducts = empty( $_POST['disable_product_grouping'] );
		$intermediateRounding = !empty( $_POST['intermediate_rounding'] );
		
		// Based on woocoommerce/includes/admin/reports/class-wc-report-sales-by-product.php
		$dataParams = array(
			
			// Following code provided by and copyright Daniel von Mitschke, released under GNU General Public License (GPL) version 2 or later, used under GPL version 3 or later (see license/LICENSE.TXT)
			// Modified by Jonathan Hall; last modified 2020-06-04
            'order_item_name' => array(
                'type' => 'order_item',
                'function' => 'GROUP_CONCAT',
                'distinct' => true,
				'join_type' => 'LEFT',
                'name' => 'product_name'
            ),
			// End code provided by Daniel von Mitschke
			'_qty' => array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => 'SUM',
				'join_type' => 'LEFT',
				'name' => 'quantity'
			),
			'_line_subtotal' => array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => $intermediateRounding ? 'PSRSUM' : 'SUM',
				'join_type' => 'LEFT',
				'name' => 'gross'
			),
			'_line_total' => array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => $intermediateRounding ? 'PSRSUM' : 'SUM',
				'join_type' => 'LEFT',
				'name' => 'gross_after_discount'
			),
			'_line_tax' => array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => $intermediateRounding ? 'PSRSUM' : 'SUM',
				'join_type' => 'LEFT',
				'name' => 'taxes'
			)
		);
		
		if ( $groupByProducts ) {
			$dataParams['_product_id'] = array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => '',
				'join_type' => 'LEFT',
				'name' => 'product_id'
			);
		}
		
		if (!empty($_POST['variations']) && $groupByProducts) {
			$dataParams['_variation_id'] = array(
				'type' => 'order_item_meta',
				'order_item_type' => 'line_item',
				'function' => '',
				'join_type' => 'LEFT',
				'name' => 'variation_id'
			);
		}
		if ( in_array('builtin::line_item_count', $_POST['fields']) ) {
			$dataParams['order_item_id'] = array(
				'type' => 'order_item',
				'order_item_type' => 'line_item',
				'function' => 'COUNT',
				'join_type' => 'LEFT',
				'name' => 'line_item_count'
			);
		}
		foreach ($_POST['fields'] as $field) {
			if (substr($field, 0, 18) == 'order_item_total::') {
				$fieldName = str_replace(array(' ', '-'), '', esc_sql(substr($field, 18))); // Remove spaces for security
				
				$dataParams[$fieldName] = array(
					'type' => 'order_item_meta',
					'order_item_type' => 'line_item',
					'function' => 'SUM',
					'join_type' => 'LEFT',
					'name' => 'order_item_total__'.$fieldName
				);
			} else if ($field == 'builtin::groupby_field' || $field == 'builtin::groupby_field2' || $field == 'builtin::groupby_field3' ) {
				
				$groupbyFieldNum = $field == 'builtin::groupby_field' ? '' : $field[22];
				
				if ( !empty($_POST['groupby'.$groupbyFieldNum]) && $_POST['groupby'.$groupbyFieldNum] != 'i_builtin::item_price' ) {
					
					if (in_array($_POST['groupby'.$groupbyFieldNum], array('o_builtin::order_month', 'o_builtin::order_quarter', 'o_builtin::order_year', 'o_builtin::order_date', 'o_builtin::order_day'))) {
						switch ($_POST['groupby'.$groupbyFieldNum]) {
							case 'o_builtin::order_month':
								$sqlFunction = 'MONTH';
								break;
							case 'o_builtin::order_quarter':
								$sqlFunction = 'QUARTER';
								break;
							case 'o_builtin::order_year':
								$sqlFunction = 'YEAR';
								break;
							case 'o_builtin::order_day':
								$sqlFunction = 'DAY';
								break;
							default:
								$sqlFunction = 'DATE';
						}
						$dataParams['post_date'] = array(
							'type' => 'post_data',
							'order_item_type' => 'line_item',
							'function' => $sqlFunction,
							'join_type' => 'LEFT',
							'name' => 'groupby_field'.$groupbyFieldNum
						);
					} else {
						$fieldName = esc_sql(substr($_POST['groupby'.$groupbyFieldNum], 2));
						$dataParams[$fieldName] = array(
							'type' => ($_POST['groupby'.$groupbyFieldNum][0] == 'i' ? 'order_item_meta' : 'meta'),
							'order_item_type' => 'line_item',
							'function' => '',
							'join_type' => 'LEFT',
							'name' => 'groupby_field'.$groupbyFieldNum
						);
						
					}
					
				}
			}
		}
		
		$where = array();
		$where_meta = array();
		if ($product_ids != null) {
			// If there are more than 10,000 product IDs, they should not be filtered in the SQL query
			if (count($product_ids) > 10000) {
				$productIdsPostFilter = true;
			} else {
				$where_meta[] = array(
					'type' => 'order_item_meta',
					'meta_key' => '_product_id',
					'operator' => 'IN',
					'meta_value' => $product_ids
				);
			}
		}
		if (!empty($_POST['exclude_free'])) {
			$where_meta[] = array(
				'meta_key' => '_line_total',
				'meta_value' => 0,
				'operator' => '!=',
				'type' => 'order_item_meta'
			);
		}
		
		// Order meta field filtering
		if ( !empty($_POST['order_meta_filter_on']) || !empty($_POST['order_meta_filter_2_on']) ) {
			$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').self::getOrderFieldFilterSql($refundOrders);
		}
		
		$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').self::getOrderItemFieldFilterSql();
		
		// Customer meta filtering
		$customerMetaFilterSql = self::getCustomerMetaFilterSql($refundOrders);
		if (!empty($customerMetaFilterSql)) {
			$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').$customerMetaFilterSql;
		}
		
		
		
		if ($_POST['report_time'] == 'custom') {
			$where[] = array(
				'key' => 'post_date',
				'operator' => '>=',
				'value' => date('Y-m-d H:i:s', $startDate)
			);
			$where[] = array(
				'key' => 'post_date',
				'operator' => '<',
				'value' => date('Y-m-d H:i:s', $endDate)
			);
		}
		
		$groupBy = [];
		
		if ($groupByProducts) {
			$groupBy[] = 'product_id';
			if (!empty($_POST['variations'])) {
				$groupBy[] = 'variation_id';
			}
		}
		
		foreach ( ['groupby', 'groupby2', 'groupby3'] as $groupbyDropdownField ) {
			if (!empty($_POST[$groupbyDropdownField])) {
				switch ($_POST[$groupbyDropdownField]) {
					case 'i_builtin::item_price':
						$groupBy[] = 'ROUND(order_item_meta__line_subtotal.meta_value / order_item_meta__qty.meta_value, 2)';
						break;
					default:
						$groupBy[] = 'groupby_field'.( $groupbyDropdownField == 'groupby' ? '' : $groupbyDropdownField[7] );
				}
			}
		}
		
		// Address issue with order_items JOIN with order_item_type being overridden
		foreach ($dataParams as $fieldKey => $field) {
			if ($field['type'] == 'order_item_meta' && isset($field['order_item_type'])) {
				unset($dataParams[$fieldKey]);
				$dataParams[$fieldKey] = $field; // move this key to the end of the array
				break;
			}
		}
		
		$reportOptions = array(
			'data' => $dataParams,
			'nocache' => true,
			'query_type' => 'get_results',
			'group_by' => implode(',', $groupBy),
			'filter_range' => ($_POST['report_time'] != 'all' && $_POST['report_time'] != 'custom'),
			'order_types' => array($refundOrders ? 'shop_order_refund' : 'shop_order'),
			/*'order_status' => $orderStatuses,*/ // Order status filtering is set via filter
			'where_meta' => $where_meta
		);
		
		if (!empty($_POST['hm_psr_debug'])) {
			$reportOptions['debug'] = true;
		}
		
		if (!empty($where)) {
			$reportOptions['where'] = $where;
		}
		
		// Order status filtering
		// Order status array has been sanitized and checked non-empty in hm_sbp_export_body() function
		$statusesStr = '';
		foreach ($_POST['order_statuses'] as $i => $orderStatus) {
			$statusesStr .= ($i ? ',\'' : '\'').esc_sql($orderStatus).'\'';
		}
		$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').' AND posts.post_status'.
			($refundOrders ? '=\'wc-completed\' AND EXISTS(SELECT 1 FROM '.$wpdb->posts.' WHERE ID=posts.post_parent AND post_status IN('.$statusesStr.'))' :
			' IN('.$statusesStr.')');
		
		@$wpdb->query('SET SESSION sort_buffer_size='.absint($_POST['db_sort_buffer_size']*1000));
		
		//ob_start();
		$result = $wc_report->get_order_report_data($reportOptions);
		//file_put_contents(__DIR__.'/debug.log', ob_get_clean());
		
		// Do post-query product ID filtering, if necessary
		if (!empty($result) && !empty($productIdsPostFilter)) {
			foreach ($result as $key => $product) {
				if (!in_array($product->product_id, $product_ids)) {
					unset($result[$key]);
				}
			}
		}
		
		return $result;
	}
	
	/*
		The following function contains code copied from from WooCommerce; see license/woocommerce-license.txt for copyright and licensing information
	*/
    public static function getShippingReportData($wc_report, $startDate, $endDate, $taxes = false, $refundOrders = false)
    {
		global $wpdb, $hm_wc_report_extra_sql;
		$hm_wc_report_extra_sql = array();
		
		$groupByProducts = empty( $_POST['disable_product_grouping'] );
		$intermediateRounding = !empty( $_POST['intermediate_rounding'] );
	
		// Based on woocoommerce/includes/admin/reports/class-wc-report-sales-by-product.php
		
		$dataParams = array(
			'cost' => array(
				'type' => 'order_item_meta',
				'order_item_type' => 'shipping',
				'function' => $intermediateRounding ? 'PSRSUM' : 'SUM',
				'join_type' => 'LEFT',
				'name' => 'gross'
			)
		);
		if ($groupByProducts) {
			$dataParams['method_id'] = array(
				'type' => 'order_item_meta',
				'order_item_type' => 'shipping',
				'function' => '',
				'join_type' => 'LEFT',
				'name' => 'product_id'
			);
		}
		if ( !$refundOrders || in_array('builtin::line_item_count', $_POST['fields']) ) {
			$dataParams['order_item_id'] = array(
				'type' => 'order_item',
				'order_item_type' => 'shipping',
				'function' => 'COUNT',
				'join_type' => 'LEFT',
				'name' => 'quantity'
			);
		}
		
		foreach ($_POST['fields'] as $field) {
			if ($field == 'builtin::groupby_field' || $field == 'builtin::groupby_field2' || $field == 'builtin::groupby_field3') {
				
				$groupbyFieldNum = $field == 'builtin::groupby_field' ? '' : $field[22];
				
				if (!empty($_POST['groupby'.$groupbyFieldNum]) && $_POST['groupby'.$groupbyFieldNum] != 'i_builtin::item_price') {
					if (in_array($_POST['groupby'.$groupbyFieldNum], array('o_builtin::order_month', 'o_builtin::order_quarter', 'o_builtin::order_year', 'o_builtin::order_date', 'o_builtin::order_day'))) {
						switch ($_POST['groupby'.$groupbyFieldNum]) {
							case 'o_builtin::order_month':
								$sqlFunction = 'MONTH';
								break;
							case 'o_builtin::order_quarter':
								$sqlFunction = 'QUARTER';
								break;
							case 'o_builtin::order_year':
								$sqlFunction = 'YEAR';
								break;
							case 'o_builtin::order_day':
								$sqlFunction = 'DAY';
								break;
							default:
								$sqlFunction = 'DATE';
						}
						$dataParams['post_date'] = array(
							'type' => 'post_data',
							'order_item_type' => 'shipping',
							'function' => $sqlFunction,
							'join_type' => 'LEFT',
							'name' => 'groupby_field'.$groupbyFieldNum
						);
					} else {
						$fieldName = esc_sql(substr($_POST['groupby'.$groupbyFieldNum], 2));
						$dataParams[$fieldName] = array(
							'type' => ($_POST['groupby'.$groupbyFieldNum][0] == 'i' ? 'order_item_meta' : 'meta'),
							'order_item_type' => 'shipping',
							'function' => '',
							'join_type' => 'LEFT',
							'name' => 'groupby_field'.$groupbyFieldNum
						);
					}
					break;
				}
			}
		}
		
		// Order meta field filtering
		$where_meta = array();
		if (!empty($_POST['order_meta_filter_on'])) {
			$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').self::getOrderFieldFilterSql($refundOrders);
		}
		
		// Customer meta filtering
		$customerMetaFilterSql = self::getCustomerMetaFilterSql($refundOrders);
		if (!empty($customerMetaFilterSql)) {
			$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').$customerMetaFilterSql;
		}
		
		if (!empty($_POST['customer_role'])) {
			/*$where_meta[] = array(
				'type' => 'order_meta',
				'meta_key' => '_customer_user',
				'operator' => ($_POST['customer_role'] == -1 ? '=' : 'IN'),
				'meta_value' => ($_POST['customer_role'] == -1 ? 0 : get_users(array('role' => esc_sql($_POST['customer_role']), 'fields' => 'ID')))
			);*/
			
			if ($_POST['customer_role'] != -1) {
				$userIds = get_users(array('role' => esc_sql($_POST['customer_role']), 'fields' => 'ID'));
			}
			$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').' AND EXISTS(
				SELECT 1 FROM '.$wpdb->postmeta.' WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').' AND meta_key=\'_customer_user\' AND '.($_POST['customer_role'] == -1 ? 'meta_value = 0' : (empty($userIds) ? 'FALSE' : 'meta_value IN ('.implode(',', $userIds).')')).')';
		}
		
		$groupBy = [];
		
		if ($groupByProducts) {
			$groupBy[] = 'product_id';
		}
		
		foreach ( ['groupby', 'groupby2', 'groupby3'] as $groupbyDropdownField ) {
			if (!empty($_POST[$groupbyDropdownField])) {
				switch ($_POST[$groupbyDropdownField]) {
					case 'i_builtin::item_price':
						$groupBy[] = '(order_item_meta_cost.meta_value * 1)';
						break;
					default:
						$groupBy[] = 'groupby_field'.( $groupbyDropdownField == 'groupby' ? '' : $groupbyDropdownField[7] );
				}
			}
		}
		
		// Address issue with order_items JOIN with order_item_type being overridden
		foreach ($dataParams as $fieldKey => $field) {
			if ($field['type'] == 'order_item_meta' && isset($field['order_item_type'])) {
				unset($dataParams[$fieldKey]);
				$dataParams[$fieldKey] = $field; // move this key to the end of the array
				break;
			}
		}
		
		$reportParams = array(
			'data' => $dataParams,
			'nocache' => true,
			'query_type' => 'get_results',
			'group_by' => implode(',', $groupBy),
			'filter_range' => ($_POST['report_time'] != 'all' && $_POST['report_time'] != 'custom'),
			'order_types' => array($refundOrders ? 'shop_order_refund' : 'shop_order'),
			'where_meta' => $where_meta
		);
		
		if (!empty($_POST['hm_psr_debug'])) {
			$reportParams['debug'] = true;
		}
		
		if ($_POST['report_time'] == 'custom') {
			$reportParams['where'] = array(
				array(
					'key' => 'post_date',
					'operator' => '>=',
					'value' => date('Y-m-d H:i:s', $startDate)
				),
				array(
					'key' => 'post_date',
					'operator' => '<',
					'value' => date('Y-m-d H:i:s', $endDate)
				)
			);
		}
		
		// Order status filtering
		// Order status array has been sanitized and checked non-empty in hm_sbp_export_body() function
		$statusesStr = '';
		foreach ($_POST['order_statuses'] as $i => $orderStatus) {
			$statusesStr .= ($i ? ',\'' : '\'').esc_sql($orderStatus).'\'';
		}
		$hm_wc_report_extra_sql['where'] = (isset($hm_wc_report_extra_sql['where']) ? $hm_wc_report_extra_sql['where'] : '').' AND posts.post_status'.
			($refundOrders ? '=\'wc-completed\' AND EXISTS(SELECT 1 FROM '.$wpdb->posts.' WHERE ID=posts.post_parent AND post_status IN('.$statusesStr.'))' :
			' IN('.$statusesStr.')');
		
		$result = $wc_report->get_order_report_data($reportParams);
		
		if ($refundOrders) {
			foreach ($result as $shipping) {
				$shipping->quantity = 0;
			}
		}
		
		if ($taxes) {
			
			$hasShippingItemClass = class_exists('WC_Order_Item_Shipping'); // WC 3.0+
			
			$reportParams['data'] = array(
				'method_id' => array(
					'type' => 'order_item_meta',
					'order_item_type' => 'shipping',
					'function' => '',
					'name' => 'product_id'
				)
			);
			if ($hasShippingItemClass) {
				$reportParams['data']['order_item_id'] = array(
					'type' => 'order_item',
					'order_item_type' => 'shipping',
					'function' => '',
					'name' => 'order_item_id'
				);
			} else {
				$reportParams['data']['taxes'] = array(
					'type' => 'order_item_meta',
					'order_item_type' => 'shipping',
					'function' => '',
					'name' => 'taxes'
				);
			}
			$reportParams['group_by'] = '';
			$taxResult = $wc_report->get_order_report_data($reportParams);
			
			foreach ($result as $shipping) {
				$shipping->taxes = 0;
				foreach ($taxResult as $i => $taxes) {
					if ($taxes->product_id == $shipping->product_id) {
						if ($hasShippingItemClass) {
							$oi = new WC_Order_Item_Shipping($taxes->order_item_id);
							$shipping->taxes += $oi->get_total_tax();
						} else {
							$taxArray = @unserialize($taxes->taxes);
							if (!empty($taxArray)) {
								foreach ($taxArray as $taxItem) {
									$shipping->taxes += $taxItem;
								}
							}
						}
						unset($taxResult[$i]);
					}
				}
			}
		}
		
		return $result;
		
	}
	
    public static function getCustomerMetaFilterSql($refundOrders)
    {
		global $wpdb;
		
		$sql = '';
		
		// Customer role filtering
		
		if (!empty($_POST['customer_role'])) {
			if ( $_POST['customer_role'] == -1 ) {
				$sql .= ' AND EXISTS(
							SELECT 1 FROM '.$wpdb->postmeta.'
							WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
								AND meta_key=\'_customer_user\'
								AND meta_value = 0
						)';
			} else {
				$sql .= ' AND EXISTS(
							SELECT 1 FROM '.$wpdb->postmeta.' pm
							JOIN '.$wpdb->usermeta.' um ON (um.user_id=pm.meta_value AND um.meta_key="'.$wpdb->prefix.'capabilities")
							WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
								AND pm.meta_key=\'_customer_user\'
								AND um.meta_value LIKE \'%"'.esc_sql($wpdb->esc_like($_POST['customer_role'])).'"%\')';
			}
		}
		
		// Customer meta filtering
		
		if (!empty($_POST['customer_meta_filter_on'])) {
			if (in_array($_POST['customer_meta_filter_op'], array('=', '!=', '<', '<=', '>', '>=', 'BETWEEN'))) {
			
				$numericValue =
					is_numeric($_POST['customer_meta_filter_value'])
					&& (
						$_POST['customer_meta_filter_op'] != 'BETWEEN'
						|| is_numeric($_POST['customer_meta_filter_value_2'])
					);
				
				$metaValue = ($numericValue ? $_POST['customer_meta_filter_value'] : '\''.esc_sql($_POST['customer_meta_filter_value']).'\'');
				if ($_POST['customer_meta_filter_op'] == 'BETWEEN') {
					if ($numericValue) {
						$metaValue .= ' AND '.$_POST['customer_meta_filter_value_2'];
					} else {
						$metaValue .= ' AND \''.esc_sql($_POST['customer_meta_filter_value_2']).'\'';
					}
				}
				
				$sql .= ' AND EXISTS(
					SELECT 1 FROM '.$wpdb->postmeta.'
					WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
						AND meta_key=\'_customer_user\'
						AND meta_value IN (
							SELECT user_id FROM '.$wpdb->usermeta.'
							WHERE meta_key=\''.esc_sql($_POST['customer_meta_filter_key']).'\'
								AND meta_value'.($numericValue ? '*1' : '').' '.$_POST['customer_meta_filter_op'].' '.$metaValue.'
						)
				)';
					
			} else if ($_POST['customer_meta_filter_op'] == 'NOTEXISTS') {
				$sql .= ' AND NOT EXISTS(
					SELECT 1 FROM '.$wpdb->postmeta.'
					WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
						AND meta_key=\'_customer_user\'
						AND meta_value NOT IN (
							SELECT user_id FROM '.$wpdb->usermeta.'
							WHERE meta_key=\''.esc_sql($_POST['customer_meta_filter_key']).'\'
						)
				)';
			}
		}
		
		return $sql;
	}
	
    public static function getFormattedVariationAttributes($product)
    {
		if (is_numeric($product)) {
			$varId = $product;
		} else if (empty($product->variation_id)) {
			return '';
		} else {
			$varId = $product->variation_id;
		}
		if (function_exists('wc_get_product_variation_attributes')) {
			$attr = wc_get_product_variation_attributes($varId);
		} else {
			$product = wc_get_product($varId);
			if (empty($product))
				return '';
			$attr = $product->get_variation_attributes();
		}
		foreach ($attr as $i => $v) {
			if ($v === '')
				unset($attr[$i]);
		}
		return implode(', ', $attr);
	}
	
    public static function getCustomFieldNames($includeDisplay = false, $productFieldsOnly = false)
    {
		global $wpdb;
		if (!isset(HM_Product_Sales_Report_Pro::$customFieldNames) || $productFieldsOnly) {
			$customFields = $wpdb->get_col('SELECT DISTINCT meta_key FROM (
												SELECT meta_key
												FROM '.$wpdb->prefix.'postmeta
												JOIN '.$wpdb->prefix.'posts ON (post_id=ID)
												WHERE post_type="product"
												ORDER BY ID DESC
												LIMIT 10000
											) fields', 0);
			if ($productFieldsOnly) {
				foreach (get_object_taxonomies('product') as $taxonomy) {
					if ($taxonomy != 'product_cat' && $taxonomy != 'product_tag') {
						$customFields[] = 'taxonomy::'.$taxonomy;
					}
				}
				return $customFields;
			}
			
			HM_Product_Sales_Report_Pro::$customFieldNames = array(
				'Product' => array_combine($customFields, $customFields),
				'Product Taxonomies' => array(),
				'Product Variation' => array(),
				'Order Item' => array()
			);
			
			foreach (get_object_taxonomies('product') as $taxonomy) {
				HM_Product_Sales_Report_Pro::$customFieldNames['Product Taxonomies']['taxonomy::'.$taxonomy] = $taxonomy;
			}
			
			$variationFields = $wpdb->get_col('SELECT DISTINCT meta_key FROM (
													SELECT meta_key
													FROM '.$wpdb->prefix.'postmeta
													JOIN '.$wpdb->prefix.'posts ON (post_id=ID)
													WHERE post_type="product_variation"
													ORDER BY ID DESC
													LIMIT 10000
												) fields', 0);
			foreach ($variationFields as $variationField)
				HM_Product_Sales_Report_Pro::$customFieldNames['Product Variation']['variation::'.$variationField] = 'Variation '.$variationField;
			
			$skipOrderItemFields = array('_product_id', '_qty', '_line_subtotal', '_line_total', '_line_tax', '_variation_id', '_line_tax_data', '_tax_class', '_refunded_item_id');
			$orderItemFields = $wpdb->get_col('SELECT DISTINCT meta_key FROM (
													SELECT meta_key
													FROM '.$wpdb->prefix.'woocommerce_order_itemmeta
													JOIN '.$wpdb->prefix.'woocommerce_order_items USING (order_item_id)
													WHERE order_item_type=\'line_item\'
													ORDER BY order_item_id DESC
													LIMIT 10000
												) fields', 0);
			foreach ($orderItemFields as $orderItemField) {
				if (!in_array($orderItemField, $skipOrderItemFields) && !empty($orderItemField) && strpos($orderItemField, ' ') === false && strpos($orderItemField, '-') === false) {
					HM_Product_Sales_Report_Pro::$customFieldNames['Order Item']['order_item_total::'.$orderItemField] = 'Total Order Item '.$orderItemField;
				}
			}
		}
		return HM_Product_Sales_Report_Pro::$customFieldNames;
	}
	
	/*
		Get fields added by other plugins.
		Plugins hooked to "hm_psr_addon_fields" must add their fields to the array in the following format:
			my_addon_field_id => array(
				'label' => 'My Addon Field',
				'cb' => my_callback_function
			);
		where "my_callback_function" takes the following arguments:
			$product: the product object returned by $wc_report->get_order_report_data(), or the product ID for zero-sales products
			$type: null for regular products, 'shipping' for shipping items, or 'nil' for zero-sales products
			$nilVariationId: for zero sales products only, the ID of the product variation if the current row is for a product variation. Otherwise null.
		and returns the field value to include in the report for the given product.
	*/
    public static function getAddonFields()
    {
		if (!isset(HM_Product_Sales_Report_Pro::$addonFields)) {
			HM_Product_Sales_Report_Pro::$addonFields = apply_filters('hm_psr_addon_fields', array());
		}
		return HM_Product_Sales_Report_Pro::$addonFields;
	}
	
    public static function getOrderFieldNames()
    {
		global $wpdb;
		if (!isset(HM_Product_Sales_Report_Pro::$orderFieldNames)) {
			HM_Product_Sales_Report_Pro::$orderFieldNames = $wpdb->get_col('
					SELECT DISTINCT meta_key FROM (
						SELECT meta_key
						FROM '.$wpdb->prefix.'postmeta
						JOIN '.$wpdb->prefix.'posts ON (post_id=ID)
						WHERE post_type="shop_order"
						ORDER BY ID DESC
						LIMIT 10000
					) fields', 0);
		}
		return HM_Product_Sales_Report_Pro::$orderFieldNames;
	}
	
    public static function getCustomerFieldNames()
    {
		global $wpdb;
		if (!isset(HM_Product_Sales_Report_Pro::$customerFieldNames)) {
			HM_Product_Sales_Report_Pro::$customerFieldNames = $wpdb->get_col('
					SELECT DISTINCT meta_key FROM (
						SELECT meta_key
						FROM '.$wpdb->usermeta.'
						ORDER BY user_id DESC
						LIMIT 10000
					) fields', 0);
		}
		return HM_Product_Sales_Report_Pro::$customerFieldNames;
	}
	
    public static function getOrderFieldFilterSql($refundOrders)
    {
		global $wpdb;
		
		$sql = '';
		
		for ($i = 1; $i < 3; ++$i) {
			
			$fieldPre = 'order_meta_filter_'.($i == 1 ? '' : $i.'_');
			
			if ( empty($_POST[$fieldPre.'on']) ) {
				continue;
			}
			
			if ($sql) {
				$sql .= $_POST[$fieldPre.'logic'] == 'OR' ? ' OR ' : ' AND ';
			}
			
			if (in_array($_POST[$fieldPre.'op'], array('=', '!=', '<', '<=', '>', '>=', 'BETWEEN'))) {
			
				$numericValue =
					empty($_POST[$fieldPre.'value_dynamic'])
					&& is_numeric($_POST[$fieldPre.'value'])
					&& (
						$_POST[$fieldPre.'op'] != 'BETWEEN'
						|| (
							empty($_POST[$fieldPre.'value_2_dynamic'])
							&& is_numeric($_POST[$fieldPre.'value_2'])
						)
					);
			
				if (!empty($_POST[$fieldPre.'value_dynamic'])) {
					$metaValue = '\''.esc_sql( self::resolveDynamicDate( $_POST[$fieldPre.'value_dynamic'], true, $_POST['use_wp_date'] ) ).'\'';
				} else if ($numericValue) {
					$metaValue = $_POST[$fieldPre.'value'];
				} else {
					$metaValue = '\''.esc_sql($_POST[$fieldPre.'value']).'\'';
				}
				
				if ($_POST[$fieldPre.'op'] == 'BETWEEN') {
					
					if (!empty($_POST[$fieldPre.'value_2_dynamic'])) {
						$metaValue .= ' AND \''.esc_sql( self::resolveDynamicDate( $_POST[$fieldPre.'value_2_dynamic'], true, $_POST['use_wp_date'] ) ).'\'';
					} else if ($numericValue) {
						$metaValue .= ' AND '.$_POST[$fieldPre.'value_2'];
					} else {
						$metaValue .= ' AND \''.esc_sql($_POST[$fieldPre.'value_2']).'\'';
					}
				}
				
				
				$sql .= 'EXISTS(
							SELECT 1 FROM '.$wpdb->postmeta.'
							WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
									AND meta_key=\''.esc_sql($_POST[$fieldPre.'key']).'\'
									AND meta_value'.($numericValue ? '*1' : '').' '.$_POST[$fieldPre.'op'].' '.$metaValue.'
						)';
				
			} else if ($_POST['order_meta_filter_op'] == 'NOTEXISTS') {
				
				$sql .= 'NOT EXISTS(
							SELECT 1 FROM '.$wpdb->postmeta.'
							WHERE post_id=posts.'.($refundOrders ? 'post_parent' : 'ID').'
									AND meta_key=\''.esc_sql($_POST[$fieldPre.'key']).'\')';
				
			}
			
		}
		
		return ' AND ( '.$sql.' )';
	}
	
    public static function getOrderItemFieldFilterSql()
    {
		global $wpdb;
		
		$sql = [];
		
		for ($i = 1; $i < 3; ++$i) {
				
			if (!empty($_POST['order_item_meta_filter_'.$i.'_on'])) {
				
				if (in_array($_POST['order_item_meta_filter_'.$i.'_op'], array('=', '!=', '<', '<=', '>', '>=', 'BETWEEN'))) {
				
					$numericValue =
						empty($_POST['order_item_meta_filter_'.$i.'_value_dynamic'])
						&& is_numeric($_POST['order_item_meta_filter_'.$i.'_value'])
						&& (
							$_POST['order_item_meta_filter_'.$i.'_op'] != 'BETWEEN'
							|| (
								empty($_POST['order_item_meta_filter_'.$i.'_value_2_dynamic'])
								&& is_numeric($_POST['order_item_meta_filter_'.$i.'_value_2'])
							)
						);
				
					if (!empty($_POST['order_item_meta_filter_'.$i.'_value_dynamic'])) {
						$metaValue = '\''.esc_sql( self::resolveDynamicDate( $_POST['order_item_meta_filter_'.$i.'_value_dynamic'], true, $_POST['use_wp_date'] ) ).'\'';
					} else if ($numericValue) {
						$metaValue = $_POST['order_item_meta_filter_'.$i.'_value'];
					} else {
						$metaValue = '\''.esc_sql($_POST['order_item_meta_filter_'.$i.'_value']).'\'';
					}
					
					if ($_POST['order_item_meta_filter_'.$i.'_op'] == 'BETWEEN') {
						
						if (!empty($_POST['order_item_meta_filter_'.$i.'_value_2_dynamic'])) {
							$metaValue .= ' AND \''.esc_sql( self::resolveDynamicDate( $_POST['order_item_meta_filter_'.$i.'_value_2_dynamic'], true, $_POST['use_wp_date'] ) ).'\'';
						} else if ($numericValue) {
							$metaValue .= ' AND '.$_POST['order_item_meta_filter_'.$i.'_value_2'];
						} else {
							$metaValue .= ' AND \''.esc_sql($_POST['order_item_meta_filter_'.$i.'_value_2']).'\'';
						}
					}
					
					
					$sql[] = 'EXISTS(
								SELECT 1 FROM '.$wpdb->prefix.'woocommerce_order_itemmeta
								WHERE order_item_id=order_items.order_item_id
										AND meta_key=\''.esc_sql($_POST['order_item_meta_filter_'.$i.'_key']).'\'
										AND meta_value'.($numericValue ? '*1' : '').' '.$_POST['order_item_meta_filter_'.$i.'_op'].' '.$metaValue.'
							)';
					
				} else if ($_POST['order_item_meta_filter_'.$i.'_op'] == 'NOTEXISTS') {
					
					$sql[] = 'NOT EXISTS(
								SELECT 1 FROM '.$wpdb->prefix.'woocommerce_order_itemmeta
								WHERE order_item_id=order_items.order_item_id
										AND meta_key=\''.esc_sql($_POST['order_item_meta_filter_'.$i.'_key']).'\'
							)';
					
				}
			}
			
		}
		
		return $sql ? ' AND ('.implode($_POST['order_item_meta_filter_2_logic'] == 'OR' ? ' OR ' : ' AND ', $sql).')' : '';
	}
	
    public static function resolveDynamicDate($value, $allowFormat = false, $useWpDate = false)
    {
		
		$pipePos = strpos($value, '|');
		
		if ( $allowFormat && $pipePos !== false ) {
			if ( $pipePos ) {
				$dateValue = trim( substr($value, 0, $pipePos) );
			}
			if ( strlen($value) > $pipePos + 1 ) {
				$format = $pipePos ? trim( substr($value, $pipePos + 1) ) : '';
			}
		} else {
			$dateValue = trim($value);
		}
		
		if ( empty($dateValue) ) {
			$dateValue = current_time('timestamp');
		} else {
			$dateValue = strtotime($dateValue, current_time('timestamp'));
			if ( empty($dateValue) ) {
				return false;
			}
		}
		
		if ( empty($format) ) {
			$format = 'Y-m-d';
		}
		
		return
			$useWpDate
			? wp_date(
				$format,
				$dateValue,
				new DateTimeZone('UTC')
			)
			: date(
				$format,
				$dateValue
			);
	}
	
    public static function loadPresetField($savedReportSettings)
    {
		echo('<form action="" method="post">
				<table class="form-table">
					<tr valign="top">
						<th scope="row">
							<label for="hm_sbp_field_report_preset">Load Preset:</label>
						</th>
						<td>
							<select name="r" id="hm_sbp_field_report_preset">');
		foreach ($savedReportSettings as $i => $reportPreset) {
			echo('<option value="'.$i.'"'.(isset($_POST['r']) && $_POST['r'] == $i ? ' selected="selected"' : '').'>'.($i == 0 ? '[Last used settings]' : htmlspecialchars($reportPreset['preset_name'])).'</option>');
		}
		echo('				</select>
							<button type="submit" name="op" value="preset-del" class="button-primary">Delete Preset</button>
						</td>
					</tr>
				</table>
			</form>');
	}
	
    public static function savePresetField()
    {
		echo('<table class="form-table">
				<tr valign="top">
					<th scope="row">
						<label for="hm_sbp_field_save_preset">Create New Preset:</label>
					</th>
					<td>
						<input type="text" name="save_preset" id="hm_sbp_field_save_preset" placeholder="Preset Name" />
					</td>
				</tr>
			</table>');
	}
	
	/*public static function savePreset(&$savedReportSettings) {
		if (!empty($_POST['save_preset']))
			$savedReportSettings[] = array_merge($savedReportSettings[0], array('preset_name' => strip_tags($_POST['save_preset'])));
	}*/
	
    // Following code copied from Easy Digital Downloads Software Licensing addon - see comment near the top of this file for details
    // Modified by Jonathan Hall, Dominika Rauk, and possibly others
    public static function licenseCheck()
    {
		if ( isset( $_POST['hm_psr_license_deactivate'] )) {
			HM_Product_Sales_Report_Pro::deactivateLicense();
		}
	
		if (get_option('hm_psr_license_status', 'invalid') == 'valid') {
			return true;
		} else {
			if( isset( $_POST['hm_psr_license_activate'] ) && !empty($_POST['hm_psr_license_key']) && ctype_alnum($_POST['hm_psr_license_key']) ) {
				update_option('hm_psr_license_key', trim($_POST['hm_psr_license_key']));
				HM_Product_Sales_Report_Pro::activateLicense();
				if (get_option('hm_psr_license_status', 'invalid') == 'valid')
					return true;
			}
            return false;
        }
    }

    public static function licenseActivateForm()
    { ?>

        <form action="" method="post" id="ags-psr_license_key_form">
            <?php wp_nonce_field('hm_psr_license_activate_nonce', 'hm_psr_license_activate_nonce'); ?>
            <div id="ags-psr_license_key_form_logo_container">
                <a href="https://aspengrovestudios.com/aspen-grove-studios-acquires-wordpress-and-woocommerce-plugin-agency-potent-plugins/"
                   target="_blank"><img src="<?php echo(esc_url(plugins_url('images/ags-logo.png', __FILE__))); ?>"
                                        alt="Aspen Grove Studios"/></a>
            </div>
            <div id="ags-psr_license_key_form_body">
                <h3>
                    <?php echo HM_PSR_ITEM_NAME; ?>
                    <small>v<?php echo HM_PSR_VERSION; ?></small>
                </h3>
                <p>Please enter the license key provided when you purchased the plugin:</p>
                <label for="hm_psr_license_activate">
                    <span>License key:</span>
				<input type="text" id="hm_psr_license_key" name="hm_psr_license_key" />
                </label>
                <p class="submit">
                    <button type="submit" name="hm_psr_license_activate" value="1">
                        Activate License
                    </button>
                </p>
		</div>
    </form>

<?php
	}
	
    public static function activateLicense()
    {
	
		// run a quick security check
		if( ! check_admin_referer( 'hm_psr_license_activate_nonce', 'hm_psr_license_activate_nonce' ) )
			return; // get out if we didn't click the Activate button

		// retrieve the license
		$license = trim( get_option( 'hm_psr_license_key' ) );

		// data to send in our API request
		$api_params = array(
			'edd_action'=> 'activate_license',
			'license' 	=> $license,
			'item_name' => urlencode( HM_PSR_ITEM_NAME ), // the name of our product in EDD
			'url'       => home_url()
		);

		// Call the custom API.
		$response = wp_remote_post( HM_PSR_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );
		
		// make sure the response came back okay
		if ( is_wp_error( $response ) )
			return false;

		// decode the license data
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		// $license_data->license will be either "valid" or "invalid"

		update_option( 'hm_psr_license_status', $license_data->license );

	}
	
    public static function deactivateLicense()
    {

		// run a quick security check
		if( ! check_admin_referer( 'hm_psr_license_deactivate_nonce', 'hm_psr_license_deactivate_nonce' ) )
			return; // get out if we didn't click the dectivate button

		// retrieve the license from the database
		$license = trim( get_option( 'hm_psr_license_key' ) );

		// data to send in our API request
		$api_params = array(
			'edd_action'=> 'deactivate_license',
			'license' 	=> $license,
			'item_name' => urlencode( HM_PSR_ITEM_NAME ), // the name of our product in EDD
			'url'       => home_url()
		);

		// Call the custom API.
		$response = wp_remote_post( HM_PSR_STORE_URL, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

		// make sure the response came back okay
		if ( is_wp_error( $response ) )
			return false;

		delete_option( 'hm_psr_license_status' );
		delete_option( 'hm_psr_license_key' );
	}
	
    public static function pluginCreditBox()
    {
        $accepted_license = get_option('hm_psr_license_key');
        $display_license = str_repeat('*', strlen($accepted_license) - 4) . substr($accepted_license, -4);
        ?>

        <div id="ags-psr_license_key_box">
            <form action="" method="post" id="ags-psr_license_key_form">

                <div id="ags-psr_license_key_form_logo_container">
                    <a href="https://aspengrovestudios.com/aspen-grove-studios-acquires-wordpress-and-woocommerce-plugin-agency-potent-plugins/?utm_source=product-sales-report-pro&amp;utm_medium=link&amp;utm_campaign=wp-plugin-credit-ags-link"
                       target="_blank"><img src="<?php echo(esc_url(plugins_url('images/ags-logo.png', __FILE__))); ?>"
                                            alt="Aspen Grove Studios"/></a>
                </div>
                <div id="ags-psr_license_key_form_body">
                    <h3>
                        <?php echo HM_PSR_ITEM_NAME; ?>
                        <small>v<?php echo HM_PSR_VERSION; ?></small>
                    </h3>
                    <label for="hm_psr_license_activate">
                        <span>License key:</span>
                        <input type="text" readonly="readonly" value="<?php echo(esc_html($display_license)); ?>"/>
                    </label>
                    <?php  echo(wp_nonce_field('hm_psr_license_deactivate_nonce', 'hm_psr_license_deactivate_nonce')); ?>
                    <p class="submit">
                        <button type="submit" name="hm_psr_license_deactivate" value="1">
                            Deactivate License Key
                        </button>
                    </p>
					</div>
					</form>
        </div>

        <?php
	}
}
	
if( !class_exists( 'HM_PSR_EDD_SL_Plugin_Updater' ) )
{
	// load our custom updater
	include( dirname( __FILE__ ) . '/EDD_SL_Plugin_Updater.php' );
}

function hm_psr_register_option()
{
	// creates our settings in the options table
	register_setting('hm_psr_license', 'hm_psr_license_key', 'hm_psr_sanitize_license' );
}
add_action('admin_init', 'hm_psr_register_option');

function hm_psr_plugin_updater()
{
	// retrieve our license key from the DB
	$license_key = trim( get_option( 'hm_psr_license_key' ) );

	// setup the updater
	$edd_updater = new HM_PSR_EDD_SL_Plugin_Updater( HM_PSR_STORE_URL, dirname(__FILE__).'/hm-product-sales-report.php', array(
			'version' 	=> HM_PSR_VERSION,		// current version number
			'license' 	=> $license_key, 		// license key (used get_option above to retrieve from DB)
			'item_name' => HM_PSR_ITEM_NAME, 	// name of this plugin
            'author' => 'Aspen Grove Studios' // author of this plugin
		)
	);

}
add_action( 'admin_init', 'hm_psr_plugin_updater', 0 );

function hm_psr_sanitize_license($new)
{
	$old = get_option( 'hm_psr_license_key' );
	if( $old && $old != $new ) {
		delete_option( 'hm_psr_license_status' ); // new license has been entered, so must reactivate
	}
	return $new;
}

// End code copied from Easy Digital Downloads Software Licensing addon
?>