<?php
/**
 * @package   Barn2\password-protected-categories
 * @author    Barn2 Plugins <support@barn2.com>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
namespace Barn2\Plugin\Password_Protected_Categories\Admin\Wizard\Steps;

use Barn2\Plugin\Password_Protected_Categories\Dependencies\Barn2\Setup_Wizard\Step;
use Barn2\Plugin\Password_Protected_Categories\Dependencies\Barn2\Setup_Wizard\Util;
use Barn2\Plugin\Password_Protected_Categories\PPC_Util;

class Password_Protection extends Step {

	/**
	 * Configure the step.
	 */
	public function __construct() {
		$this->set_id( 'password-protection' );
		$this->set_name( __( 'Password protection', 'password-protected-categories' ) );
		$this->set_description( __( 'Customize the login form for your protected categories', 'password-protected-categories' ) );
		$this->set_title( __( 'Password protection', 'password-protected-categories' ) );
		$this->set_tooltip( __( 'The password login form appears if someone tries to access a password protected category. You can also insert login forms anywhere using the shortcode [category_login]', 'password-protected-categories' ) );
		$this->set_hidden( true );
	}

	/**
	 * {@inheritdoc}
	 */
	public function setup_fields() {
		$fields = [];

		$pages = PPC_Util::get_pages( true );

		$selected_page = PPC_Util::get_option( 'central_login_page' );

		if ( is_array( $selected_page ) && isset( $selected_page[0]['key'] ) ) {
			$selected_page = $selected_page[0]['key'];
		}

		if ( ! empty( $selected_page ) && isset( $pages[ $selected_page ] ) ) {
			$selected_page = [
				[
					'key'   => absint( $selected_page ),
					'label' => $pages[ $selected_page ]
				]
			];
		} else {
			$selected_page = [];
		}

		$fields['password_expires'] = [
			'type'        => 'number',
			'label'       => esc_html__( 'Password expiry', 'password-protected-categories' ),
			'description' => esc_html__( 'How long the category stays unlocked before the password expires.', 'password-protected-categories' ),
			'value'       => absint( PPC_Util::get_option( 'password_expires' ) )
		];

		$fields['central_login_page'] = [
			'type'        => 'select',
			'label'       => __( 'Central login page', 'password-protected-categories' ),
			'alt_label'   => esc_html__( 'Select page', 'password-protected-categories' ),
			'description' => __( "Optional: set this if you want a central password entry page, where users can 'unlock' their category. You can also add a password form anywhere using the shortcode [category_login].", 'password-protected-categories' ),
			'options'     => Util::parse_array_for_dropdown( $pages ),
			'value'       => $selected_page,
		];

		$fields['form_title'] = [
			'type'        => 'text',
			'label'       => esc_html__( 'Title', 'password-protected-categories' ),
			'description' => esc_html__( 'The title of the login page. You can use the placeholder {title} to show the name of the current post or category.', 'password-protected-categories' ),
			'value'       => PPC_Util::get_option( 'form_title' ),
		];

		$fields['form_message'] = [
			'type'        => 'textarea',
			'label'       => esc_html__( 'Login message', 'password-protected-categories' ),
			'description' => esc_html__( 'The message for the login form. You can use HTML here.', 'password-protected-categories' ),
			'value'       => PPC_Util::get_option( 'form_message' )
		];

		return $fields;
	}

	/**
	 * Handle submission.
	 *
	 * @return void
	 */
	public function submit() {
		check_ajax_referer( 'barn2_setup_wizard_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			$this->send_error( esc_html__( 'You are not authorized.', 'password-protected-categories' ) );
		}

		$values  = $this->get_submitted_values();
		$options = PPC_Util::get_options();

		$central_login_page = $values['central_login_page'];

		$form_title   = $values['form_title'];
		$form_message = $values['form_message'];

		$options['central_login_page'] = $central_login_page[0]['key'];
		$options['form_title']         = $form_title;
		$options['form_message']       = $form_message;

		PPC_Util::save_category_login_page( $central_login_page[0]['key'], PPC_Util::get_central_login_page_id() );

		PPC_Util::update_options( $options );

		wp_send_json_success();
	}

}
