<?php

namespace Barn2\Plugin\Password_Protected_Categories;

use Barn2\PPC_Lib\Util;

/**
 * Utility functions for Password Protected Categories.
 *
 * @package   Barn2\password-protected-categories
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class PPC_Util {

	const OPTION_NAME   = 'ppc_options';
	const COOKIE_PREFIX = 'wp-postpass_';

	private static $options = false;

	// PLUGIN OPTIONS

	public static function default_options() {
		return [
			'password_expires'       => 10,
			'show_protected'         => false,
			'form_title'             => __( 'Login Required', 'password-protected-categories' ),
			'form_button'            => __( 'Login', 'password-protected-categories' ),
			'form_label'             => __( 'Password: ', 'password-protected-categories' ),
			'form_label_placeholder' => false,
			'form_message'           => __( 'This content is password protected. To view it please enter your password below:', 'password-protected-categories' )
		];
	}

	/**
	 * Retrieve the plugin options.
	 *
	 * @return array The plugin options array
	 */
	public static function get_options() {
		if ( false === self::$options ) {
			self::$options = \wp_parse_args( \get_option( self::OPTION_NAME, [] ), self::default_options() );
		}
		return self::$options;
	}

	/**
	 * Update the plugin options.
	 *
	 * @param array $options The complete list of updated options.
	 */
	public static function update_options( $options ) {
		\update_option( self::OPTION_NAME, $options );
	}

	/**
	 * Programmatically update an option value.
	 *
	 * @param string $key
	 * @param mixed $value
	 * @param boolean $bypass_cap
	 * @return bool
	 */
	public static function update_option( $key = '', $value = false, $bypass_cap = false ) {
		if ( ! current_user_can( 'manage_options' ) && ! $bypass_cap ) {
			return;
		}

		$options = self::get_options();

		// If no key, exit.
		if ( empty( $key ) ) {
			return false;
		}

		if ( empty( $value ) ) {
			$remove_option = self::delete_option( $key );
			return $remove_option;
		}

		/**
		 * Filter the final value of an option before being saved into the database.
		 *
		 * @param mixed $value the value about to be saved.
		 * @param string $key the key of the option that is being saved.
		 */
		$value = apply_filters( 'ppc_update_option', $value, $key );

		// Next let's try to update the value.
		$options[ $key ] = $value;
		$did_update      = self::update_options( $options );

		return $did_update;
	}

	/**
	 * Remove a PPC option from the database.
	 *
	 * @param string $key
	 * @return bool
	 */
	public static function delete_option( $key, $bypass_cap = false ) {
		if ( ! current_user_can( 'manage_options' ) && ! $bypass_cap ) {
			return;
		}

		$options = self::get_options();

		// Next let's try to update the value.
		if ( isset( $options[ $key ] ) ) {
			unset( $options[ $key ] );
		}

		$did_update = update_option( self::OPTION_NAME, $options );

		return $did_update;

	}

	/**
	 * Retrive a specific plugin option using the $option key specified.
	 *
	 * @param string $option The option key
	 * @return mixed The option if set, or the default option
	 */
	public static function get_option( $option ) {
		$options = self::get_options();
		$value   = isset( $options[ $option ] ) ? $options[ $option ] : '';

		// Back-compat: old checkbox settings were saved as 'yes'/'no'
		if ( 'yes' === $value ) {
			$value = true;
		} elseif ( 'no' === $value ) {
			$value = false;
		}

		return $value;
	}

	/**
	 * Get the option name to use in the "name" attributes for form fields on the plugin settings page.
	 *
	 * @param string $option_key The option key to format
	 * @return The formatted option name
	 */
	public static function get_option_name( $option_key ) {
		return \sprintf( '%s[%s]', self::OPTION_NAME, $option_key );
	}

	// PASSWORD FORM

	public static function get_login_page_title() {
		return \apply_filters( 'ppc_login_page_title', self::replace_tags( self::get_option( 'form_title' ) ) );
	}

	public static function get_login_form_message() {
		return \apply_filters( 'ppc_login_form_message', self::get_option( 'form_message' ) );
	}

	public static function replace_tags( $text ) {
		$title = '';

		if ( self::is_protectable_category() ) {
			$title = \single_term_title( '', false );
		} elseif ( self::is_protectable_single_post() ) {
			$title = \get_the_title();
		}

		return \strtr(
			$text,
			[
				'{title}' => $title
			]
		);
	}

	public static function set_password_cookie( $term, $password ) {
		require_once \ABSPATH . \WPINC . '/class-phpass.php';

		$hasher             = new \PasswordHash( 8, true );
		$expires_after_days = self::get_option( 'password_expires' );

		if ( ! $expires_after_days ) {
			$expires_after_days = 10;
		}

		$cookie_expiry = \apply_filters( 'ppc_category_password_expires', \time() + ( $expires_after_days * \DAY_IN_SECONDS ) );
		$referer       = \wp_get_referer();

		if ( $referer ) {
			$secure = ( 'https' === \parse_url( $referer, \PHP_URL_SCHEME ) );
		} else {
			$secure = false;
		}

		$cookie_value = "{$term->term_id}|{$term->taxonomy}|" . $hasher->HashPassword( \wp_unslash( $password ) );
		\setcookie( self::COOKIE_PREFIX . \COOKIEHASH, $cookie_value, $cookie_expiry, \COOKIEPATH, \COOKIE_DOMAIN, $secure );
	}

	// TERMS

	/**
	 * Retrieve a list of all protectable taxonomies from the list of currently registered taxonomies.
	 *
	 * @return array A list of taxonomy names which can be protected by the plugin
	 */
	public static function get_protectable_taxonomies() {
		$taxonomies = \get_taxonomies( [ 'hierarchical' => true ] );

		if ( Util::is_protected_categories_active() ) {
			$taxonomies = \array_diff( $taxonomies, [ 'product_cat' ] );
		}
		return $taxonomies;
	}

	/**
	 * Returns the currently unlocked category or term (if any) as an array in the following format:
	 *
	 * array(
	 *     'term_id' => 12
	 *     'taxonomy' => 'category'
	 * )
	 *
	 * @return boolean|array The taxonomy array or false if no terms are currently unlocked.
	 */
	public static function get_unlocked_term() {
		// @todo Use unique cookie for each category
		if ( ! isset( $_COOKIE[ self::COOKIE_PREFIX . \COOKIEHASH ] ) ) {
			return false;
		}

		$tax_term = \explode( '|', $_COOKIE[ self::COOKIE_PREFIX . \COOKIEHASH ], 3 );

		if ( 3 !== \count( $tax_term ) ) {
			return false;
		}

		return [
			'term_id'  => (int) $tax_term[0],
			'taxonomy' => $tax_term[1],
			'password' => $tax_term[2]
		];
	}

	/**
	 * Wrapper function to get_terms() to handle parameter change in WP 4.5.
	 *
	 * @global string $wp_version
	 * @param string $args The $args to pass to get_terms()
	 * @return array An array of WP_Term objects or an empty array if none found
	 */
	public static function get_terms( $args = [] ) {
		global $wp_version;

		if ( empty( $args['taxonomy'] ) ) {
			$args['taxonomy'] = '';
		}
		// Arguments for get_terms() changed in WP 4.5
		if ( \version_compare( $wp_version, '4.5', '>=' ) ) {
			$terms = \get_terms( $args );
		} else {
			$tax = $args['taxonomy'];
			unset( $args['taxonomy'] );
			$terms = \get_terms( $tax, $args );
		}

		if ( \is_array( $terms ) ) {
			return $terms;
		} else {
			return [];
		}
	}

	/**
	 * Retrieve an array of all hidden terms for the current user. If the user can
	 * view private posts (e.g. user is an administrator) then these terms will not be
	 * included in the result.
	 *
	 * If the user has previously unlocked a term, then this will be excluded from the result.
	 *
	 * @param array $taxonomies A list of taxonomies for which to retrieve the hidden terms
	 * @param $fields The fields to retrieve. @see get_terms()
	 * @return array An array of WP_Term objects or an empty array if none found
	 */
	public static function get_hidden_terms( $taxonomies, $fields = 'all' ) {
		$visibility = [];

		if ( ! self::get_option( 'show_protected' ) ) {
			$visibility[] = 'password';
			$visibility[] = 'protected';
		}
		if ( ! \current_user_can( 'read_private_posts' ) ) {
			$visibility[] = 'private';
		}

		if ( ! $visibility || ! $taxonomies ) {
			return [];
		}

		$unlocked = self::get_unlocked_term();

		return self::get_terms(
			[
				'ppc_check'  => true,
				'taxonomy'   => $taxonomies,
				'fields'     => $fields,
				'hide_empty' => true,
				'exclude'    => $unlocked ? $unlocked['term_id'] : [],
				'meta_query' => [
					[
						'key'     => 'visibility',
						'value'   => $visibility,
						'compare' => 'IN'
					]
				]
			]
		);
	}

	/**
	 * Get an array of password protected terms for the specified taxonomies. If no taxonomies passed,
	 * retrieve all password protected terms.
	 *
	 * @param array $taxonomies An array of taxonomy names (optional)
	 * @return array An array of password protected terms (WP_Term objects)
	 */
	public static function get_password_protected_terms( $taxonomies = [] ) {
		$taxonomies = $taxonomies ? \array_intersect( $taxonomies, self::get_protectable_taxonomies() ) : self::get_protectable_taxonomies();

		if ( ! $taxonomies ) {
			return [];
		}

		$backwards_terms = self::get_terms(
			[
				'ppc_check'  => true,
				'taxonomy'   => $taxonomies,
				'fields'     => 'all',
				'hide_empty' => false,
				'meta_query' => [
					[
						'key'     => 'visibility',
						'value'   => 'password',
						'compare' => '='
					]
				]
			]
		);

		$new_terms = self::get_terms(
			[
				'ppc_check'  => true,
				'taxonomy'   => $taxonomies,
				'fields'     => 'all',
				'hide_empty' => false,
				'meta_query' => [
					[
						'key'     => 'visibility',
						'value'   => 'protected',
						'compare' => '='
					],
					[
						'key'     => 'password',
						'compare' => 'EXIST'
					]
				]
			]
		);

		return array_merge( $backwards_terms, $new_terms );
	}

	/**
	 * Get the list of passwords for the specified term.
	 *
	 * @param int $term_id The term ID for the term.
	 * @param boolean $single Whether to return a single value.
	 * @return array The passwords array. Will be an empty array if there are no passwords.
	 */
	public static function get_term_passwords( $term_id, $single = false ) {
		$passwords = \get_term_meta( $term_id, 'password', $single );

		if ( empty( $passwords ) || ! is_array( $passwords ) ) {
			return [];
		}

		// if ( ! empty( $passwords ) ) {
		// $first = reset( $passwords );
		// Prevent possible conflit with WooCommerce Protected Categories which stores passwords under the same meta key.
		// if ( is_array( $first ) ) {
		// return [];
		// }
		// }

		return $passwords;
	}

	// TERM VISIBILITY

	/**
	 * Get the visibility for the specified term.
	 *
	 * @param int $term_id The term ID for the term.
	 * @return string The visibility - one of 'public', 'password' or 'private', 'protected'.
	 */
	public static function get_visibility( $term_id ) {
		$visibility = \get_term_meta( (int) $term_id, 'visibility', true );

		if ( ! $visibility || ! in_array( $visibility, [ 'public', 'password', 'private', 'protected' ] ) ) {
			$visibility = 'public';
		}

		return $visibility;
	}

	/**
	 * Retrieve the Term_Visibility instance for the specified term.
	 *
	 * @param WP_Term $term The term to retrieve the visibility for
	 * @return boolean|Term_Visibility
	 */
	public static function get_term_visibility( $term ) {
		if ( ! ( $term instanceof \WP_Term ) ) {
			return false;
		}

		$cache = self::get_term_visibility_cache();

		if ( ! \array_key_exists( $term->term_id, $cache ) ) {
			$cache[ $term->term_id ] = new Term_Visibility( $term );
			self::update_term_visibility_cache( $cache );
		}
		return $cache[ $term->term_id ];
	}

	/**
	 * Retrive a list of Term_Visibility objects for the specified post.
	 * If no post is specified, the current post object is used.
	 *
	 * @param int|WP_Post $post The post ID or post object
	 * @param string $taxonomy The taxonomy to retrive visibilites for, or false to retrieve all applicable taxonomies
	 * @return array An array of Term_Visibility objects
	 */
	public static function get_the_term_visibility( $post = null, $taxonomy = false ) {

		$post = $post ? \get_post( $post ) : \get_queried_object();

		// Bail if no post
		if ( ! ( $post instanceof \WP_Post ) ) {
			return [];
		}

		// Defer to WooCommerce PPC if it's also active.
		if ( 'product' === $post->post_type && Util::is_protected_categories_active() ) {
			return [];
		}

		$terms = [];

		if ( ! $taxonomy || ! \taxonomy_exists( $taxonomy ) ) {
			$taxonomies = \array_intersect( \get_object_taxonomies( $post ), self::get_protectable_taxonomies() );

			if ( $taxonomies ) {
				foreach ( $taxonomies as $taxonomy ) {
					$post_terms = \get_the_terms( $post, $taxonomy );

					if ( $post_terms && \is_array( $post_terms ) ) {
						$terms = \array_merge( $terms, $post_terms );
					}
				}
			}
		} else {
			$terms = \get_the_terms( $post, $taxonomy );
		}

		return self::to_term_visibilities( $terms );
	}

	public static function to_term_visibilities( $terms ) {
		if ( ! $terms || ! \is_array( $terms ) ) {
			return [];
		}

		$result = [];
		$cache  = self::get_term_visibility_cache();

		foreach ( $terms as $term ) {
			if ( \array_key_exists( $term->term_id, $cache ) ) {
				$result[] = $cache[ $term->term_id ];
			} else {
				$cache[ $term->term_id ] = new Term_Visibility( $term );
				$result[]                = $cache[ $term->term_id ];
			}
		}

		self::update_term_visibility_cache( $cache );

		return $result;
	}

	public static function get_term_visibility_cache() {
		$cache = \wp_cache_get( 'ppc_visibilities' );

		if ( false !== $cache || ! \is_array( $cache ) ) {
			$cache = [];
		}
		return $cache;
	}

	public static function update_term_visibility_cache( $term_visibilities ) {
		\wp_cache_set( 'ppc_visibilities', $term_visibilities );
	}

	// CONDITIONALS

	/**
	 * Is this a hidden post (i.e. private or password protected)?
	 * Defaults to the current post if not specified.
	 *
	 * @param int|WP_Post $post Post ID or WP_Post object
	 * @param boolean $show_if_protected Whether to show the post if its password protected
	 * @return boolean
	 */
	public static function is_hidden_post( $post = null ) {

		if ( $visibilities = self::get_the_term_visibility( $post ) ) {
			foreach ( $visibilities as $visibility ) {
				if ( $visibility->is_hidden() ) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Are we viewing a single post (for any post type) which can be protected by the plugin?
	 *
	 * Returns the same as is_singular() but with an extra check for WooCommerce products. If WooCommerce Protected Categories is installed,
	 * products are excluded from this check.
	 *
	 * @return boolean
	 */
	public static function is_protectable_single_post() {
		$protectable = is_singular();

		if ( function_exists( 'is_product' ) && Util::is_protected_categories_active() ) {
			$protectable = $protectable && ! \is_product();
		}

		return $protectable;
	}

	/**
	 * Are we viewing a category/taxonomy archive page (for any post type) which can be protected by the plugin?
	 *
	 * Returns the same as is_category() || is_tax() but with an extra check for WooCommerce product categories. If WooCommerce Protected Categories is installed,
	 * product categories are excluded from this check.
	 *
	 * @return boolean
	 */
	public static function is_protectable_category() {
		return \is_category() || \is_tax( self::get_protectable_taxonomies() );
	}

	// FORMATTING

	/**
	 * Removes any new line characters and long whitespace sequences (2 or more) from HTML output so that
	 * wpautop doesn't mess up the formatting.
	 *
	 * @param string $text The text to sanitize.
	 * @return string The sanitized text, which can be passed safely to wpautop.
	 */
	public static function sanitize_whitespace_for_autop( $text ) {
		return \preg_replace( '/\R|\s{2,}/', '', $text );
	}

	/**
	 * Determines whether the supplied category or categories are protected.
	 *
	 * Checks the supplied categories including all ancestors of those categories (if any). Returns
	 * one of the following values:
	 *
	 *  - 'password' - One or more categories is password protected
	 *  - 'user'     - One or more categories is protected to specific users
	 *  - 'role'     - One or more categories is protected to specific user roles
	 *  - 'private'  - One or more categories is private
	 *  - false      - All categories are public or at least one category has been 'unlocked' (see below).
	 *
	 * The function will return false (i.e. not protected) if all categories including ancestors are public.
	 *
	 * It also returns false if at least one protected category has been unlocked - e.g. the correct password
	 * has been entered, or the user has the required role (depending on the protection type). In this instance,
	 * the category is only considered 'unlocked' only if the there are no child categories of that category
	 * which are protected by another means.
	 *
	 * The function will always return false (i.e. unlocked) if at least one category is unlocked, regardless
	 * of the other categories supplied, even if the other categories are protected.
	 *
	 * If two or more protected categories are found, or if one protected category has multiple types of protection,
	 * the function will return the first type of protection found, in the following order of  precedence:
	 * password, private, user, role. This can be controlled using the 'ppc_category_protection_priority_order' filter.
	 *
	 * @param array|Term_Visibility $categories The category or array of Term_Visibility objects to check.
	 * @return boolean|string false if not protected, otherwise 'password', 'private', 'user' or 'role' to denote the protection type.
	 */
	public static function is_protected( $categories ) {
		if ( ! $categories ) {
			return false;
		}

		if ( $categories instanceof Term_Visibility ) {
			$categories = [ $categories ];
		}

		$protection = [];

		foreach ( $categories as $category ) {
			$full_hierarchy = $category->ancestors();
			\array_unshift( $full_hierarchy, $category );

			$level = 0;

			foreach ( $full_hierarchy as $pcategory ) {
				if ( $pcategory->is_unlocked() ) {
					return false;
				}

				self::build_protection_for_level( $protection, $pcategory, $level );
			}
		}

		if ( empty( $protection ) ) {
			return false;
		}

		$lowest_protection = \reset( $protection );

		foreach ( \apply_filters( 'ppc_category_protection_priority_order', [ 'protected', 'password', 'private', 'user', 'role' ] ) as $protection_type ) {
			if ( \in_array( $protection_type, $lowest_protection ) ) {
				return $protection_type;
			}
		}

		// Shouldn't ever get here, but return false (not protected) just in case.
		return false;
	}

	private static function build_protection_for_level( &$protection, $category, $level = 0 ) {
		if ( $category->has_password_protection() ) {
			$protection[ $level ][] = 'password';
		}
		if ( $category->has_role_protection() ) {
			$protection[ $level ][] = 'role';
		}
		if ( $category->has_user_protection() ) {
			$protection[ $level ][] = 'user';
		}
		if ( $category->has_private_protection() ) {
			$protection[ $level ][] = 'private';
		}
	}

	/**
	 * Sanitize the html content of a tooltip.
	 *
	 * @param string $var toolip content
	 * @return string
	 */
	private static function sanitize_tooltip( $var ) {
		return htmlspecialchars(
			wp_kses(
				html_entity_decode( $var ),
				[
					'br'     => [],
					'em'     => [],
					'strong' => [],
					'small'  => [],
					'span'   => [],
					'ul'     => [],
					'li'     => [],
					'ol'     => [],
					'p'      => [],
				]
			)
		);
	}

	/**
	 * Print an help tooltip.
	 *
	 * @param string $tip help tooltip.
	 * @param boolean $allow_html
	 * @return string
	 */
	public static function help_tip( $tip, $allow_html = false ) {

		if ( $allow_html ) {
			$tip = self::sanitize_tooltip( $tip );
		} else {
			$tip = esc_attr( $tip );
		}

		return '<span class="b2-help-tip" data-tip="' . $tip . '"></span>';

	}

	/**
	 * Get the ID number of the central login page.
	 *
	 * @return string
	 */
	public static function get_central_login_page_id() {
		return self::get_option( 'central_login_page' );
	}

	/**
	 * Programmatically insert the category login shortcode into the selected
	 * central login page.
	 *
	 * When an old value is found, remove the shortcode from the previous page.
	 *
	 * @param string $value new page id
	 * @param string $old_value old page id
	 * @return void
	 */
	public static function save_category_login_page( $value, $old_value ) {

		// Add the login shortcode to selected page.
		$login_shortcode = \sprintf( '[%s]', Login_Shortcode::SHORTCODE );

		if ( $value ) {
			$page = \get_post( $value );

			if ( $page instanceof \WP_Post && false === strpos( $page->post_content, '[' . Login_Shortcode::SHORTCODE ) ) {
				$content = $page->post_content ? ( $page->post_content . "\n\n" . $login_shortcode ) : $login_shortcode;

				\wp_update_post(
					[
						'ID'           => $page->ID,
						'post_content' => $content
					]
				);
			}
		}

		if ( $old_value !== $value ) {
			// Remove from old page, if we have one.
			$page = \get_post( $old_value );

			if ( $page instanceof \WP_Post && false !== strpos( $page->post_content, '[' . Login_Shortcode::SHORTCODE ) ) {
				$content = \trim( \preg_replace( sprintf( '/\[%s.*?\]/', Login_Shortcode::SHORTCODE ), '', $page->post_content ) );

				\wp_update_post(
					[
						'ID'           => $page->ID,
						'post_content' => $content
					]
				);
			}
		}

	}

	/**
	 * Get an array of pages of the site.
	 *
	 * @param bool $exclude_empty whether or not an empty option should be displayed within the dropdown.
	 * @return array
	 */
	public static function get_pages( $exclude_empty = false ) {
		$pages   = get_pages();
		$options = [];

		if ( ! $exclude_empty ) {
			$options[] = '';
		}

		if ( ! empty( $pages ) && is_array( $pages ) ) {
			foreach ( $pages as $page ) {
				$options[ absint( $page->ID ) ] = esc_html( $page->post_title );
			}
		}

		return $options;

	}

	/**
	 * Helper method to display a styled deprecation notice
	 * for categories marked as private.
	 *
	 * We need the inline styling because WP moves notices below the 1st
	 * h1 tag automatically.
	 *
	 * @return void
	 */
	public static function display_deprecated_notice() {
		?>
		<style>
			.b2-deprecated-notice {
				background: #ffffff;
				border: 1px solid #c3c4c7;
				border-left-color: #007cba;
				border-left-width: 1px;
				border-left-width: 4px;
				box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
				margin: 5px 0 15px;
				padding: 12px;
			}

			.b2-deprecated-notice p {
				margin: 0 !important;
			}
		</style>
		<div class="b2-deprecated-notice">
			<p><?php esc_html_e( 'This category is marked as private. This option has now been deprecated and is still working on your site, but is no longer available to select. We recommend using the "User Roles" protection option instead.', 'password-protected-categories' ); ?></p>
		</div>
		<?php
	}

}
