<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Import compatibility.
 *
 * Should work with most if the data is imported as meta.
 *
 * @class    Iconic_WLV_Compat_Import
 * @version  1.0.0
 * @category Class
 * @author   Iconic
 */
class Iconic_WLV_Compat_Import {
	/**
	 * Run.
	 */
	public static function run() {
		add_action( 'save_post', array( __CLASS__, 'check_meta' ), 10, 3 );
	}

	/**
	 * Check post meta and replace with custom DB.
	 *
	 * @param $post_id
	 * @param $post
	 * @param $update
	 */
	public static function check_meta( $post_id, $post, $update ) {
		$page = filter_input( INPUT_GET, 'page', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

		if ( empty( $page ) ) {
			return;
		}

		if ( ! str_contains( $page, 'pmxi' ) ) {
			return;
		}

		$post_type = get_post_type( $post_id );
		if ( 'cpt_iconic_wlv' !== $post_type ) {
			return;
		}

		$meta_keys = array(
			'iconic_wlv_product_skus',
			'iconic_wlv_product_ids',
			'iconic_wlv_attributes',
			'iconic_wlv_show_image',
			'iconic_wlv_style',
			'iconic_wlv_link_by_taxonomy',
			'iconic_wlv_link_by_terms',
		);

		$final_meta  = array();
		$final_terms = array();

		foreach ( $meta_keys as $key ) {
			$value = get_post_meta( $post_id, $key, true );

			if ( ! $value ) {
				continue;
			}

			if ( 'iconic_wlv_link_by_terms' === $key ) {
				$terms = explode( ',', $value );

				foreach ( $terms as $term ) {
					$term = get_term_by( 'slug', $term, 'product_cat' );
					if ( ! is_wp_error( $term ) && ! empty( $term ) ) {
						$final_terms[] = $term->term_id;
					}
				}

				continue;
			}

			$meta                       = self::format_meta( $key, $value );
			$final_meta[ $meta['key'] ] = $meta['value'];

			delete_post_meta( $post_id, $key );
		}

		Iconic_WLV_Database::update_linked_variations_meta( $post_id, $final_meta, $final_terms );
	}

	/**
	 * Format meta value for saving.
	 *
	 * @param string $meta_key
	 * @param string $meta_value
	 *
	 * @return mixed
	 */
	public static function format_meta( $meta_key, $meta_value ) {
		$meta_key = str_replace( 'iconic_wlv_', '', $meta_key );

		switch ( $meta_key ) {
			case 'product_skus':
			case 'product_ids':
			case 'attributes':
				$meta_value = explode( ',', $meta_value );
				break;
		}

		if ( is_array( $meta_value ) ) {
			$meta_value = Iconic_WLV_Helpers::get_clean_and_unique_array( $meta_value );
		}

		if ( $meta_key === 'product_skus' ) {
			$meta_key   = 'product_ids';
			$meta_value = self::get_ids_from_skus( $meta_value );
		}

		if ( 'link_by_taxonomy' === $meta_key ) {
			$meta_key   = 'link_by';
			$meta_value = $meta_value;
		}

		if ( $meta_key === 'product_ids' ) {
			$meta_value = array_map( 'strval', $meta_value );

			foreach ( $meta_value as $index => $id ) {
				if ( false === get_post_status( $id ) ) {
					unset( $meta_value[ $index ] );
				}
			}
		}

		return array(
			'key'   => $meta_key,
			'value' => $meta_value,
		);
	}

	/**
	 * Get IDs from SKUs.
	 *
	 * @param array $meta_value
	 *
	 * @return array
	 */
	public static function get_ids_from_skus( $meta_value ) {
		if ( empty( $meta_value ) ) {
			return $meta_value;
		}

		foreach ( $meta_value as $index => $sku ) {
			$id = wc_get_product_id_by_sku( $sku );

			if ( ! $id ) {
				unset( $meta_value[ $index ] );
				continue;
			}

			$meta_value[ $index ] = $id;
		}

		return $meta_value;
	}
}