<?php
if ( ! defined( 'WPINC' ) ) {
	wp_die();
}

/**
 * Class Iconic_WLV_Database
 */
class Iconic_WLV_Database {
	/**
	 * Database version.
	 *
	 * @var string
	 */
	protected static $version = '1.0.4';

	/**
	 * Linked variations DB name.
	 *
	 * @var string
	 */
	public static $linked_variations_db_name = 'iconic_woo_linked_variations';

	/**
	 * Install/update the DB table.
	 */
	public static function install() {
		if ( version_compare( get_site_option( 'iconic_wlv_db_version' ), self::$version, '>=' ) ) {
			return;
		}

		global $wpdb;

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		$table_name = $wpdb->prefix . self::$linked_variations_db_name;

		$sql = "CREATE TABLE $table_name (
		`id` int(11) unsigned NOT NULL AUTO_INCREMENT,
		`link_by` varchar(100),
		`product_ids` longtext,
		`attributes` longtext,
		`show_image` tinyint(1) DEFAULT NULL,
		`post_id` bigint(20) DEFAULT NULL,
		`style` text,
		PRIMARY KEY (`id`)
		) $collate;
		
		CREATE TABLE `{$table_name}_terms` (
		`lv_id` int(11) NOT NULL,
		`term_id` int(11) NOT NULL
		) $collate;
		";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );

		update_option( 'iconic_wlv_db_version', self::$version );
	}

	/**
	 * Get meta for post.
	 *
	 * @return null|object|bool
	 */
	public static function get_linked_variations_meta( $post_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$linked_variations_db_name;

		$prepare = $wpdb->prepare(
			"SELECT * FROM $table_name WHERE post_id = %d",
			$post_id
		);

		return $wpdb->get_row( $prepare );
	}

	/**
	 * Update linked variations meta.
	 *
	 * @param int   $post_id
	 * @param array $meta
	 */
	public static function update_linked_variations_meta( $post_id, $meta, $terms = array() ) {
		global $wpdb;

		foreach ( $meta as $key => $value ) {
			$meta[ $key ] = maybe_serialize( $value );
		}

		$table_name = $wpdb->prefix . self::$linked_variations_db_name;

		$exists = self::get_linked_variations_meta( $post_id );

		if ( ! is_null( $exists ) ) {
			$wpdb->update(
				$table_name,
				$meta,
				array( 'post_id' => $post_id ),
				null,
				array( '%d' )
			);
		} else {
			$meta['post_id'] = $post_id;

			$wpdb->insert(
				$table_name,
				$meta
			);
		}

		$lv_id = ! empty( $exists ) ? $exists->id : $wpdb->insert_id;
		self::update_term_ids( $lv_id, $terms );
	}

	/**
	 * Update term IDs in wp_iconic_woo_linked_variations_terms table.
	 *
	 * @param int   $lv_id Linked Variation ID.
	 * @param array $terms Terms array.
	 *
	 * @return void
	 */
	public static function update_term_ids( $lv_id, $terms ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'iconic_woo_linked_variations_terms';

		$wpdb->delete(
			$table_name,
			array(
				'lv_id' => $lv_id,
			),
			array( '%d' )
		);

		if ( empty( $terms ) || ! is_array( $terms ) ) {
			return;
		}

		foreach ( $terms as $term ) {
			$wpdb->insert(
				$table_name,
				array(
					'lv_id'   => $lv_id,
					'term_id' => $term,
				),
				array( '%d', '%d' )
			);
		}
	}

	/**
	 * Delete linked variations meta.
	 *
	 * @param int $post_id Post ID.
	 */
	public static function delete_linked_variations_meta( $post_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$linked_variations_db_name;

		$wpdb->delete( $table_name, array( 'post_id' => $post_id ), array( '%d' ) );
	}

	/**
	 * Delete associated rows from `wp_iconic_woo_linked_variations_terms` table.
	 *
	 * @param int $post_id Linked Variation Post ID.
	 *
	 * @return void
	 */
	public static function delete_linked_variation_terms( $post_id ) {
		$lv_group = new Iconic_WLV_Linked_Variations_Group( $post_id );

		if ( empty( $lv_group ) || empty( $lv_group->meta ) ) {
			return;
		}

		global $wpdb;

		$wpdb->delete(
			$wpdb->prefix . 'iconic_woo_linked_variations_terms',
			array(
				'lv_id' => $lv_group->meta->id,
			),
			array( '%d' )
		);
	}

	/**
	 * Get group ID containing product.
	 *
	 * @param int $product_id
	 *
	 * @return bool|int
	 */
	public static function get_linked_variations_group_id_for_product( $product_id ) {
		$product_id = apply_filters( 'iconic_wlv_product_id', $product_id );

		static $group_ids = array();

		if ( array_key_exists( $product_id, $group_ids ) ) {
			return $group_ids[ $product_id ];
		}

		global $wpdb;

		$group_ids[ $product_id ] = false;

		$table_name = $wpdb->prefix . self::$linked_variations_db_name;

		// Check for specific products setting.
		$result = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT wlv.* FROM {$wpdb->prefix}iconic_woo_linked_variations AS wlv
				INNER JOIN {$wpdb->posts} AS p ON p.ID = wlv.post_id
				WHERE product_ids LIKE '%%\"%d\"%%'
				AND p.post_status = 'publish'
				AND ( wlv.link_by IS NULL OR wlv.link_by = 'specific_products' )",
				$product_id
			)
		);

		// Check for taxonomies. We only do this if there are results returned by the first query.
		if ( empty( $result ) ) {
			$result = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT lv.* FROM 
					{$wpdb->prefix}iconic_woo_linked_variations lv, {$wpdb->prefix}iconic_woo_linked_variations_terms lvt, {$wpdb->prefix}term_taxonomy tt, {$wpdb->prefix}term_relationships tr, {$wpdb->prefix}posts p
					WHERE 
					lv.link_by IS NOT NULL
					AND lv.link_by != 'specific_products'
					AND lv.link_by = tt.taxonomy
					AND tt.term_taxonomy_id = tr.term_taxonomy_id
					AND lv.id = lvt.lv_id
					AND lvt.term_id = tt.term_id
					AND tr.object_id = p.ID
					AND p.ID = %d",
					$product_id
				),
			);
		}

		if ( empty( $result ) ) {
			return $group_ids[ $product_id ];
		}

		$group_id     = absint( $result->post_id );
		$group_status = get_post_status( $group_id );

		if ( ! in_array( $group_status, array( 'publish', 'private', 'draft' ) ) ) {
			return $group_ids[ $product_id ];
		}

		$group_ids[ $product_id ] = $group_id;

		return $group_ids[ $product_id ];
	}

	/**
	 * Get terms to select products by.
	 *
	 * @param int $lv_id ID of the linked variations table.
	 *
	 * @return array
	 */
	public static function get_select_by_terms( $lv_id ) {
		global $wpdb;

		return $wpdb->get_col(
			$wpdb->prepare(
				"select term_id from {$wpdb->prefix}iconic_woo_linked_variations_terms where lv_id = %d",
				$lv_id
			)
		);
	}
}
