<?php

namespace PixelYourSite\Pinterest;

use PixelYourSite;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

function maybeMigrate() {
	
	if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
		return;
	}
	
	if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
		return;
	}
	
	$pinterest_version = get_option( 'pys_pinterest_version', false );
	
	// migrate from 1.x
	if ( $pinterest_version && version_compare( $pinterest_version, "3.0.0", '<' ) ) {

        migrateV1Options();
		migrate_v1_pinterest_events();

		update_option( 'pys_pinterest_version', "3.0.0" );
	
	}
	
}

function migrate_v1_pinterest_events() {
    $customEvents = array();

    $events = get_all_v1_pinterest_events();
    foreach ( $events as $post ) {
        setup_postdata( $post );
        foreach ( build_v7_events_from_v1( $post ) as $event ) {
            $customEvents[] = $event;
        }
    }

    wp_reset_postdata();

    foreach ( $customEvents as $eventParams ) {
        PixelYourSite\CustomEventFactory::create( $eventParams );
    }
}

function get_all_v1_pinterest_events() {
    $query = new \WP_Query( array(
        'post_type'      => 'pys_pinterest_event',
        'posts_per_page' => -1,
    ) );

    return $query->have_posts() ? $query->posts : array();
}

function build_v7_events_from_v1( $post ) {
    $commonParams = get_common_event_params( $post );
    $v6_type = get_post_meta( $post->ID, '_type', true );

    if ( $v6_type === 'on_page' ) {
        return array( build_on_page_event( $post, $commonParams ) );
    }

    return build_dynamic_events( $post, $commonParams );
}

function get_common_event_params( $post ) {
    $state = get_post_meta( $post->ID, '_state', true );
    $type = get_post_meta( $post->ID, '_pinterest_event_type', true );
    $custom_type = $type === 'CustomEvent' ? get_post_meta( $post->ID, '_pinterest_event_custom_name', true ) : null;

    $params = array();
    $props = get_post_meta( $post->ID, '_pinterest_event_properties', true );
    if ( is_array( $props ) ) {
        foreach ( $props as $name => $value ) {
            $params[] = array( 'name' => $name, 'value' => $value );
        }
    }

    return array(
        'title'                       => $post->post_title,
        'enabled'                     => $state === 'active',
        'delay'                       => null,
        'triggers'                    => array(),
        'url_filters'                 => array(),
        'pinterest_enabled'           => true,
        'pinterest_event_type'        => $type,
        'pinterest_custom_event_type' => $custom_type,
        'pinterest_params_enabled'    => !empty( $params ),
        'pinterest_custom_params'     => $params,
    );
}

function build_on_page_event( $post, $baseParams ) {
    $triggers_raw = get_post_meta( $post->ID, '_on_page_triggers', true );
    $triggers = array();

    if ( is_array( $triggers_raw ) ) {
        foreach ( $triggers_raw as $trigger ) {
            if ( ! empty( $trigger ) ) {
                $triggers[] = array( 'rule' => 'contains', 'value' => $trigger );
            }
        }
    }

    $event = $baseParams;
    $event['delay'] = (int) get_post_meta( $post->ID, '_delay', true );
    $event['trigger_type'] = 'page_visit';
    $event['page_visit_triggers'] = $triggers;

    return $event;
}

function build_dynamic_events( $post, $baseParams ) {
    $rawTriggers = get_post_meta( $post->ID, '_dynamic_triggers', true );
    $triggerGroups = array();

    if ( is_array( $rawTriggers ) ) {
        foreach ( $rawTriggers as $trigger ) {
            if ( ! empty( $trigger['value'] ) ) {
                $type = $trigger['type'];
                $triggerGroups[$type][] = array(
                    'rule'  => in_array( $type, array( 'url_click' ) ) ? 'contains' : null,
                    'value' => $trigger['value'],
                );
            }
        }
    }

    $urlFilters = get_post_meta( $post->ID, '_dynamic_url_filters', true );
    $urlFilters = is_array( $urlFilters ) ? array_filter( $urlFilters ) : array();

    $events = array();
    foreach ( $triggerGroups as $type => $triggers ) {
        $event = $baseParams;
        $event['trigger_type'] = $type;
        $event["{$type}_triggers"] = $triggers;
        $event['url_filters'] = $urlFilters;
        $events[] = $event;
    }

    return $events;
}

function migrateV1Options() {

    $v1 = get_option('pys_pinterest', array());
    $v2 = buildV2Options($v1);


    // Cleanup
    $v2 = cleanNullValues($v2);
	
	// update settings
	PixelYourSite\Pinterest()->updateOptions( $v2 );
	PixelYourSite\Pinterest()->reloadOptions();
	
}

function buildV2Options($v1) {
    return array(
        'license_key'     => getValue($v1, 'license_key'),
        'license_status'  => getValue($v1, 'license_status'),
        'license_expires' => getValue($v1, 'license_expires'),
        'adsense_enabled' => getValue($v1, 'adsense_enabled'),
        'click_event_enabled' => getValue($v1, 'click_event_enabled'),
        'watchvideo_event_enabled' => getValue($v1, 'youtube_enabled'),
        'search_event_enabled' => getValue($v1, 'search_event_enabled'),
        'woo_purchase_enabled' => getValue($v1, 'woo_purchase_enabled'),
        'woo_initiate_checkout_enabled' => getValue($v1, 'woo_initiate_checkout_enabled'),
        'woo_view_content_enabled' => getValue($v1, 'woo_view_content_enabled'),
        'woo_view_category_enabled' => getValue($v1, 'woo_view_category_enabled'),
        'woo_affiliate_enabled' => getValue($v1, 'woo_affiliate_enabled'),
        'woo_paypal_enabled' => getValue($v1, 'woo_paypal_enabled'),
        'edd_purchase_enabled' => getValue($v1, 'edd_purchase_enabled'),
        'edd_initiate_checkout_enabled' => getValue($v1, 'edd_initiate_checkout_enabled'),
        'edd_add_to_cart_enabled' => getValue($v1, 'edd_add_to_cart_enabled'),
        'edd_view_content_enabled' => getValue($v1, 'edd_view_content_enabled'),
        'edd_view_category_enabled' => getValue($v1, 'edd_view_category_enabled'),
        'pixel_id' => buildPixelId($v1),
    );
}

function getValue($array, $key) {
    return isset($array[$key]) ? $array[$key] : null;
}

function buildPixelId($v1) {
    if (isset($v1['pixel_id'])) {
        return is_array($v1['pixel_id']) ? $v1['pixel_id'] : array($v1['pixel_id']);
    }
    return null;
}

function cleanNullValues($array) {
    return array_filter($array, function ($value) {
        return $value !== null;
    });
}
