<?php

namespace PixelYourSite\Pinterest;
use function PixelYourSite\getBrandForWooItem;
use function PixelYourSite\getWPMLProductId;
use function PixelYourSite\isWPMLActive;
use function PixelYourSite\wooGetOrderIdFromRequest;
use function PixelYourSite\wooIsRequestContainOrderId;
use PixelYourSite;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

function getWooCustomAudiencesOptimizationParams( $post_id ) {
	
	$post = get_post( $post_id );
	
	$params = array(
		'content_name'  => '',
		'category_name' => '',
	);
	
	if ( ! $post ) {
		return $params;
	}
	
	if ( $post->post_type == 'product_variation' ) {
		$post_id = $post->post_parent; // get terms from parent
	}
	
	$params['content_name']  = $post->post_title;
	$params['category_name'] = implode( ', ', PixelYourSite\getObjectTerms( 'product_cat', $post_id ) );
	
	return $params;
	
}

function getWooSingleAddToCartParams( $product_id, $qty = 1, $is_external = false ) {

    $params = [
        'post_type' => 'product',
        'line_items'  => [], // Ensure this is always an array
    ];
    $product = wc_get_product( $product_id );
    if( !$product ) {
        return $params; // Return empty params if product is not found
    }
	$line_item = array(
		'product_id'       => getWooProductContentId($product_id),
        'product_name'  => $product->get_name(),
		'product_quantity' => $qty,
        'product_price' => PixelYourSite\getWooProductPriceToDisplay( $product_id, $qty ),
        'product_category' => implode( ', ', PixelYourSite\getObjectTerms( 'product_cat', $product_id ) ),
	);

    if (function_exists('PixelYourSite\getBrandForWooItem')) {
        $brand = PixelYourSite\getBrandForWooItem($product_id);
        if ($brand) {
            $line_item['product_brand'] = $brand;
        }
    }
	
	// content_name, category_name, tags
    $line_item['tags'] = implode( ', ', PixelYourSite\getObjectTerms( 'product_tag', $product_id ) );
	
	// set option names
	$value_enabled_option = $is_external ? 'woo_affiliate_value_enabled' : 'woo_add_to_cart_value_enabled';
	$value_option_option  = $is_external ? 'woo_affiliate_value_option' : 'woo_add_to_cart_value_option';
	$value_global_option  = $is_external ? 'woo_affiliate_value_global' : 'woo_add_to_cart_value_global';
	$value_percent_option = $is_external ? '' : 'woo_add_to_cart_value_percent';
	
	// currency, value
	if ( PixelYourSite\PYS()->getOption( $value_enabled_option ) ) {

		$value_option   = PixelYourSite\PYS()->getOption( $value_option_option );
		$global_value   = PixelYourSite\PYS()->getOption( $value_global_option, 0 );
		$percents_value = PixelYourSite\PYS()->getOption( $value_percent_option, 100 );
		
		$params['value']    = PixelYourSite\getWooEventValue( $value_option, $global_value, $percents_value,$product_id,$qty );
		$params['currency'] = get_woocommerce_currency();
		
	}

    $params['line_items'][] = $line_item;
	
	if ( $is_external ) {
		$params['action'] = 'affiliate button click';
	}
	
	return $params;
	
}

/**
 * @param PixelYourSite\SingleEvent $event
 */
function getWooEventCartSubtotal($event) {
    $subTotal = 0;
    $include_tax = get_option( 'woocommerce_tax_display_cart' ) == 'incl';
    foreach ($event->args['products'] as $product) {
        $subTotal += $product['subtotal'];
        if($include_tax) {
            $subTotal += $product['subtotal_tax'];
        }
    }
    return pinterest_round($subTotal);
}

/**
 * @param PixelYourSite\SingleEvent $event
 */
function getWooEventCartTotal($event) {

    return getWooEventCartSubtotal($event);
}
/**
 * @param PixelYourSite\SingleEvent $event
 */
function getWooEventOrderTotal( $event ) {

    $order = getValidWooOrder( $event );
    if ( !$order ) {
        return 0;
    }

    if ( PixelYourSite\PYS()->getOption( 'woo_event_value' ) !== 'custom' ) {
        return pinterest_round( $order->get_total() );
    }

    $total = calculateCustomWooOrderTotal( $order );
    return pinterest_round( $total );

}
function getValidWooOrder( $event ) {
    if ( empty( $event->args['order_id'] ) ) {
        return null;
    }

    $order = wc_get_order( $event->args['order_id'] );
    if ( !$order ) {
        return null;
    }

    return $order;
}
function calculateCustomWooOrderTotal( $order ) {
    $include_tax      = PixelYourSite\PYS()->getOption( 'woo_tax_option' ) === 'included';
    $include_shipping = PixelYourSite\PYS()->getOption( 'woo_shipping_option' ) === 'included';
    $include_fees     = PixelYourSite\PYS()->getOption( 'woo_fees_option' ) === 'included';

    $total = calculateOrderItemsTotal( $order, $include_tax );

    if ( $include_shipping ) {
        $total += $order->get_shipping_total();
        if ( $include_tax ) {
            $total += $order->get_shipping_tax();
        }
    }

    if ( $include_fees ) {
        foreach ( $order->get_fees() as $fee ) {
            $total += $fee->get_total();
        }
    }

    return $total;
}

function calculateOrderItemsTotal( $order, $include_tax ) {
    $total = 0;
    foreach ( $order->get_items() as $item ) {
        $line_total = $item->get_total();
        if ( $include_tax ) {
            $line_total += $item->get_total_tax();
        }
        $total += $line_total;
    }
    return $total;
}
/**
 * @param PixelYourSite\SingleEvent $event
 * @return array
 */
function getWooCartParams( $event ) {
	
	$params = array(
		'post_type' => 'product',
	);
	
	$line_items = array();
	$withTax = 'incl' === get_option( 'woocommerce_tax_display_cart' );

	foreach ($event->args['products'] as $product ) {

        $product_id = getWooCartItemId( $product );
        $content_id = getWooProductContentId( $product_id );

        $price = $product['subtotal'];
        if($withTax) {
            $price += $product['subtotal_tax'];
        }

        $item = array(
			'product_id' => $content_id,
			'product_quantity' => $product['quantity'],
			'product_price' => $product['quantity'] > 0 ? pinterest_round($price/$product['quantity']) : $price,
			'product_name' => $product['name'],
			'product_category' => implode( ', ', array_column($product['categories'],'name') ),
			'tags' => implode( ', ', $product['tags'] )
		);
		if (function_exists('PixelYourSite\getBrandForWooItem')) {
			$brand = PixelYourSite\getBrandForWooItem($product_id);
			if ($brand) {
				$item['product_brand'] = $brand;
			}
		}

		$line_items[] = $item;
	}
	
	$params['line_items'] = $line_items;
	$params = array_merge( $params, getWooCouponParams() );
	$subtotal = getWooEventCartTotal($event);
	if ( $event->getId() == 'woo_initiate_checkout' ) {
	
		$params['num_items'] = WC()->cart->get_cart_contents_count();

		$value_enabled_option = 'woo_initiate_checkout_value_enabled';
		$value_option_option  = 'woo_initiate_checkout_value_option';
		$value_global_option  = 'woo_initiate_checkout_value_global';
		$value_percent_option = 'woo_initiate_checkout_value_percent';

		$params['subtotal'] = $subtotal;
	
	} elseif ( $event->getId() == 'woo_paypal' ) {
	
		$params['num_items'] = WC()->cart->get_cart_contents_count();

		$value_enabled_option = 'woo_paypal_value_enabled';
		$value_option_option  = 'woo_paypal_value_option';
		$value_global_option  = 'woo_paypal_value_global';
		$value_percent_option = '';

		$params['subtotal'] = $subtotal;

		$params['action'] = 'PayPal';
	
	} else {
		
		$value_enabled_option = 'woo_add_to_cart_value_enabled';
		$value_option_option  = 'woo_add_to_cart_value_option';
		$value_global_option  = 'woo_add_to_cart_value_global';
		$value_percent_option = 'woo_add_to_cart_value_percent';
		
	}
	
	if ( PixelYourSite\PYS()->getOption( $value_enabled_option ) ) {
		

		$value_option   = PixelYourSite\PYS()->getOption( $value_option_option );
		$global_value   = PixelYourSite\PYS()->getOption( $value_global_option, 0 );
		$percents_value = PixelYourSite\PYS()->getOption( $value_percent_option, 100 );

        if(function_exists('PixelYourSite\getWooEventValueProducts')) {
            $params['value'] = PixelYourSite\getWooEventValueProducts( $value_option, $global_value, $percents_value,$subtotal,$event->args);
        } else {
            $params['value'] = PixelYourSite\getWooEventValueCart( $value_option, $global_value, $percents_value);
        }

		$params['currency'] = get_woocommerce_currency();
		
	}
	
	return $params;
	
}

function getWooPurchaseParams( $context ) {
    if ( ! wooIsRequestContainOrderId() ) {
        return false;
    }

    $order_id = wooGetOrderIdFromRequest();
    $order    = wc_get_order( $order_id );
    if ( ! $order ) {
        return false;
    }

    $params = buildBaseWooParams();

    $num_items = 0;
    $params['line_items']     = buildWooLineItems( $order, $num_items );
    $params['order_quantity'] = $num_items;

    $params['order_id']       = $order->get_id();

	// add "value" only on Purchase event
    if ( $context === 'Purchase' ) {
        $params['value'] = getWooPurchaseValue( $order );
    }

    $params = array_merge( $params, getWooAddressAndPayment( $order ) );

    $params['shipping'] = getWooShippingMethods( $order );

    $params = array_merge( $params, getWooCouponParams( $order ) );

    $params = array_merge( $params, getWooFeesParams( $order ) );
    $params['total'] = calculateWooTotalWithFees( $order, $params['fees'] ?? 0 );
	$params['tax']   = (float) $order->get_total_tax( 'edit' );

	if ( PixelYourSite\isWooCommerceVersionGte( '2.7' ) ) {
		$params['shipping_cost'] = (float) $order->get_shipping_total( 'edit' ) + (float) $order->get_shipping_tax( 'edit' );
	} else {
		$params['shipping_cost'] = (float) $order->get_total_shipping() + (float) $order->get_shipping_tax();
	}

    $params = array_merge( $params, getWooCustomerLifetimeParams() );
    return $params;
}

function buildBaseWooParams() {
    return [
        'post_type' => 'product',
        'currency'  => get_woocommerce_currency(),
    ];
}

function buildWooLineItems( $order, &$num_items ) {
    $line_items = [];

    foreach ( $order->get_items( 'line_item' ) as $item ) {
        $product_id = getWooCartItemId( $item );
        $content_id = getWooProductContentId( $product_id );
        $cd_params  = getWooCustomAudiencesOptimizationParams( $product_id );
        $tags       = PixelYourSite\getObjectTerms( 'product_tag', $product_id );

        $tmp_item = [
            'product_id'       => $content_id,
            'product_quantity' => $item['qty'],
            'product_price'    => PixelYourSite\getWooProductPriceToDisplay( $product_id, 1 ),
            'product_name'     => $cd_params['content_name'],
            'product_category' => $cd_params['category_name'],
            'tags'             => implode( ', ', $tags )
        ];

	    if (function_exists('PixelYourSite\getBrandForWooItem')) {
		    $brand = PixelYourSite\getBrandForWooItem($product_id);
		    if ($brand) {
			    $tmp_item['product_brand'] = $brand;
		    }
	    }


	    $line_items[] = $tmp_item;

        $num_items += $item['qty'];
    }
    return $line_items;
}

function getWooPurchaseValue( $order ) {
    $value_option   = PixelYourSite\PYS()->getOption( 'woo_purchase_value_option' );
    $global_value   = PixelYourSite\PYS()->getOption( 'woo_purchase_value_global', 0 );
    $percents_value = PixelYourSite\PYS()->getOption( 'woo_purchase_value_percent', 100 );

    return PixelYourSite\getWooEventValueOrder( $value_option, $order, $global_value, $percents_value );
}

function getWooShippingMethods( $order ) {
    $labels = array_map(
        fn($shipping) => $shipping['name'] ?? null,
        $order->get_items( 'shipping' )
    );

    return implode( ', ', array_filter( $labels ) );
}

function getWooCouponParams( $order = null ) {
	// If there is an object of order, we get coupons from it
	if ( $order instanceof WC_Order ) {
		$coupons = $order->get_coupons();
	} else {
		// Otherwise - from the current basket
		$coupons = WC()->cart ? WC()->cart->get_applied_coupons() : [];
	}

	if ( empty( $coupons ) ) {
		return [ 'promo_code_used' => 'no' ];
	}

	// For ordering - these are objects, for the basket - lines
	if ( $order instanceof WC_Order ) {
		$codes = array_map( fn( $coupon ) => $coupon->get_code(), $coupons );
	} else {
		$codes = $coupons;
	}

	return [
		'promo_code_used' => 'yes',
		'promo_code'      => implode( ', ', $codes )
	];
}
function getWooFeesParams( $order ) {
    $fees        = $order->get_fees();
    $fee_amount  = array_reduce( $fees, fn( $sum, $fee ) => $sum + $fee->get_total(), 0 );

    $include_fees = PixelYourSite\PYS()->getOption( 'woo_fees_option' ) == 'included' ? true : false;
    if(!$include_fees){
        $fees = $order->get_fees();
        $fee_amount = 0;

        foreach ($fees as $fee) {
            $fee_amount += $fee->get_total();
        }
    }

    return $fee_amount > 0 ? [ 'fees' => $fee_amount ] : [];
}

function calculateWooTotalWithFees( $order, $fee_amount ) {
    $include_fees = PixelYourSite\PYS()->getOption( 'woo_fees_option' ) === 'included';
    $use_custom   = PixelYourSite\PYS()->getOption( 'woo_event_value' ) === 'custom';

    $total = $order->get_total();
    if ( $use_custom && ! $include_fees ) {
        $total -= $fee_amount;
    }

    return (float) $total;
}

function getWooCustomerLifetimeParams() {
    if ( ! (
            PixelYourSite\PYS()->getOption( 'enable_woo_transactions_count_param' ) ||
            PixelYourSite\PYS()->getOption( 'enable_woo_predicted_ltv_param' ) ||
            PixelYourSite\PYS()->getOption( 'enable_woo_average_order_param' )
        ) || ! method_exists( PixelYourSite\EventsWoo::class, 'getCustomerTotals' ) ) {
        return [];
    }

    $data = PixelYourSite\EventsWoo()->getCustomerTotals();
    if ( empty( $data ) ) {
        return [];
    }

    return [
        'lifetime_value'     => $data['ltv'],
        'average_order'      => $data['avg_order_value'],
        'transactions_count' => $data['orders_count'],
    ];
}


function getWooAddressAndPayment( $order ) {
    return PixelYourSite\isWooCommerceVersionGte( '3.0.0' ) ? [
        'town'    => $order->get_billing_city(),
        'state'   => $order->get_billing_state(),
        'country' => $order->get_billing_country(),
        'payment' => $order->get_payment_method_title(),
    ] : [
        'town'    => $order->billing_city,
        'state'   => $order->billing_state,
        'country' => $order->billing_country,
        'payment' => $order->payment_method_title,
    ];
}

/**
 * @param $product_id
 * @return string
 */

function getWooProductContentId( $product_id ) {
    if(isWPMLActive()) {
        $product_id = getWPMLProductId($product_id, PixelYourSite\Pinterest());
    }
    if ( PixelYourSite\Pinterest()->getOption( 'woo_content_id' ) === 'product_sku' ) {
        $content_id = get_post_meta( $product_id, '_sku', true );
    } else {
        $content_id = (int) $product_id;
    }

    $prefix = PixelYourSite\Pinterest()->getOption( 'woo_content_id_prefix' );
    $suffix = PixelYourSite\Pinterest()->getOption( 'woo_content_id_suffix' );

    return $prefix . $content_id . $suffix;
}

function getWooCartItemId( $product ) {

    if ( PixelYourSite\Pinterest()->getOption( 'woo_variable_as_simple' )
        && isset( $product['parent_id'] )
        && $product['parent_id'] !== 0
    ) {
        $product_id = $product['parent_id'];
    } else {
        $product_id = $product['product_id'];
    }

    return $product_id;

}

/**
 * @deprecated
 * @param string $context
 * @return string[]
 */
function getWooCartParamsOld( $context = 'cart' ) {

    $line_items = array();

    foreach ( WC()->cart->cart_contents as $cart_item ) {

        $product_id = getWooCartItemId( $cart_item );
        $content_id = getWooProductContentId( $product_id );


        // content_name, category_name, tags
        $cd_params = getWooCustomAudiencesOptimizationParams( $product_id );
        $tags = PixelYourSite\getObjectTerms( 'product_tag', $product_id );

        $line_item = array(
            'product_id' => $content_id,
            'product_quantity' => $cart_item['quantity'],
            'product_price' => PixelYourSite\getWooProductPriceToDisplay( $product_id, 1 ),
            'product_name' => $cd_params['content_name'],
            'product_category' => $cd_params['category_name'],
            'tags' => implode( ', ', $tags )
        );
	    if (function_exists('PixelYourSite\getBrandForWooItem')) {
		    $brand = PixelYourSite\getBrandForWooItem($product_id);
		    if ($brand) {
			    $line_item['product_brand'] = $brand;
		    }
	    }

        $line_items[] = $line_item;

    }

    if( empty( $line_items ) ) {
        return [];
    }
    $params = array(
        'post_type' => 'product',
    );

    $params['line_items'] = $line_items;

    if ( $context == 'InitiateCheckout' ) {

        $params['num_items'] = WC()->cart->get_cart_contents_count();

        $value_enabled_option = 'woo_initiate_checkout_value_enabled';
        $value_option_option  = 'woo_initiate_checkout_value_option';
        $value_global_option  = 'woo_initiate_checkout_value_global';
        $value_percent_option = 'woo_initiate_checkout_value_percent';

        $params['subtotal'] = PixelYourSite\getWooCartSubtotal();

    } elseif ( $context == 'PayPal' ) {

        $params['num_items'] = WC()->cart->get_cart_contents_count();

        $value_enabled_option = 'woo_paypal_value_enabled';
        $value_option_option  = 'woo_paypal_value_option';
        $value_global_option  = 'woo_paypal_value_global';
        $value_percent_option = '';

        $params['subtotal'] = PixelYourSite\getWooCartSubtotal();

        $params['action'] = 'PayPal';

    } else {

        $value_enabled_option = 'woo_add_to_cart_value_enabled';
        $value_option_option  = 'woo_add_to_cart_value_option';
        $value_global_option  = 'woo_add_to_cart_value_global';
        $value_percent_option = 'woo_add_to_cart_value_percent';

    }

    if ( PixelYourSite\PYS()->getOption( $value_enabled_option ) ) {

        $value_option   = PixelYourSite\PYS()->getOption( $value_option_option );
        $global_value   = PixelYourSite\PYS()->getOption( $value_global_option, 0 );
        $percents_value = PixelYourSite\PYS()->getOption( $value_percent_option, 100 );

        $params['value']    = PixelYourSite\getWooEventValueCart( $value_option, $global_value, $percents_value);
        $params['currency'] = get_woocommerce_currency();

    }

    return $params;

}
