<?php

namespace PixelYourSite;

require_once PYS_PINTEREST_PATH . '/modules/pinterest/pinterest-server-async-task.php';

use PixelYourSite;

if ( !defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class PinterestServer {

    private const WOO_VERSION_THRESHOLD = '3.0.0';
    private const ALPHANUMERIC_ONLY_PATTERN = '/[^\w]/';
    private const DIGITS_ONLY_PATTERN = '/\D/';
    private const GET_PERSISTENCE_USER_DATA = 'PixelYourSite\get_persistence_user_data';

	private static $_instance;

    private        $pinterestAsyncTask;
	private        $access_token;
	private        $isDebug;
	private        $woo_order = 0;
	private        $edd_order = 0;

	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function __construct() {

		$this->isDebug = PYS()->getOption( 'debug_enabled' );

		add_action( 'wp_ajax_pys_pinterest_api_event', array(
			$this,
			"catchPinterestAjaxEvent"
		) );
		add_action( 'wp_ajax_nopriv_pys_pinterest_api_event', array(
			$this,
			"catchPinterestAjaxEvent"
		) );

		// initialize pinterest event async task
		$this->pinterestAsyncTask = new PinterestAsyncTask();

	}

	/**
	 * Send event in shutdown hook (not work in ajax)
	 * @param SingleEvent[] $events
	 */
	public function sendEventsAsync( $events ) {

		$serverEvents = array();

		foreach ( $events as $event ) {
            if (!array_key_exists(strtolower($event->payload['name']), $this->eventNameMap)) {
                continue;
            }
            $ids = $event->payload['pixelIds'];
            $serverEvents[] = [
                "pixelIds" => $ids,
                "event" => $this->mapEventToServerEvent($event)
            ];
		}

		if ( !empty($serverEvents)) {
			do_action( 'pys_send_pinterest_server_event', $serverEvents );
		}
	}

	/**
	 * Send Event Now
	 *
	 * @param SingleEvent[] $events
	 */
	public function sendEventsNow( $events ) {

		foreach ( $events as $event ) {
            if (!array_key_exists(strtolower($event->payload['name']), $this->eventNameMap)) {
                continue;
            }
			$serverEvent = $this->mapEventToServerEvent( $event );
			$ids = $event->payload[ 'pixelIds' ];

			$this->sendEvent( $ids, $serverEvent );
		}
	}

	public function trackAddToCartEvent( $product_id, $quantity, $variation_id ) {

		if ( EventsWoo()->isReadyForFire( "woo_add_to_cart_on_button_click" ) && PYS()->getOption( 'woo_add_to_cart_catch_method' ) == "add_cart_js" ) {
			// it ok. We send server method after js, and take event id from cookies
			Pinterest()->getLog()->debug( ' trackAddToCartEvent send Pinterest server without browser event' );

			if ( !empty( $variation_id ) && $variation_id > 0 && ( !Pinterest()->getOption( 'woo_variable_as_simple' ) ) ) {
				$_product_id = $variation_id;
			} else {
				$_product_id = $product_id;
			}

			$event = new SingleEvent( "woo_add_to_cart_on_button_click", EventTypes::$DYNAMIC, 'woo' );
			$event->args = [
				'productId' => $_product_id,
				'quantity'  => $quantity
			];
			$event->params[ 'uri' ] = self::getRequestUri( PYS()->getOption( 'enable_remove_source_url_params' ) );

            add_filter( 'pys_conditional_post_id', function () use ( $product_id ) {
                return $product_id;
            } );
			$events = Pinterest()->generateEvents( $event );
			remove_all_filters( 'pys_conditional_post_id' );

			foreach ( $events as $singleEvent ) {

				if ( isset( $_COOKIE[ "pys_pinterest_event_id" ] ) ) {
					$singleEvent->payload[ 'eventID' ] = json_decode( stripslashes( $_COOKIE[ "pys_pinterest_event_id" ] ) )->AddToCart;
				}
			}

			do_action( 'pys_send_pinterest_server_event', $events );
		}
	}

	/**
	 * If server message is blocked by gprg or it dynamic
	 * we send data by ajax request from js and send the same data like browser event
	 */
	public function catchPinterestAjaxEvent() {

		Pinterest()->getLog()->debug( ' catchPinterestAjaxEvent send to Pinterest server from ajax' );
		$event = $_POST[ 'event' ];
		$data = isset( $_POST[ 'data' ] ) ? $_POST[ 'data' ] : array();
		$ids = $_POST[ 'ids' ];
		$eventID = $_POST[ 'eventID' ];
		$wooOrder = isset( $_POST[ 'woo_order' ] ) ? $_POST[ 'woo_order' ] : null;
		$eddOrder = isset( $_POST[ 'edd_order' ] ) ? $_POST[ 'edd_order' ] : null;

		if ( empty( $_REQUEST[ 'ajax_event' ] ) || !wp_verify_nonce( $_REQUEST[ 'ajax_event' ], 'ajax-event-nonce' ) ) {
			wp_die();
			return;
		}

		if ( $event == "hCR" ) { $event = "CompleteRegistration"; }// de mask completer registration event if it was hidden

		$singleEvent = $this->dataToSingleEvent( $event, $data, $eventID, $ids, $wooOrder, $eddOrder );

		$this->sendEventsNow( [ $singleEvent ] );

		wp_die();
	}

	/**
	 * @param $eventName
	 * @param $params
	 * @param $eventID
	 * @param $ids
	 * @param $wooOrder
	 * @param $eddOrder
	 * @return SingleEvent
	 */
	private function dataToSingleEvent( $eventName, $params, $eventID, $ids, $wooOrder, $eddOrder ) {
		$singleEvent = new SingleEvent( "", "" );

		$payload = [
			'name'      => $eventName,
			'eventID'   => $eventID,
			'woo_order' => $wooOrder,
			'edd_order' => $eddOrder,
			'pixelIds'  => $ids
		];
		$singleEvent->addParams( $params );
		$singleEvent->addPayload( $payload );

		return $singleEvent;
	}


	/**
	 * Send event for each pixel id
	 * @param array $pixel_Ids //array of facebook ids
	 * @param $event //One Facebook event object
	 */
	public function sendEvent( $pixel_Ids, $event ) {

		if ( !$event || apply_filters( 'pys_disable_server_event_filter', false ) ) {
			return;
		}

		if ( !$this->access_token ) {
			$this->access_token = Pinterest()->getApiTokens();
		}

		foreach ( $pixel_Ids as $pixel_Id ) {

			if ( !Pinterest()->enabled() || empty( $this->access_token[ $pixel_Id ] || empty( $this->access_token[ $pixel_Id ][ 'server_id' ] ) ) || empty( $this->access_token[ $pixel_Id ][ 'ad_account' ] ) ) {
                continue;
            }

			$data = new \stdClass;
			$data->data[] = $event;

			$url = "https://api.pinterest.com/v5/ad_accounts/{$this->access_token[ $pixel_Id ][ 'ad_account' ]}/events";
			// Add ?test=true if test_mode is enabled
			if ( Pinterest()->getOption('test_mode') ) {
				$url .= '?test=true';
			}
			$headers = array(
				'Content-Type'  => 'application/json',
				'Authorization' => "Bearer " . $this->access_token[ $pixel_Id ][ 'server_id' ]
			);

			Pinterest()->getLog()->debug( ' Send Pinterest server event', $data );
			try {
				$client = new \PYS_PRO_GLOBAL\GuzzleHttp\Client();
				$response = $client->request( 'POST', $url, [
					'headers' => $headers,
					'body'    => json_encode( $data )
				] );
				Pinterest()->getLog()->debug( ' Response from Pinterest server', $response );

			} catch ( \Exception $e ) {
				Pinterest()->getLog()->error( 'Error send Pinterest server event ' . $e->getMessage() );
			}
		}
	}
    private $eventNameMap = [
        'checkout'         => 'checkout',
        'addtocart'        => 'add_to_cart',
        'pagevisit'        => 'page_visit',
        'signup'           => 'signup',
        'watchvideo'       => 'watch_video',
        'lead'             => 'lead',
        'search'           => 'search',
        'viewcategory'     => 'view_category',
        'addpaymentinfo'   => 'add_payment_info',
        'addtowishlist'    => 'add_to_wishlist',
        'initiatecheckout' => 'initiate_checkout',
        'subscribe'        => 'subscribe',
        'viewcontent'      => 'view_content',
        'custom'           => 'custom',
    ];
	public function mapEventToServerEvent( $event ) {

		$eventData = $event->getData();

		$eventData = EventsManager::filterEventParams( $eventData, $event->getCategory(), [
			'event_id' => $event->getId(),
			'pixel'    => Pinterest()->getSlug()
		] );

        $eventName = $eventData['name'];
        $eventName = $this->eventNameMap[$eventName] ?? strtolower($eventName);
		$wooOrder = isset( $event->payload[ 'woo_order' ] ) ? $event->payload[ 'woo_order' ] : null;
		$eddOrder = isset( $event->payload[ 'edd_order' ] ) ? $event->payload[ 'edd_order' ] : null;

		$user_data = $this->getUserData( $wooOrder, $eddOrder );
		$custom_data = $this->paramsToCustomData( $eventData[ 'params' ] );

        $uri = $this->getEventUri($event);

		$serverEvent = new \stdClass;
		$serverEvent->event_name = $eventName;
		$serverEvent->event_time = time();
		$serverEvent->event_source_url = $uri;
		$serverEvent->action_source = 'website';
		$serverEvent->user_data = $user_data;
		$serverEvent->custom_data = $custom_data;

		if ( isset( $event->payload[ 'eventID' ] ) ) {
			$serverEvent->event_id = $event->payload[ 'eventID' ];
		} else {
			$serverEvent->event_id = '';
		}

		return $serverEvent;
	}

    private function getUserData($wooOrder = null, $eddOrder = null) {
        if (!$this->access_token) {
            $this->access_token = Pinterest()->getApiTokens();
        }

        if ($this->shouldUseWoo($wooOrder)) {
            $userData = $this->getWooUserData($wooOrder);
        } elseif ($this->shouldUseEdd($eddOrder)) {
            $userData = $this->getEddUserData($eddOrder);
        } else {
            $userData = $this->getRegularUserData();
        }

        $userData->client_ip_address = self::getIpAddress();
        $userData->client_user_agent = self::getHttpUserAgent();

        if(isset($_COOKIE['_epik']) && !empty($_COOKIE['_epik'])) {
            $userData->click_id = $_COOKIE['_epik'];
        }

        if (PixelYourSite\EventsManager::isTrackExternalId() && PixelYourSite\PYS()->get_pbid()) {
            $userData->external_id[] = PixelYourSite\PYS()->get_pbid();
        }
        return apply_filters("pys_pinterest_server_user_data", $userData);
    }

    private function shouldUseWoo($wooOrder): bool {
        return PixelYourSite\isWooCommerceActive()
            && isEventEnabled('woo_purchase_enabled')
            && ($wooOrder || (PYS()->woo_is_order_received_page(is_order_received_page()) && wooIsRequestContainOrderId()));
    }

    private function shouldUseEdd($eddOrder): bool {
        return PixelYourSite\isEddActive()
            && isEventEnabled('edd_purchase_enabled')
            && ($eddOrder || edd_is_success_page());
    }

    private function getWooUserData($wooOrder) {
        $order_id = $wooOrder ?? wooGetOrderIdFromRequest();
        $order = wc_get_order($order_id);
        if (!$order) {
            return $this->getRegularUserData();
        }

        $this->woo_order = $order_id;

        $email = PixelYourSite\isWooCommerceVersionGte(self::WOO_VERSION_THRESHOLD) ? $order->get_billing_email() : $order->billing_email;
        $phone = PixelYourSite\isWooCommerceVersionGte(self::WOO_VERSION_THRESHOLD) ? $order->get_billing_phone() : $order->billing_phone;
        $fname = PixelYourSite\isWooCommerceVersionGte(self::WOO_VERSION_THRESHOLD) ? $order->get_billing_first_name() : $order->billing_first_name;
        $lname = PixelYourSite\isWooCommerceVersionGte(self::WOO_VERSION_THRESHOLD) ? $order->get_billing_last_name() : $order->billing_last_name;

        $userData = new \stdClass;
        $this->addAddressDataFromOrder($userData, $order);
        $this->addPersonalData($userData, $email, $phone, $fname, $lname);

        return $userData;
    }

    private function getEddUserData($eddOrder) {
        $payment_id = $eddOrder ?? (int) edd_get_purchase_id_by_key(getEddPaymentKey());
        $this->edd_order = $eddOrder;

        $user_info = edd_get_payment_meta_user_info($payment_id);
        $fname = $user_info['first_name'] ?? '';
        $lname = $user_info['last_name'] ?? '';
        $email = edd_get_payment_user_email($payment_id);
        $phone = $user_info['phone'] ?? '';

        $userData = new \stdClass;
        $this->addPersonalData($userData, $email, $phone, $fname, $lname);

        return $userData;
    }

    private function addAddressDataFromOrder(&$data, $order) {
        $fields = [
            'ct' => $order->get_billing_city() ?? $order->billing_city,
            'st' => $order->get_billing_state() ?? $order->billing_state,
            'zp' => $order->get_billing_postcode() ?? $order->billing_postcode,
            'country' => $order->get_billing_country() ?? $order->billing_country,
        ];

        foreach ($fields as $key => $val) {
            if (!empty($val)) {
                $filtered = ($key === 'ct') ? preg_replace(self::ALPHANUMERIC_ONLY_PATTERN, '', $val) : $val;
                $data->$key[] = hash('sha256', mb_strtolower($filtered), false);
            }
        }
    }

    private function addPersonalData(&$data, $email, $phone, $fname, $lname) {
        if (function_exists(self::GET_PERSISTENCE_USER_DATA)) {
            $persist = call_user_func(self::GET_PERSISTENCE_USER_DATA, $email, $fname, $lname, $phone);
            $this->hashAndSetField($data, 'em', $persist['em'] ?? $email);
            $this->hashAndSetField($data, 'ph', $persist['tel'] ?? $phone, true);
            $this->hashAndSetField($data, 'fn', $persist['fn'] ?? $fname);
            $this->hashAndSetField($data, 'ln', $persist['ln'] ?? $lname);
        } elseif (isset($_COOKIE['pys_advanced_form_data'])) {
            $form = json_decode(stripslashes($_COOKIE['pys_advanced_form_data']), true);
            $this->hashAndSetField($data, 'em', $email ?: ($form['email'] ?? ''));
            $this->hashAndSetField($data, 'ph', $phone ?: ($form['phone'] ?? ''), true);
            $this->hashAndSetField($data, 'fn', $fname ?: ($form['first_name'] ?? ''));
            $this->hashAndSetField($data, 'ln', $lname ?: ($form['last_name'] ?? ''));
        }
    }

    private function hashAndSetField(&$data, $field, $value, $digitsOnly = false) {
        if (!empty($value)) {
            if ($digitsOnly) {
                $value = preg_replace(self::DIGITS_ONLY_PATTERN, '', $value);
            }
            $data->$field[] = hash('sha256', mb_strtolower($value), false);
        }
    }


    private function getRegularUserData() {
		$user = wp_get_current_user();
		$userData = new \stdClass;

		if ( $user->ID ) {
            $this->populateUserDataFromMeta($user, $userData);
            $this->populateWooCommerceData($user, $userData);
		}

        $this->populatePersistenceData($user, $userData);
		return apply_filters( "pys_pinterest_server_user_data", $userData );
	}

	private function paramsToCustomData( $data ) {

		$custom_data = new \stdClass;

        $this->processLineItems($data, $custom_data);

		if ( isset( $data[ 'currency' ] ) ) {
			$custom_data->currency = $data[ 'currency' ];
		}

        $order_id = ($this->woo_order !== 0) ? $this->woo_order : $this->edd_order;

		if ( $order_id !== 0 ) {
			$custom_data->order_id = (string) $order_id;
		}

		if ( !empty( $_GET[ 's' ] ) ) {
			$custom_data->search_string = $_GET[ 's' ];
		}

		return apply_filters( "pys_pinterest_server_custom_data", $custom_data );
	}

    private function processLineItems($data, &$custom_data) {

        if (isset($data['line_items']) && is_array($data['line_items'])) {
            $contents = [];
            $ids = [];
            $cost = 0;
            $num_items = 0;

            foreach ($data['line_items'] as $item) {
                $this->processSingleItem($item, $contents, $ids, $cost, $num_items);
            }

            $custom_data->contents = $contents;
            $custom_data->value = (string) $cost;
            $custom_data->content_ids = $ids;
            $custom_data->num_items = $num_items;
        } else {
            $custom_data->content_ids = [$data['post_id']];
            $custom_data->contents = [];
        }
    }

    private function processSingleItem($item, &$contents, &$ids, &$cost, &$num_items) {
        if (isset($item['product_quantity'])) {
            $contents[] = [
                'id'    => $item['product_id'] ?? '',
                'item_name'  => $item['product_name'] ?? '',
                'item_price' => (string) $item['product_price'],
                'item_category' => $item['product_category'] ?? '',
                'item_brand' => $item['product_brand'] ?? '',
                'quantity' => (int) $item['product_quantity'], // default quantity is 1
            ];
            $cost += $item['product_quantity'] * ($item['product_price'] ?? 0);
            $num_items++;
        }

        if (isset($item['product_id'])) {
            $ids[] = strval($item['product_id']);
        }
    }

	private static function getRequestUri( $removeQuery = false ) {
		$request_uri = null;

		if ( !empty( $_SERVER[ 'REQUEST_URI' ] ) ) {
			$start = ( isset( $_SERVER[ 'HTTPS' ] ) && $_SERVER[ 'HTTPS' ] === 'on' ? "https" : "http" ) . "://";
			$request_uri = $start . ($_SERVER['HTTP_HOST'] ?? parse_url(get_site_url(), PHP_URL_HOST)) . $_SERVER[ 'REQUEST_URI' ];
		}
		if ( $removeQuery && isset( $_SERVER[ 'QUERY_STRING' ] ) ) {
			$request_uri = str_replace( "?" . $_SERVER[ 'QUERY_STRING' ], "", $request_uri );
		}

		return $request_uri;
	}

	private static function getIpAddress() {
		$headersToScan = array(
			'HTTP_CLIENT_IP',
			'HTTP_X_FORWARDED_FOR',
			'HTTP_X_FORWARDED',
			'HTTP_X_CLUSTER_CLIENT_IP',
			'HTTP_FORWARDED_FOR',
			'HTTP_FORWARDED',
			'REMOTE_ADDR'
		);

		foreach ( $headersToScan as $header ) {
			if ( array_key_exists( $header, $_SERVER ) ) {
				$ip_list = explode( ',', $_SERVER[ $header ] );
				foreach ( $ip_list as $ip ) {
					$trimmed_ip = trim( $ip );
					if ( self::isValidIpAddress( $trimmed_ip ) ) {
						return $trimmed_ip;
					}
				}
			}
		}

		return "127.0.0.1";
	}

	private static function isValidIpAddress( $ip_address ) {
		return filter_var( $ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6 | FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );
	}

	private static function getHttpUserAgent() {
		$user_agent = null;

		if ( !empty( $_SERVER[ 'HTTP_USER_AGENT' ] ) ) {
			$user_agent = $_SERVER[ 'HTTP_USER_AGENT' ];
		}

		return $user_agent;
	}

    private function getEventUri($event) {
        if (isset($event->params['uri'])) {
            return $event->params['uri'];
        }

        $uri = self::getRequestUri(PYS()->getOption('enable_remove_source_url_params'));
        if (isset($_POST['url'])) {
            $uri = $this->sanitizeUri($_POST['url']);
        }

        return $uri;
    }

    private function sanitizeUri($url) {
        if (PYS()->getOption('enable_remove_source_url_params')) {
            $list = explode("?", esc_url_raw($url));
            return is_array($list) && !empty($list) ? $list[0] : esc_url_raw($url);
        }
        return esc_url_raw($url);
    }

    private function populateUserDataFromMeta($user, &$userData) {
        $user_first_name = $user->get('user_firstname');
        $user_last_name = $user->get('user_lastname');
        $user_email = $user->get('user_email');
        $user_phone = $user->get('billing_phone');

        $userData->fn[] = hash('sha256', mb_strtolower($user_first_name), false);
        $userData->ln[] = hash('sha256', mb_strtolower($user_last_name), false);
        $userData->em[] = hash('sha256', mb_strtolower($user_email), false);
        $userData->ph[] = hash('sha256', preg_replace(self::DIGITS_ONLY_PATTERN, '', $user_phone), false);
    }

    private function populateWooCommerceData($user, &$userData) {
        if (PixelYourSite\isWooCommerceActive()) {
            $userData->ct[] = hash('sha256', mb_strtolower(preg_replace(self::ALPHANUMERIC_ONLY_PATTERN, '', $user->get('billing_city'))), false);
            $userData->st[] = hash('sha256', mb_strtolower($user->get('billing_state')), false);
            $userData->country[] = hash('sha256', mb_strtolower($user->get('shipping_country')), false);
            $userData->zp[] = hash('sha256', $user->get('billing_postcode'), false);
        }
    }

    private function populatePersistenceData($user, &$userData) {
        $user_email = $user->get('user_email');
        $user_first_name = $user->get('user_firstname');
        $user_last_name = $user->get('user_lastname');
        $user_phone = $user->get('billing_phone');

        if (function_exists(self::GET_PERSISTENCE_USER_DATA)) {
            $user_persistence_data = call_user_func(self::GET_PERSISTENCE_USER_DATA, $user_email, $user_first_name, $user_last_name, $user_phone);
            if (!empty($user_persistence_data['em'])) {
                $userData->em[] = hash('sha256', mb_strtolower($user_persistence_data['em']), false);
            }
            if (!empty($user_persistence_data['fn'])) {
                $userData->fn[] = hash('sha256', mb_strtolower($user_persistence_data['fn']), false);
            }
            if (!empty($user_persistence_data['ln'])) {
                $userData->ln[] = hash('sha256', mb_strtolower($user_persistence_data['ln']), false);
            }
            if (!empty($user_persistence_data['tel'])) {
                $userData->ph[] = hash('sha256', preg_replace(self::DIGITS_ONLY_PATTERN, '', $user_persistence_data['tel']), false);
            }
        }
    }

}

/**
 * @return PinterestServer
 */
function PinterestServer() {
	return PinterestServer::instance();
}

PinterestServer();
