<?php

/**
 * @link: https://developers.pinterest.com/docs/ad-tools/conversion-tag/
 */

namespace PixelYourSite;

use PixelYourSite\PinterestMainHelper;

use function PixelYourSite\Pinterest\getEddDownloadContentId;
use function PixelYourSite\Pinterest\getWooProductContentId;
use function PixelYourSite\Pinterest\pinterest_round;

if ( !defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Pinterest extends Settings implements Pixel, Plugin {

	private static $instance;

	private $configured;

	private $coreCompatible;

	private $logger;

	private $pageVisitEvent = false;

    private $pinterestMainHelper;

	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;

	}

	public function __construct() {

		// cache status
		if ( Pinterest\isPysProActive() ) {
			$this->coreCompatible = Pinterest\pysProVersionIsCompatible();
		} else {
			$this->coreCompatible = Pinterest\pysFreeVersionIsCompatible();
		}

		parent::__construct( 'pinterest' );
        $this->pinterestMainHelper = new PinterestMainHelper($this);

       if($this->enabled() && $this->isServerApiEnabled()){
           require_once PYS_PINTEREST_PATH . '/modules/pinterest/pinterest-server.php';
       }


		require_once PYS_PINTEREST_PATH . '/modules/pinterest/pinterest-logger.php';

		$this->locateOptions( PYS_PINTEREST_PATH . '/modules/pinterest/options_fields.json', PYS_PINTEREST_PATH . '/modules/pinterest/options_defaults.json' );

		// migrate after event post type registered
		add_action( 'pys_register_pixels', 'PixelYourSite\Pinterest\maybeMigrate' );

		add_action( 'pys_register_plugins', function ( $core ) {
			/** @var PYS $core */
			$core->registerPlugin( $this );
		} );

		if ( !$this->coreCompatible ) {
			return;
		}

		add_action( 'pys_register_pixels', function ( $core ) {
			/** @var PYS $core */
			$core->registerPixel( $this );
		} );

		if ( $this->configured() ) {

			// output debug info
			if ( !PYS()->getOption( 'hide_version_plugin_in_console' ) ) {
				add_action( 'wp_head', function () {
					echo "<script type='text/javascript' id='pinterest-version-script'>console.log('PixelYourSite Pinterest version " . PYS_PINTEREST_VERSION . "');</script>\r\n";
				}, 2 );
			}
			// load addon's public JS
			add_action( 'wp_enqueue_scripts', function () {
				wp_enqueue_script( 'pys-pinterest', PYS_PINTEREST_URL . '/dist/scripts/public.js', array( 'pys' ), PYS_PINTEREST_VERSION );
			} );

		}

		$this->logger = new Pinterest_logger;
		add_action( 'init', array(
			$this,
			'init'
		), 9 );

		add_action( 'pys_admin_pixel_ids', 'PixelYourSite\Pinterest\renderPixelIdField' );
		add_filter( 'pys_admin_secondary_nav_tabs', 'PixelYourSite\Pinterest\adminSecondaryNavTabs' );
        add_filter('pys_pinterest_settings_sanitize_verify_meta_tag_field', array($this, 'sanitize_verify_meta_tag_field'));
		add_action( 'pys_admin_pinterest_settings', 'PixelYourSite\Pinterest\renderSettingsPage' );
		add_action( 'wp_head', array(
			$this,
			'output_meta_tag'
		) );
	}

	public function init() {
		$this->logger->init();
	}

	/**
	 * Returns cached core compatibility status.
	 *
	 * @return bool
	 */
	public function getCoreCompatible() {
		return $this->coreCompatible;
	}

	public function enabled() {
		if ( isSuperPackActive() && SuperPack()->getOption( 'enabled' ) ) {
			return true;
		} else {
			return $this->getOption( 'main_pixel_enabled' );
		}
	}

	public function configured() {

		$license_status = $this->getOption( 'license_status' );
		$pixel_id = $this->getAllPixels();
		$disabled = false;
		$main_pixel_enabled = $this->getOption( 'main_pixel_enabled' );

		if ( isSuperPackActive() && version_compare( SuperPack()->getPluginVersion(), '3.1.1.1', '>=' ) ) {
			$disabledPixel = apply_filters( 'pys_pixel_disabled', array(), $this->getSlug() );
			$disabled = in_array( '1', $disabledPixel ) && in_array( 'all', $disabledPixel );
			$main_pixel_enabled = true;
		}

		$this->configured = !empty( $license_status ) // license was activated before
			&& count( $pixel_id ) > 0
			&& $main_pixel_enabled
			&& !$disabled;

		return $this->configured;
	}

	public function getPixelIDs() {

		if( isSuperPackActive()
			&& SuperPack()->getOption( 'enabled' )
			&& SuperPack()->getOption( 'additional_ids_enabled' )
            && !$this->getOption( 'main_pixel_enabled' ))
        {
            return apply_filters( "hide_pixels", [] );
		}

        $ids = (array) $this->getOption( 'pixel_id' );

        $hide_pixels = apply_filters('hide_pixels', array());

        $pixels = array_filter($ids, static function ($element) use ($hide_pixels) {
            return !in_array($element, $hide_pixels);
        });
        $pixels = array_values($pixels);

        return $pixels; // return first id only
	}

	public function getAllPixels() {
		return $this->getPixelIDs();
	}

	/**
	 * @param SingleEvent $event
	 */
	public function getAllPixelsForEvent( $event ) {
		$pixels = array();
		$main_pixel = $this->getPixelIDs();

		if(isSuperPackActive('3.0.0')
			&& SuperPack()->getOption( 'enabled' )
			&& SuperPack()->getOption( 'additional_ids_enabled' ))
		{
			if ( !empty( $main_pixel ) ) {
				$main_pixel_options = $this->getOption( 'main_pixel' );
				if ( !empty( $main_pixel_options ) && isset( $main_pixel_options[ 0 ] ) ) {
					$main_pixel_options = $this->normalizeSPOptions( $main_pixel[ 0 ], $main_pixel_options[ 0 ] );
				} else {
					$main_pixel_options = $this->normalizeSPOptions( $main_pixel[ 0 ], '' );
				}
				$pixel_options = SuperPack\SPPixelId::fromArray( $main_pixel_options );
				if ( $pixel_options->isValidForEvent( $event ) && $pixel_options->isConditionalValidForEvent( $event ) ) {
					$pixels = array_merge( $pixels, $main_pixel );
				}
			}
		} elseif ( $this->getOption( 'main_pixel_enabled' ) ) {
			$pixels = array_merge( $pixels, $main_pixel );
		}

		return $pixels;
	}

	public function getPixelOptions() {

        $options = array(
            'pixelIds'            => $this->getPixelIDs(),
            'advancedMatching'    => $this->getOption( 'enhanced_matching_enabled' ) ? Pinterest\getEnhancedMatchingParams() : array(),
            'wooVariableAsSimple' => $this->getOption( 'woo_variable_as_simple' ),
            'serverApiEnabled'    => $this->getOption( 'use_server_api' ),
        );
        if(isSuperPackActive('3.3.1') && SuperPack()->getOption( 'enabled' ) && SuperPack()->getOption( 'enable_hide_this_tag_by_tags' )){
            $options['hide_pixels'] = $this->getHideInfoPixels();
        }

        return $options;

	}

	public function getPluginName() {
		return 'PixelYourSite Pinterest Add-On';
	}

	public function getPluginFile() {
		return PYS_PINTEREST_PLUGIN_FILE;
	}

	public function getPluginVersion() {
		return PYS_PINTEREST_VERSION;
	}

    public function getPluginIcon() {
        return PYS_PINTEREST_PLUGIN_ICON;
    }

	public function adminUpdateLicense() {
		if ( PYS()->adminSecurityCheck() ) {
			updateLicense( $this );
		}
	}

	public function adminRenderPluginOptions() {
		// for backward compatibility with PRO < 7.0.6
	}

	public function updatePlugin() {
		// for backward compatibility with PRO < 7.0.6
	}

	/**
	 * @param CustomEvent $event
	 */
	public function renderCustomEventOptions( $event ) {
		/** @noinspection PhpIncludeInspection */
		include_once PYS_PINTEREST_PATH . '/modules/pinterest/views/html-main-events-edit.php';
	}

	/**
	 * Create pixel event and fill it
	 * @param SingleEvent $event
	 */
	public function generateEvents( $event ) {
		$pixelEvents = [];
		$pixelIds = $this->getAllPixelsForEvent( $event );

        $disabledPixel = Pinterest\getDisabledPixels($this->getSlug());
		// filter disabled pixels
        $pixelIds = Pinterest\filterDisabledPixels($pixelIds, $disabledPixel);

        if(Pinterest\isEventWithProducts($event->getId()) && isSuperPackActive('5.3.1')
            && SuperPack()->getOption( 'enabled' ) )
        {
            Pinterest\filterEventsPixel($event, $this, $pixelEvents);
        }
        else
        {
            if ( count( $pixelIds ) > 0 ) {
                $pixelEvent = clone $event;
                if ( $this->addParamsToEvent( $pixelEvent ) ) {
                    $pixelEvent->addPayload( [ 'pixelIds' => $pixelIds ] );
                    $pixelEvents[] = $pixelEvent;
                }
            }
        }
		return $pixelEvents;
	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
	public function addParamsToEvent( &$event ) {
        if (!$this->configured()) {
            return false;
        }

        $handler = new PinterestEventHandler();
        $eventId = $event->getId();

        switch (true) {
            case $eventId === 'init_event':
                $isActive = $handler->handleInitEvent(
                    $event,
                    $this->pageVisitEvent,
                    [$this, 'getPageViewEventParams']
                );
                break;

            case in_array($eventId, ['automatic_event_signup', 'automatic_event_login', 'automatic_event_search']):
                $isActive = $handler->handleAutomaticEvent($event, [$this, 'getOption']);
                break;

            case strpos($eventId, 'automatic_event_') === 0:
                $isActive = $handler->handleSimpleAutomaticEvent($event, [$this, 'getOption']);
                break;

            case $eventId === 'woo_view_content':
                $isActive = $handler->handleWooViewContentEvent($event, [$this, 'getWooPageVisitEventParams']);
                $this->pageVisitEvent = $handler->getPageVisitEvent();
                break;

            case in_array($eventId, ['woo_add_to_cart_on_cart_page', 'woo_add_to_cart_on_checkout_page']):
                $isActive = $handler->handleWooAddToCartOnCartPageEvent(
                    $event,
                    [$this, 'getWooAddToCartOnCartEventParams'],
                    [$this, 'setWooAddToCartOnCartEventParams']
                );
                break;

            case $eventId === 'woo_remove_from_cart':
                $isActive = $handler->handleWooRemoveFromCartEvent(
                    $event,
                    [$this, 'getWooRemoveFromCartParams'],
                    [$this, 'setWooRemoveFromCartParams']
                );
                break;

            case $eventId === 'woo_view_category':
                $isActive = $handler->handleWooViewCategoryEvent($event, [$this, 'getWooViewCategoryEventParams']);
                break;

            case $eventId === 'woo_initiate_checkout':
                $isActive = $handler->handleWooInitiateCheckoutEvent(
                    $event,
                    [$this, 'getWooInitiateCheckoutEventParams'],
                    [$this, 'setWooInitiateCheckoutEventParams']
                );
                break;

            case $eventId === 'woo_purchase':
                $isActive = $handler->handleWooPurchaseEvent(
                    $event,
                    [$this, 'getWooCheckoutEventParams'],
                    [$this, 'addWooCheckoutEventParams']
                );
                break;

            case $eventId === 'woo_paypal':
                $isActive = $handler->handleWooPayPalEvent($event, [$this, 'setWooPayPalEventParams']);
                break;

            case in_array($eventId, ['woo_frequent_shopper', 'woo_vip_client', 'woo_big_whale', 'woo_FirstTimeBuyer', 'woo_ReturningCustomer']):
                $isActive = $handler->handleWooAdvancedMarketingEvent($event, [$this, 'getWooAdvancedMarketingEventParams']);
                break;

            case $eventId === 'woo_add_to_cart_on_button_click':
                $isActive = $handler->handleWooAddToCartOnButtonClickEvent(
                    $event,
                    [$this, 'getOption'],
                    [$this, 'getWooAddToCartOnButtonClickEventParams']
                );
                break;

            case $eventId === 'woo_affiliate':
                $isActive = $handler->handleWooAffiliateEvent(
                    $event,
                    [$this, 'getOption'],
                    [$this, 'getWooAffiliateEventParams']
                );
                break;

            case $eventId === 'edd_view_content':
                $isActive = $handler->handleEddViewContentEvent($event, [$this, 'getEddPageVisitEventParams']);
                $this->pageVisitEvent = $handler->getPageVisitEvent();
                break;

            case $eventId === 'edd_add_to_cart_on_checkout_page':
                $isActive = $handler->handleEddAddToCartOnCheckoutPageEvent($event, [$this, 'setEddCartEventParams']);
                break;

            case $eventId === 'edd_remove_from_cart':
                $isActive = $handler->handleEddRemoveFromCartEvent(
                    $event,
                    [$this, 'getEddRemoveFromCartParams'],
                    [$this, 'setEddRemoveFromCartParams']
                );
                break;

            case $eventId === 'edd_view_category':
                $isActive = $handler->handleEddViewCategoryEvent($event, [$this, 'getEddViewCategoryEventParams']);
                break;

            case in_array($eventId, ['edd_initiate_checkout', 'edd_purchase']):
                $isActive = $handler->handleEddInitiateCheckoutEvent($event, [$this, 'setEddCartEventParams']);
                break;

            case in_array($eventId, ['edd_frequent_shopper', 'edd_vip_client', 'edd_big_whale']):
                $isActive = $handler->handleEddAdvancedMarketingEvent($event, [$this, 'setEddCartEventParams']);
                break;

            case $eventId === 'edd_add_to_cart_on_button_click':
                $isActive = $handler->handleEddAddToCartOnButtonClickEvent(
                    $event,
                    [$this, 'getOption'],
                    [$this, 'getEddAddToCartOnButtonClickEventParams']
                );
                break;

            case $eventId === 'wcf_view_content':
                $isActive = $handler->handleWcfViewContentEvent($event, [$this, 'getWcfViewContentEventParams']);
                $this->pageVisitEvent = $handler->getPageVisitEvent();
                break;

            case in_array($eventId, ['wcf_add_to_cart_on_bump_click', 'wcf_add_to_cart_on_next_step_click']):
                $isActive = $handler->handleWcfAddToCartEvent($event, [$this, 'prepareWcfAddToCart']);
                break;

            case $eventId === 'wcf_remove_from_cart_on_bump_click':
                $isActive = $handler->handleWcfRemoveFromCartEvent($event, [$this, 'prepareWcfRemoveFromCart']);
                break;

            case in_array($eventId, ['wcf_step_page', 'wcf_bump', 'wcf_page']):
                $isActive = $handler->handleSimpleWcfEvent($event, [$this, 'getOption']);
                break;

            case $eventId === 'custom_event':
                $isActive = $handler->handleCustomEvent($event, [$this, 'getCustomEventParams']);
                break;

            default:
                $isActive = false;
                break;
        }

        if ($isActive) {
            $event->payload['eventID'] = Pinterest\pys_generate_token();
        }

        return $isActive;
	}

	public function getEventData( $eventType, $args = null ) {
		return false;
	}

	public function outputNoScriptEvents() {

        if (!$this->configured() || $this->getOption('disable_noscript')) {
            return;
        }

        $helper = new PinterestNoScriptHelper($this->getPixelIDs());
        $helper->outputNoScriptEvents(PYS()->getEventsManager());


	}

	public function renderAddonNotice() {
        return false;
	}


	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function getWcfViewContentEventParams( &$event ) {
		if ( !$this->getOption( 'woo_view_content_enabled' ) || empty( $event->args[ 'products' ] ) || !$this->getPageViewEventParams() ) {
			return false;
		}
		$product_data = $event->args[ 'products' ][ 0 ];
		$params = array(
			'post_type'     => 'product',
			'product_id'    => getWooProductContentId( $product_data[ 'id' ] ),
			'product_price' => getWooProductPriceToDisplay( $product_data[ 'id' ], 1, $product_data[ 'price' ] ),
			'tags'          => implode( ', ', $product_data[ 'tags' ] ),
			'content_name'  => $product_data[ 'name' ],
			'category_name' => implode( ', ', array_column( $product_data[ 'categories' ], "name" ) ),
			'currency'      => $event->args[ 'currency' ]
		);
		if ( PYS()->getOption( 'woo_view_content_value_enabled' ) ) {
			$value_option = PYS()->getOption( 'woo_view_content_value_option' );
			$global_value = PYS()->getOption( 'woo_view_content_value_global', 0 );
			$percents_value = PYS()->getOption( 'woo_view_content_value_percent', 100 );

			if ( function_exists( 'PixelYourSite\getWooProductValue' ) ) { // new api, can remove old
				$params[ 'value' ] = getWooProductValue( [
					"valueOption" => $value_option,
					"global"      => $global_value,
					"percent"     => $percents_value,
					"product_id"  => $product_data[ 'id' ],
					"qty"         => $product_data[ 'quantity' ],
					"price"       => $product_data[ 'price' ],
				] );
			} else { // old
				$params[ 'value' ] = getWooEventValue( $value_option, $global_value, $percents_value, $product_data[ 'id' ], $product_data[ 'quantity' ] );
			}


		}

		$event->addParams( $params );
		$event->addPayload( [
			'name'  => 'viewcontent',
			'delay' => (int) PYS()->getOption( 'woo_view_content_delay' ),
		] );
		return true;
	}

    public function getWooPageVisitEventParams() {
		global $post;

		if ( !$this->getOption( 'woo_view_content_enabled' ) || !$this->getPageViewEventParams() ) {
			return false;
		}

        $product = wc_get_product( $post->ID );
        if(!$product) {
            return false;
        }

        $params = $this->getWooViewContentParams($product);

		return array(
			'name'  => 'viewcontent',
			'data'  => $params,
			'delay' => (int) PYS()->getOption( 'woo_view_content_delay' ),
		);

	}

    private function getWooViewContentParams($product, $quantity = 1)
    {
        if(Pinterest\isPysProActive() && $this->getOption('woo_variable_data_select_product') && !$this->getOption( 'woo_variable_as_simple')){
            $variable_id = getVariableIdByAttributes($product);
            if($variable_id){
                $params['content_type'] = 'product';
            }
        }
        $line_item = array(
            'product_id'    => getWooProductContentId( $variable_id ?? $product->get_id() ),
            'product_name'  => $product->get_name(),
            'product_price' => getWooProductPriceToDisplay( $variable_id ?? $product->get_id() ),
            'product_category' => implode( ', ', getObjectTerms( 'product_cat', $product->get_id() ) ),
            'product_quantity' => $quantity, // default quantity is 1
        );

        if (function_exists('getBrandForWooItem')) {
            $brand = getBrandForWooItem($product->get_id());
            if ($brand) {
                $line_item['product_brand'] = $brand;
            }
        }

        // content_name, category_name, tags
        $line_item[ 'tags' ] = implode( ', ', getObjectTerms( 'product_tag', $variable_id ?? $product->get_id() ) );

        $params['line_items'][] = $line_item;
        // currency, value
        if ( PYS()->getOption( 'woo_view_content_value_enabled' ) ) {

            $value_option = PYS()->getOption( 'woo_view_content_value_option' );
            $global_value = PYS()->getOption( 'woo_view_content_value_global', 0 );
            $percents_value = PYS()->getOption( 'woo_view_content_value_percent', 100 );

            $params[ 'value' ] = getWooEventValue( $value_option, $global_value, $percents_value, $variable_id ?? $product->get_id(), 1 );
            $params[ 'currency' ] = get_woocommerce_currency();

        }

        return $params;
    }

	/**
	 * @param SingleEvent $event
	 */
    public function prepareWcfAddToCart( &$event ) {
		if ( !$this->getOption( 'woo_add_to_cart_enabled' ) || empty( $event->args[ 'products' ] ) ) {
			return false; // return if args is empty
		}

		$params = array(
			'post_type' => 'product',
		);
		$value = 0;
		$line_items = [];
		// set option names
		$value_enabled_option = 'woo_add_to_cart_value_enabled';
		$value_option_option = 'woo_add_to_cart_value_option';
		$value_global_option = 'woo_add_to_cart_value_global';
		$value_percent_option = 'woo_add_to_cart_value_percent';

		foreach ( $event->args[ 'products' ] as $product_data ) {
			$content_id = getWooProductContentId( $product_data[ 'id' ] );

			$line_item = array(
				'product_id'       => $content_id,
				'product_quantity' => $product_data[ 'quantity' ],
				'product_price'    => getWooProductPriceToDisplay( $product_data[ 'id' ], 1, $product_data[ 'price' ] ),
				'product_name'     => $product_data[ 'name' ],
				'product_category' => implode( ', ', array_column( $product_data[ 'categories' ], "name" ) ),
				'tags'             => implode( ', ', $product_data[ 'tags' ] )
			);

			if (function_exists('getBrandForWooItem')) {
				$brand = getBrandForWooItem($product_data[ 'id' ]);
				if ($brand) {
					$line_item['product_brand'] = $brand;
				}
			}


			// currency, value
			if ( PYS()->getOption( $value_enabled_option ) ) {

				$value_option = PYS()->getOption( $value_option_option );
				$global_value = PYS()->getOption( $value_global_option, 0 );
				$percents_value = PYS()->getOption( $value_percent_option, 100 );


				if ( function_exists( 'PixelYourSite\getWooProductValue' ) ) { // new api, can remove old
					$value += getWooProductValue( [
						"valueOption" => $value_option,
						"global"      => $global_value,
						"percent"     => $percents_value,
						"product_id"  => $product_data[ 'id' ],
						"qty"         => $product_data[ 'quantity' ],
						"price"       => $product_data[ 'price' ],
					] );

				} else { // old use in free

					$value += getWooEventValue( $value_option, $global_value, $percents_value, $product_data[ 'id' ], $product_data[ 'quantity' ] );
					$line_item[ 'val2' ] = $value;
				}


			}

			$line_items[] = $line_item;
		}

		if ( count( $line_items ) == 1 ) {
			$params = array_merge( $params, $line_items[ 0 ] );
		} else {
			$params[ 'line_items' ] = $line_items;
		}
		if ( PYS()->getOption( $value_enabled_option ) ) {
			$params[ "value" ] = $value;
			$params[ 'currency' ] = get_woocommerce_currency();
		}


		$event->addParams( $params );
		$event->addPayload( [
			'name' => 'addtocart',
		] );

		return true;
	}

    public function getWooAddToCartOnButtonClickEventParams( $product_id, $quantity ) {

		$params = Pinterest\getWooSingleAddToCartParams( $product_id, $quantity, false );

		$data = array(
			'params' => $params,
		);

		$product = wc_get_product( $product_id );
		if ( $product->get_type() == 'grouped' ) {
			$grouped = array();
			foreach ( $product->get_children() as $childId ) {
				$grouped[ $childId ] = array(
					'content_id' => getWooProductContentId( $childId ),
					'price'      => getWooProductPriceToDisplay( $childId )
				);
			}
			$data[ 'grouped' ] = $grouped;
		}

		return $data;

	}

	/**
	 * @return array|false
	 * @deprecated
	 */
    public function getWooAddToCartOnCartEventParams() {

		if ( !$this->getOption( 'woo_add_to_cart_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooCartParamsOld();

		return array(
			'name' => 'addtocart',
			'data' => $params,
		);

	}

	/**
	 * @param SingleEvent $event
	 * @return boolean
	 */
    public function setWooAddToCartOnCartEventParams( &$event ) {

		if ( !$this->getOption( 'woo_add_to_cart_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooCartParams( $event );

		$event->addParams( $params );
		$event->addPayload( [ 'name' => 'addtocart' ] );

		return true;

	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function prepareWcfRemoveFromCart( &$event ) {
		if ( !$this->getOption( 'woo_remove_from_cart_enabled' ) || empty( $event->args[ 'products' ] ) ) {
			return false;
		}

		$product_data = $event->args[ 'products' ][ 0 ];
		$product_id = $product_data[ 'id' ];
		$content_id = getWooProductContentId( $product_id );
		$price = getWooProductPriceToDisplay( $product_data[ 'id' ], 1, $product_data[ 'price' ] );

		$params = array(
			'post_type'        => 'product',
			'product_id'       => $content_id,
			'product_quantity' => $product_data[ 'quantity' ],
			'product_price'    => $price,
			'product_name'     => $product_data[ 'name' ],
			'product_category' => implode( ', ', array_column( $product_data[ 'categories' ], "name" ) ),
			'tags'             => implode( ', ', $product_data[ 'tags' ] )
		);

		$event->addParams( $params );

		$event->addPayload( [
			'name' => "RemoveFromCart",
		] );

		return true;
	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function setWooRemoveFromCartParams( &$event ) {
		if ( !$this->getOption( 'woo_remove_from_cart_enabled' ) ) {
			return false;
		}

		$data = $this->getWooRemoveFromCartParams( $event->args[ 'item' ] );

		$event->addParams( $data[ 'data' ] );
		$event->addPayload( [ 'name' => $data[ 'name' ] ] );
		return true;
	}

    public function getWooRemoveFromCartParams( $cart_item ) {

		if ( !$this->getOption( 'woo_remove_from_cart_enabled' ) ) {
			return false;
		}
		$product_id = Pinterest\getWooCartItemId( $cart_item );
		$content_id = getWooProductContentId( $product_id );


		$price = getWooProductPriceToDisplay( $product_id );

		// content_name, category_name, tags

		$params = array(
			'post_type'        => 'product',
			'product_id'       => $content_id,
			'product_quantity' => $cart_item[ 'quantity' ],
			'product_price'    => $price,
			'product_name'     => '',
			'product_category' => '',
			'tags'             => implode( ', ', getObjectTerms( 'product_tag', $product_id ) )
		);

		return array(
			'name' => "RemoveFromCart",
			'data' => $params
		);

	}

    public function getWooViewCategoryEventParams(&$event) {
		global $posts;

		if ( !$this->getOption( 'woo_view_category_enabled' ) ) {
			return false;
		}

		$params = array(
			'post_type' => 'product',
		);

		$term = get_term_by( 'slug', get_query_var( 'term' ), 'product_cat' );

		if ( !$term ) {
            return false;
        }
        $line_items = [];

        if(isset($event->args['products'])) {
            foreach ($event->args['products'] as $product_data) {
                $product = wc_get_product($product_data['product_id']);
                if(!$product) {
                    continue;
                }
                $line_item = array(
                    'product_id' => getWooProductContentId($product->get_id()),
                    'product_quantity' => 1,
                    'product_price' => getWooProductPriceToDisplay($product->get_id()),
                    'product_name' => $product->get_name(),
                    'product_category' => implode(', ', getObjectTerms('product_cat', $product->get_id())),
                );

                if (function_exists('getBrandForWooItem')) {
                    $brand = getBrandForWooItem($product->get_id());
                    if ($brand) {
                        $line_item['product_brand'] = $brand;
                    }
                }

                $line_items[] = $line_item;
            }
        }

        $params[ 'line_items' ] = $line_items;

		return array(
			'name' => 'viewcategory',
			'data' => $params,
		);

	}


	/**
	 * @param SingleEvent $event
	 * @return boolean
	 */
    public function setWooInitiateCheckoutEventParams( &$event ) {

		if ( !$this->getOption( 'woo_initiate_checkout_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooCartParams( $event );

		$event->addParams( $params );
		$event->addPayload( [ 'name' => 'initiatecheckout', ] );

		return true;
	}

	/**
	 * @return array|false
	 * @deprecated
	 */
    public function getWooInitiateCheckoutEventParams() {

		if ( !$this->getOption( 'woo_initiate_checkout_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooCartParamsOld( 'InitiateCheckout' );

		return array(
			'name' => 'initiatecheckout',
			'data' => $params,
		);

	}

	/**
	 * @param SingleEvent $event
	 * @return false
     * @deprecated use addWooCheckoutEventParams
	 */
    public function getWooCheckoutEventParams(SingleEvent $event ): bool
    {

		if ( !$this->getOption( 'woo_purchase_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooPurchaseParams( 'Purchase' );


		$event->addParams( $params );
		$event->addPayload( [
			'name' => 'checkout',
		] );
		return true;

	}

	/**
	 * @param SingleEvent $event
	 * @return array|false
	 */
    public function addWooCheckoutEventParams( &$event ) {
		if ( !$this->getOption( 'woo_purchase_enabled' ) ) {
			return false;
		}

		$line_items = [];
		$order_quantity = 0;
		$value_option = PYS()->getOption( 'woo_purchase_value_option' );
		$global_value = PYS()->getOption( 'woo_purchase_value_global', 0 );
		$percents_value = PYS()->getOption( 'woo_purchase_value_percent', 100 );
		$withTax = 'incl' === get_option( 'woocommerce_tax_display_cart' );
		$tax = 0;
		foreach ( $event->args[ 'products' ] as $product_data ) {

			$product_id = Pinterest\getWooCartItemId( $product_data );
			$content_id = getWooProductContentId( $product_id );
			$price = $product_data[ 'subtotal' ];
			if ( $withTax ) {
				$price += $product_data[ 'subtotal_tax' ];
			}
			$line_item = [
				'product_id'       => $content_id,
				'product_quantity' => $product_data[ 'quantity' ],
				'product_price'    => $product_data[ 'quantity' ] > 0 ? pinterest_round( $price / $product_data[ 'quantity' ] ) : $price,
				'product_name'     => $product_data[ 'name' ],
				'product_category' => implode( ', ', array_column( $product_data[ 'categories' ], 'name' ) ),
				'tags'             => implode( ', ', $product_data[ 'tags' ] )
			];

			if (function_exists('getBrandForWooItem')) {
				$brand = getBrandForWooItem($product_id);
				if ($brand) {
					$line_item['product_brand'] = $brand;
				}
			}
			$line_items[] = $line_item;
			$order_quantity += $product_data[ 'quantity' ];
			$tax += $product_data[ 'total_tax' ];
		}

		$tax += $event->args[ 'shipping_tax' ];
		$shipping_cost = $event->args[ 'shipping_cost' ];
		if ( $withTax ) {
			$shipping_cost += $event->args[ 'shipping_tax' ];
		}
		$total = Pinterest\getWooEventOrderTotal( $event );
		if ( function_exists( 'PixelYourSite\getWooEventValueProducts' ) ) {
			$value = getWooEventValueProducts( $value_option, $global_value, $percents_value, $total, $event->args );
		} else {
			// remove after update free
			$value = getWooEventValueOrder( $value_option, wc_get_order( $event->args[ 'order_id' ] ), $global_value, $percents_value );
		}


		$params = array(
			'post_type'          => 'product',
			'line_items'         => $line_items,
			'order_quantity'     => $order_quantity,
            'order_id'           => (string) $event->args[ 'order_id' ],
			'currency'           => $event->args[ 'currency' ],
			'town'               => $event->args[ 'town' ],
			'state'              => $event->args[ 'state' ],
			'country'            => $event->args[ 'country' ],
			'payment'            => $event->args[ 'payment_method' ],
			'shipping'           => $event->args[ 'shipping' ],
			'value'              => pinterest_round( $value ),
			'promo_code_used'    => $event->args[ 'coupon_used' ],
			'promo_code'         => $event->args[ 'coupon_name' ],
			'total'              => pinterest_round( $total ),
			'tax'                => pinterest_round( $tax ),
			'shipping_cost'      => $shipping_cost,
			'lifetime_value'     => isset( $event->args[ 'predicted_ltv' ] ) ? $event->args[ 'predicted_ltv' ] : "",
			'average_order'      => isset( $event->args[ 'average_order' ] ) ? $event->args[ 'average_order' ] : "",
			'transactions_count' => isset( $event->args[ 'transactions_count' ] ) ? $event->args[ 'transactions_count' ] : "",
            'fees'               => isset( $event->args[ 'fees' ]) ? $event->args[ 'fees' ] : ""
		);
		$event->addParams( $params );
		$event->addPayload( [
			'name' => 'checkout',
		] );
		return true;
	}


    public function getWooAffiliateEventParams( $product_id, $quantity ) {

		if ( !$this->getOption( 'woo_affiliate_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooSingleAddToCartParams( $product_id, $quantity, true );

		return array(
			'params' => $params,
		);

	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function setWooPayPalEventParams( &$event ) {

		if ( !$this->getOption( 'woo_paypal_enabled' ) ) {
			return false;
		}

		// we're using Cart date as of Order not exists yet
		$params = Pinterest\getWooCartParams( $event );

		$event->addParams( $params );
		$event->addPayload( [ 'name' => getWooPayPalEventName() ] );

		return true;
	}

    public function getWooAdvancedMarketingEventParams( $eventType ) {

		if ( !$this->getOption( $eventType . '_enabled' ) ) {
			return false;
		}

		$params = Pinterest\getWooPurchaseParams( $eventType );

		switch ( $eventType ) {
			case 'woo_frequent_shopper':
				$eventName = 'FrequentShopper';
				break;

			case 'woo_vip_client':
				$eventName = 'VipClient';
				break;
            case 'woo_FirstTimeBuyer':
                $eventName = 'FirstTimeBuyer';
                break;
            case 'woo_ReturningCustomer':
                $eventName = 'ReturningCustomer';
                break;
			default:
				$eventName = 'BigWhale';
		}

		return array(
			'name' => $eventName,
			'data' => $params,
		);

	}

	/**
	 * @param CustomEvent $customEvent
	 *
	 * @return array|bool
	 */
    public function getCustomEventParams( $event ) {
        global $post;

        $customEvent = $event->args;
		$event_type = $customEvent->getPinterestEventType();

		if ( !$customEvent->isPinterestEnabled() || empty( $event_type ) ) {
			return false;
		}

		$params = array();

		// add pixel params
		if ( $customEvent->isPinterestParamsEnabled() ) {

			// add custom params
			$customParams = $customEvent->getPinterestCustomParams();

			foreach ( $customParams as $custom_param ) {
				$params[ $custom_param[ 'name' ] ] = $custom_param[ 'value' ];
			}

		}
        if(isWooCommerceActive()) {
            if ($customEvent->pinterest_track_single_woo_data && is_product() && ($post->post_type === 'product' || $event->getPayloadValue('productId'))) {
                $product_id = $event->getPayloadValue('productId') ?? $post->ID;
                $quantity = $event->getPayloadValue('quantity') ?? 1;

                $product = wc_get_product($product_id);

                if ($product) {
                    $eventData = $this->getWooViewContentParams($product, $quantity);
                    $params = array_merge($params, $eventData);
                }

            } elseif ($customEvent->pinterest_track_cart_woo_data) {
                $order = EventsWoo()->getOrder();
                if ($order) {
                    $params = array_merge($params, Pinterest\getWooPurchaseParams('Purchase'));
                } else {
                    $viewContentEvent = EventsWoo()->getEvent('woo_initiate_checkout');
                    if ($viewContentEvent) {
                        $params = array_merge($params, Pinterest\getWooCartParamsOld('InitiateCheckout'));
                    }
                }
            }
        }

		// SuperPack Dynamic Params feature
		$params = apply_filters( 'pys_superpack_dynamic_params', $params, 'pinterest' );

		return array(
			'name'  => $customEvent->getPinterestEventType(),
			'data'  => $params,
			'delay' => $customEvent->getDelay(),
		);

	}


    public function getEddPageVisitEventParams() {
		global $post;

		if ( !$this->getOption( 'edd_page_visit_enabled' ) || !$this->getPageViewEventParams() ) {
			return false;
		}

		$params = array(
			'post_type'  => 'product',
			'product_id' => Pinterest\getEddDownloadContentId( $post->ID )
		);

		// content_name, category_name, tags
		$params[ 'tags' ] = implode( ', ', getObjectTerms( 'download_tag', $post->ID ) );
		$params = array_merge( $params, Pinterest\getEddCustomAudiencesOptimizationParams( $post->ID ) );

		// currency, value
		if ( PYS()->getOption( 'edd_view_content_value_enabled' ) ) {

			if ( PYS()->getOption( 'edd_event_value' ) == 'custom' ) {
				$amount = getEddDownloadPrice( $post->ID );
			} else {
				$amount = getEddDownloadPriceToDisplay( $post->ID );
			}

			$value_option = PYS()->getOption( 'edd_view_content_value_option' );
			$global_value = PYS()->getOption( 'edd_view_content_value_global', 0 );
			$percents_value = PYS()->getOption( 'edd_view_content_value_percent', 100 );

			$params[ 'value' ] = getEddEventValue( $value_option, $amount, $global_value, $percents_value );
			$params[ 'currency' ] = edd_get_currency();

		}

		$params[ 'product_price' ] = getEddDownloadPriceToDisplay( $post->ID );

		return array(
			'name'  => 'viewcontent',
			'data'  => $params,
			'delay' => (int) PYS()->getOption( 'edd_view_content_delay' ),
		);

	}

    public function getEddAddToCartOnButtonClickEventParams( $download_id ) {
		global $post;

		// maybe extract download price id
		if ( strpos( $download_id, '_' ) !== false ) {
			list( $download_id, $price_index ) = explode( '_', $download_id );
		} else {
			$price_index = null;
		}

		// content_name, category_name, tags
		$cd_params = Pinterest\getEddCustomAudiencesOptimizationParams( $download_id );

		$params = array(
			'post_type'        => 'product',
			'product_id'       => Pinterest\getEddDownloadContentId( $download_id ),
			'product_quantity' => 1,
			'product_name'     => $cd_params[ 'content_name' ],
			'product_category' => $cd_params[ 'category_name' ],
			'product_price'    => getEddDownloadPriceToDisplay( $download_id, $price_index ),
			'tags'             => implode( ', ', getObjectTerms( 'download_tag', $download_id ) ),
		);

		// currency, value
		if ( PYS()->getOption( 'edd_add_to_cart_value_enabled' ) ) {

			if ( PYS()->getOption( 'edd_event_value' ) == 'custom' ) {
				$amount = getEddDownloadPrice( $download_id, $price_index );
			} else {
				$amount = getEddDownloadPriceToDisplay( $download_id, $price_index );
			}

			$value_option = PYS()->getOption( 'edd_add_to_cart_value_option' );
			$percents_value = PYS()->getOption( 'edd_add_to_cart_value_percent', 100 );
			$global_value = PYS()->getOption( 'edd_add_to_cart_value_global', 0 );

			$params[ 'value' ] = getEddEventValue( $value_option, $amount, $global_value, $percents_value );
			$params[ 'currency' ] = edd_get_currency();

		}

		$license = getEddDownloadLicenseData( $download_id );
		$params = array_merge( $params, $license );

		return $params;

	}

    public function getEddCartEventParams( $context, $value_enabled, $value_option, $global_value, $percents_value ) {
		$params = [];
		if ( $context === 'AddToCart' || $context === 'InitiateCheckout' ) {
			$cart = edd_get_cart_contents();
		} else {
			$cart = edd_get_payment_meta_cart_details( edd_get_purchase_id_by_key( getEddPaymentKey() ), true );
		}

		$total = (float) 0;
		$total_as_is = (float) 0;

		$licenses = array(
			'transaction_type'   => null,
			'license_site_limit' => null,
			'license_time_limit' => null,
			'license_version'    => null
		);

        $productData = $this->pinterestMainHelper->processGetEddCartEventProducts($cart, $total, $total_as_is, $licenses, $context, $value_enabled);

        $params = array_merge( $params, $licenses );
        $params['line_items'] = $productData['line_items'];
        $params['num_items'] = $productData['num_items'];

		// currency, value
		if ( $value_enabled ) {

			if ( PYS()->getOption( 'edd_event_value' ) === 'custom' ) {
				$amount = $total;
			} else {
				$amount = $total_as_is;
			}

			$params[ 'value' ] = getEddEventValue( $value_option, $amount, $global_value, $percents_value );
			$params[ 'currency' ] = edd_get_currency();

		}

		$params = array_merge( $params, $licenses );

		if ( $context === 'Checkout' ) {
            $this->pinterestMainHelper->processCheckoutParams( $params );
		}

		return $params;
	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function setEddCartEventParams( $event ): bool
    {
        $eventData = $this->pinterestMainHelper->getEddCartEventData($event->getId());

        if ( !$eventData ) {
            return false;
        }

        $data           = $eventData['data'];
        $value_enabled  = $eventData['value_enabled'];
        $value_option   = $eventData['value_option'];
        $percents_value = $eventData['percents_value'];
        $global_value   = $eventData['global_value'];

        $params = ['post_type' => 'product'];

		if ( $event->args == null ) { // remove when update free
			$params = array_merge( $params, $this->getEddCartEventParams( $data[ 'name' ], $value_enabled, $value_option, $global_value, $percents_value ) );
		} else {

			$total = 0;
			$total_as_is = 0;
			$tax = 0;
            $licenses = array(
                'transaction_type'   => null,
                'license_site_limit' => null,
                'license_time_limit' => null,
                'license_version'    => null
            );

            $productData = $this->pinterestMainHelper->processSetEddCartEventProducts($event, $total, $total_as_is, $tax, $licenses);
            $params = array_merge( $params, $licenses );
            $params['line_items'] = $productData['line_items'];
            $params['order_quantity'] = $productData['num_items'];

			//add fee
			$this->pinterestMainHelper->applyFeeEddToTotals( $event, $total, $total_as_is, $tax);
			// currency, value
            if ( $value_enabled ) {
                $amount = ( PYS()->getOption('edd_event_value') === 'custom' ) ? $total : $total_as_is;
                $params['value'] = getEddEventValue($value_option, $amount, $global_value, $percents_value);
                $params['currency'] = edd_get_currency();
            }

            if ( $event->getId() === 'edd_purchase' ) {
                $params += $this->pinterestMainHelper->processPurchaseParams($event, $total, $total_as_is, $tax);
            }
		}
		$event->addParams( $params );
		$event->addPayload( $data );
		return true;
	}

	/**
	 * @param SingleEvent $event
	 * @return bool
	 */
    public function setEddRemoveFromCartParams( &$event ) {
		if ( !$this->getOption( 'edd_remove_from_cart_enabled' ) ) {
			return false;
		}
		$data = $this->getEddRemoveFromCartParams( $event->args[ 'item' ] );
		$event->addParams( $data[ 'data' ] );
		$event->addPayload( [ 'name' => $data[ 'name' ] ] );
		return true;
	}

    public function getEddRemoveFromCartParams( $cart_item ) {

		if ( !$this->getOption( 'edd_remove_from_cart_enabled' ) ) {
			return false;
		}

		$download_id = $cart_item[ 'id' ];
		$price_index = !empty( $cart_item[ 'options' ] ) && !empty( $cart_item[ 'options' ][ 'price_id' ] ) ? $cart_item[ 'options' ][ 'price_id' ] : null;

		// content_name, category_name, tags
		$cd_params = Pinterest\getEddCustomAudiencesOptimizationParams( $download_id );

		$params = array(
			'post_type'        => 'product',
			'product_id'       => Pinterest\getEddDownloadContentId( $download_id ),
			'product_quantity' => $cart_item[ 'quantity' ],
			'product_price'    => getEddDownloadPriceToDisplay( $download_id, $price_index ),
			'product_name'     => $cd_params[ 'content_name' ],
			'product_category' => $cd_params[ 'category_name' ],
			'tags'             => implode( ', ', getObjectTerms( 'download_tag', $download_id ) )
		);

		return array(
			'name' => 'RemoveFromCart',
			'data' => $params
		);

	}

    public function getEddViewCategoryEventParams(&$event) {
		global $posts;
		if ( !$this->getOption( 'edd_view_category_enabled' ) ) {
			return false;
		}

		$params = array(
			'post_type' => 'product',
		);

        $term = get_term_by( 'slug', get_query_var( 'term' ), 'download_category' );
        if(!$term) return false;
        $parent_ids = get_ancestors( $term->term_id, 'download_category', 'taxonomy' );

        $download_categories = array();
        $download_categories[] = $term->name;

        foreach ( $parent_ids as $term_id ) {
            $parent_term = get_term_by( 'id', $term_id, 'download_category' );
            $download_categories[] = $parent_term->name;
        }
        $params[ 'content_name' ] = $term->name;
        $params[ 'content_category' ] = implode( ', ', $download_categories );

        $total_value = 0;

        $line_items = [];

        for ( $i = 0; $i < count( $posts ) && $i < 10; $i ++ ) {

            $line_item = array(
                'product_id' => getEddDownloadContentId($posts[ $i ]->ID),
                'product_quantity' => 1,
                'product_price' => getEddDownloadPriceToDisplay( $posts[ $i ]->ID ),
                'product_name' => $posts[ $i ]->post_title,
                'product_category' => implode( '/', getObjectTerms( 'download_category', $posts[ $i ]->ID ) ),
            );

            if (function_exists('getBrandForWooItem')) {
                $brand = getBrandForWooItem($posts[ $i ]->ID);
                if ($brand) {
                    $line_item['product_brand'] = $brand;
                }
            }

            $line_items[] = $line_item;
            $total_value += $line_item['product_price'];

        }

        $params = array(
            'currency' => edd_get_currency(),
            'value'    => $total_value,
            'line_items'           => $line_items,
        );

		return array(
			'name' => 'viewcategory',
			'data' => $params,
		);

	}

    public function output_meta_tag() {
		$metaTags = (array) $this->getOption( 'verify_meta_tag' );
		foreach ( $metaTags as $tag ) {
			echo $tag;
		}
	}

	/**
	 * @return bool
	 */
	public function isServerApiEnabled() {
		return $this->getOption( "use_server_api" );
	}

	public function getApiTokens() {

		$tokens = array();
		$pixel_ids = (array) $this->getOption( 'pixel_id' );
		if ( !empty($pixel_ids) ) {
			$server_ids = (array) $this->getOption( 'server_access_api_token' );
			$account_id = (array) $this->getOption( 'ad_account_id' );
			$tokens[ $pixel_ids[ 0 ] ] = array(
				'server_id'  => reset( $server_ids ),
				'ad_account' => reset( $account_id ),
			);
		}

		return $tokens;
	}

	public function getPageViewEventParams( $settings = false ) {

		$cpt = get_post_type();
		$params = array();
		$items = array();
        $shouldRun  = true;
        $result     = null;
		if ($this->pinterestMainHelper->shouldProcessWooCommercePageView( $cpt )) {

            global $product, $post;

            $id = $product instanceof \WC_Product ? $product->get_id() : ($post instanceof \WP_Post ? $post->ID : '');

            $items[] = array(
                "id"           => $id,
                "content_type" => 'product'
            );
		} elseif ( $this->pinterestMainHelper->shouldProcessEddPageView( $cpt ) ) {
            global $post;
            $items[] = array(
                "id"           => $post->ID,
                "content_type" => 'product'
            );
		} elseif ( $this->pinterestMainHelper->shouldProcessPageView($cpt)) {
            $shouldRun = $this->pinterestMainHelper->shouldProcessCustomPostType( $cpt );
		} else {
            $shouldRun = false;
		}

        if(!$shouldRun){
            return false;
        }

        if ( PYS()->getEventsManager()->doingAMP ) {
            $result = $settings
                ? [ 'name' => 'pagevisit', 'data' => [] ]
                : true;
        } elseif ( $settings ) {
            $params['items'] = $items;
            $result = [ 'name' => 'pagevisit', 'data' => $params ];
        } else {
            $result = true;
        }

        return $result;
	}

	public function getLog() {
		return $this->logger;
	}
    public function sanitize_verify_meta_tag_field($values) {
        $values = is_array( $values ) ? $values : array();
        $sanitized = array();
        $allowed_html = array(
            'meta' => array(
                'name' => array(),
                'content' => array(),
            ),
        );
        foreach ( $values as $key => $value ) {

            $value = wp_kses($value, $allowed_html);
            $new_value = $this->sanitize_textarea_field( $value );

            if ( ! empty( $new_value ) && ! in_array( $new_value, $sanitized ) ) {
                $sanitized[ $key ] = $new_value;
            }

        }

        return $sanitized;
    }
}

/**
 * @return Pinterest
 */
function Pinterest() {
	return Pinterest::instance();
}

Pinterest();
