<?php
/**
 * Allows plugins to use their own update API.
 *
 * @author oscargare
 * @version 2.4.12
 * @package WCPBC/Updater
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCPBC_Plugin_Updater Class
 */
class WCPBC_Plugin_Updater {

	/**
	 * Singleton.
	 *
	 * @var WCPBC_Plugin_Updater
	 */
	private static $instance = null;

	/**
	 * Plugin name.
	 *
	 * @var string
	 */
	protected $name;

	/**
	 * Plugin slug.
	 *
	 * @var string
	 */
	protected $slug;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	protected $version;

	/**
	 * API data.
	 *
	 * @var array
	 */
	protected $api_data;

	/**
	 * Cache key.
	 *
	 * @var string
	 */
	protected $cache_key;

	/**
	 * Cache hash.
	 *
	 * @var string
	 */
	protected $cache_hash;


	/**
	 * Class constructor.
	 *
	 * @param string $_plugin_file Path to the plugin file.
	 * @param string $_version     Plugin version.
	 * @param array  $_api_data    Optional data to send with API calls.
	 */
	protected function __construct( $_plugin_file, $_version, $_api_data = array() ) {

		$this->name     = plugin_basename( $_plugin_file );
		$this->slug     = basename( $_plugin_file, '.php' );
		$this->version  = $_version;
		$this->api_data = wp_parse_args(
			$_api_data,
			array(
				'item_id'     => 0,
				'license_key' => '',
				'api_key'     => '',
			)
		);

		$this->cache_key  = 'plugin_updater_cache';
		$this->cache_hash = md5( wp_json_encode( $this->api_data ) . $this->version . wcpbc()->version );

		// Set up hooks.
		$this->init_hooks();
	}

	/**
	 * Set up WordPress filters to hook into WP's update process.
	 *
	 * @return void
	 */
	protected function init_hooks() {
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'transient_update_plugins' ), 25 );
		add_filter( 'plugins_api', array( $this, 'plugin_information' ), 10, 3 );
	}

	/**
	 * Check for Updates at the defined API endpoint and modify the update array.
	 *
	 * This function dives into the update API just when WordPress creates its update array,
	 * then adds a custom API call and injects the custom plugin data retrieved from the API.
	 * It is reassembled from parts of the native WordPress plugin update code.
	 * See wp-includes/update.php line 121 for the original wp_update_plugins() function.
	 *
	 * @uses WC_Plugin_API_Wrapper::update_check()
	 *
	 * @param array $transient Update array build by WordPress.
	 * @return array Modified update array with custom plugin data.
	 */
	public function transient_update_plugins( $transient ) {

		if ( ! is_object( $transient ) ) {
			$transient = new stdClass();
		}

		$data = $this->get_update_data();

		if ( empty( $data ) ) {
			return $transient;
		}

		$item = [
			'id'          => 'pricebasedcountry-com-' . $this->api_data['item_id'],
			'slug'        => $this->slug,
			'plugin'      => $this->name,
			'new_version' => $data->version,
			'tested'      => $data->tested,
			'package'     => $data->package,
		];

		if ( version_compare( $this->version, $data->version, '<' ) ) {
			$item['package']                    = add_query_arg( 'token', $this->api_data['api_key'], $item['package'] );
			$transient->response[ $this->name ] = (object) $item;
			unset( $transient->no_update[ $this->name ] );

		} else {
			$transient->no_update[ $this->name ] = (object) $item;
			unset( $transient->response[ $this->name ] );
		}

		return $transient;
	}

	/**
	 * Get update data the extensions.
	 *
	 * @return array Update data
	 */
	private function get_update_data() {
		$data = $this->get_cached_version_info();

		if ( empty( $data ) ) {

			$data = WC_Plugin_API_Wrapper::update_check(
				$this->api_data['item_id'],
				$this->api_data['api_key'],
				$this->api_data['license_key'],
				$this->version
			);

			$this->set_version_info_cache( $data );
		}

		if ( is_wp_error( $data ) ) {

			$data = false;
		}

		return $data;
	}

	/**
	 * Return the cached version info.
	 *
	 * @return mixed Version info object or false is expired.
	 */
	protected function get_cached_version_info() {

		$cache = WCPBC_Helper_Options::get( $this->cache_key );

		if ( ! isset( $cache['timeout'], $cache['hash'], $cache['value'] ) || time() > $cache['timeout'] || $cache['hash'] !== $this->cache_hash ) {
			// Cache is expired.
			return false;
		}

		return $cache['value'];
	}

	/**
	 * Set cached version info.
	 *
	 * @param mixed $value Value to be cached.
	 */
	protected function set_version_info_cache( $value = '' ) {
		$data = array(
			'timeout' => strtotime( '+10 hours', time() ),
			'hash'    => $this->cache_hash,
			'value'   => $value,
		);
		WCPBC_Helper_Options::update( $this->cache_key, $data );
	}

	/**
	 * Updates information on the "View version x.x details" page with custom data.
	 *
	 * @uses api_request()
	 *
	 * @param mixed  $result The result object or array. Default false.
	 * @param string $action The type of information being requested from the Plugin Installation API.
	 * @param object $args Plugin API arguments.
	 * @return object
	 */
	public function plugin_information( $result, $action, $args ) {
		if ( 'plugin_information' !== $action ) {
			return $result;
		}

		if ( ! isset( $args->slug ) || $args->slug !== $this->slug ) {
			return $result;
		}

		$response = WC_Plugin_API_Wrapper::plugin_information(
			$this->api_data['item_id'],
			$this->api_data['api_key'],
			$this->api_data['license_key']
		);

		$response->sections = (array) $response->sections;

		return $response;
	}

	/**
	 * Init the updater.
	 */
	public static function init() {
		if ( ! is_null( self::$instance ) ) {
			return;
		}

		$options = WCPBC_License_Settings::instance();

		if ( $options->is_valid_key() ) {

			self::$instance = new WCPBC_Plugin_Updater(
				WCPBC_PRO_PLUGIN_FILE,
				WC_Product_Price_Based_Country_Pro::VERSION,
				array(
					'item_id'     => 1450,
					'license_key' => $options->get_license_key(),
					'api_key'     => $options->get_api_key(),
				)
			);
		}
	}
}
