<?php
/**
 * Installation related functions and actions.
 *
 * @package WCPBC_PRO
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_Install Class.
 */
class WCPBC_PRO_Install {

	/**
	 * Updates and callbacks that need to be run per version.
	 *
	 * @var array
	 */
	private static $updates = [ '2.1.9', '3.5.0', '3.6.3' ];

	/**
	 * Init hooks
	 */
	public static function init() {
		add_action( 'admin_init', [ __CLASS__, 'run' ] );
	}

	/**
	 * Check version and run updaters is required.
	 */
	public static function run() {
		if ( defined( 'IFRAME_REQUEST' ) ) {
			return;
		}

		self::check_version();
		self::do_license_settings_redirect();

		if ( ! WCPBC_License_Settings::instance()->is_license_active() ) {
			add_action( 'woocommerce_settings_price-based-country', [ __CLASS__, 'welcome_notice' ], 0 );
			add_action( 'after_plugin_row_' . plugin_basename( WCPBC_PRO_PLUGIN_FILE ), [ __CLASS__, 'plugin_update_row' ], 20 );
			add_filter( 'wp_get_update_data', [ __CLASS__, 'update_data' ] );

		}
	}

	/**
	 * Check version and run updaters is required.
	 */
	private static function check_version() {

		$db_version = get_option( 'wc_price_based_country_pro_version', false );

		if ( WC_Product_Price_Based_Country_Pro::VERSION === $db_version ) {
			return;
		}

		update_option( 'wc_price_based_country_pro_version', WC_Product_Price_Based_Country_Pro::VERSION );

		if ( ! $db_version ) {

			self::install();
			self::maybe_set_activation_redirect();

		} else {

			foreach ( self::$updates as $version ) {
				$callback = 'update_' . str_replace( '.', '', $version );
				if ( version_compare( $db_version, $version, '<' ) && is_callable( [ __CLASS__, $callback ] ) ) {
					self::{$callback}();
				}
			}
		}
	}

	/**
	 * Plugin install
	 */
	private static function install() {
		if ( ! get_option( 'wc_price_based_currency_format', false ) ) {
			$currency_pos = get_option( 'woocommerce_currency_pos' );
			switch ( $currency_pos ) {
				case 'left':
					$format = '[symbol-alt][price]';
					break;
				case 'right':
					$format = '[price][symbol-alt]';
					break;
				case 'left_space':
					$format = '[symbol-alt]&nbsp;[price]';
					break;
				case 'right_space':
					$format = '[price]&nbsp;[symbol-alt]';
					break;
				default:
					$format = '[price][symbol-alt]';
			}

			update_option( 'wc_price_based_currency_format', $format );
		}

		if ( ! get_option( 'wc_price_based_country_exchange_rate_api', false ) ) {
			update_option( 'wc_price_based_country_exchange_rate_api', 'floatrates' );
		}
	}

	/**
	 * Handle redirects to setup license after install. It must ignore the network/bulk plugin updaters.
	 */
	public static function do_license_settings_redirect() {
		global $current_tab, $current_section;

		if ( ! isset( $_COOKIE['_wcpbc_pro_activation_redirect'] ) || wp_doing_ajax() ) {
			return;
		}

		setcookie( '_wcpbc_pro_activation_redirect', '0', time() - 3600 );

		// On these pages, disable the redirect.
		if ( ( 'price-based-country' === $current_tab && 'license' === $current_section ) ||
			is_network_admin() ||
			isset( $_GET['activate-multi'] ) || // phpcs:ignore WordPress.Security.NonceVerification
			! current_user_can( 'manage_woocommerce' )
			|| WCPBC_License_Settings::instance()->is_license_active()
		) {
			return;
		}

		wp_safe_redirect( admin_url( 'admin.php?page=wc-settings&tab=price-based-country&section=license&wcpbcpro_welcome=1' ) );
		exit;
	}

	/**
	 * Update to 2.1.9
	 */
	private static function update_219() {
		wp_clear_scheduled_hook( 'wc_price_based_country_pro_cron' );
	}

	/**
	 * Update to 3.5.0
	 */
	private static function update_350() {
		wp_clear_scheduled_hook( 'wc_price_based_country_weekly_scheduled_events' );
		WCPBC_License_Settings::instance()->upgrade_350();
	}

	/**
	 * Update to 3.6.3
	 */
	private static function update_363() {
		delete_transient( 'wcs_report_subscription_events_by_date' );
		delete_transient( 'wcs_report_upcoming_recurring_revenue' );
	}

	/**
	 * Display welcome notice.
	 */
	public static function welcome_notice() {
		global $current_tab, $current_section;

		if ( ! isset( $_GET['wcpbcpro_welcome'] ) || 'price-based-country' !== $current_tab || 'license' !== $current_section ) { // phpcs:ignore WordPress.Security.NonceVerification
			return;
		}
		printf(
			'<div id="wcpbcpro_welcome_notice" style="padding:1em;" class="notice inline is-dismissible notice-success"><h3 class="notice-title">%s</h3><p>%s</p></div>',
			esc_html__( 'Welcome to Price Based on Country Pro', 'wc-price-based-country-pro' ),
			// Translators: HTML tags.
			sprintf( esc_html__( 'Thank you for upgrading to Price Based on Country Pro! To get started, enter your %1$sLicense Key%2$s below and click %1$sActivate%2$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>' )
		);
	}

	/**
	 * Display a warning message after plugin row if the license is not set.
	 */
	public static function plugin_update_row() {
		$wp_list_table = _get_list_table( 'WP_Plugins_List_Table' );
		$slug          = basename( WCPBC_PRO_PLUGIN_FILE, '.php' );
		// translators: HTML Tags.
		$message = sprintf( __( '%1$sWarning!%2$s You didn\'t set your license key, which means you\'re missing out on updates and support! Enter your %3$slicense key%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=price-based-country&section=license' ) ) . '">', '</a>' );
		?>
		<tr class="plugin-update-tr active" id="<?php echo esc_attr( $slug . '-nolicense' ); ?> "><td colspan="<?php echo esc_attr( $wp_list_table->get_column_count() ); ?>" class="plugin-update colspanchange"><div class="update-message notice inline notice-error notice-alt">
			<p>
				<?php echo wp_kses_post( $message ); ?>
			</p>
		</div></td></tr>
		<script>
		jQuery(document).ready(function($){
			$('tr[data-plugin="woocommerce-price-based-country-pro-addon/woocommerce-price-based-country-pro-addon.php"]').addClass('update');
		});
		</script>
		<?php
	}

	/**
	 * Increments the plugins count to display an alert on the plugin menu item.
	 *
	 * @param array $update_data Update data array.
	 * @return array
	 */
	public static function update_data( $update_data ) {
		if ( current_user_can( 'update_plugins' ) ) {
			$update_data['counts']['plugins']++;
		}
		return $update_data;
	}
}
