<?php
/**
 * WooCommerce Price Based on Country Pro Add-on setup
 *
 * @package WCPBC
 * @version 2.6.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_Product_Price_Based_Country_Pro Class
 */
class WC_Product_Price_Based_Country_Pro {

	const VERSION = '3.7.1';

	/**
	 * Legacy version.
	 *
	 * @var string
	 */
	public static $version = self::VERSION;

	/**
	 * Admin notices.
	 *
	 * @var array
	 */
	private static $admin_notices;

	/**
	 * Min WC required version.
	 *
	 * @var string
	 */
	private static $min_wc_version = '4.0';

	/**
	 * Min PBC required version.
	 *
	 * @var string
	 */
	private static $min_pbc_version = '3.4.13';

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public static function plugin_url() {
		return plugin_dir_url( WCPBC_PRO_PLUGIN_FILE );
	}

	/**
	 * Get the plugin path.
	 *
	 * @return string
	 */
	public static function plugin_path() {
		return plugin_dir_path( WCPBC_PRO_PLUGIN_FILE );
	}

	/**
	 * Init plugin, Hook actions and filters
	 *
	 * @since 1.0
	 */
	public static function init() {

		self::$admin_notices = array();

		spl_autoload_register( [ __CLASS__, 'autoload' ] );

		add_action( 'plugins_loaded', array( __CLASS__, 'init_plugin' ), 20 );
		add_action( 'init', array( __CLASS__, 'load_textdomain' ) );
		add_action( 'admin_notices', array( __CLASS__, 'display_notices' ) );

		// Functions.
		include_once dirname( __FILE__ ) . '/wcpbc-pro-functions.php';
	}

	/**
	 * Auto-load plugin classes on demand to reduce memory consumption.
	 *
	 * @param string $classname Class name.
	 */
	public static function autoload( $classname ) {

		$classmap = [
			'WCPBC_Exchange_Rates_Provider'      => '/abstracts/class-wcpbc-exchange-rates-provider.php',
			'WCPBC_Backward_Compatibility'       => '/class-wcpbc-backward-compatibility.php',
			'WCPBC_Recurring_Actions'            => '/class-wcpbc-recurring-actions.php',
			'WCPBC_Update_Exchange_Rates'        => '/class-wcpbc-update-exchange-rates.php',
			'WCPBC_Frontend_Currency'            => '/class-wcpbc-frontend-currency.php',
			'WCPBC_Integrations_Pro'             => '/class-wcpbc-integrations-pro.php',
			'WCPBC_Shortcodes'                   => '/class-wcpbc-shortcodes.php',
			'WCPBC_Pricing_Zone_Pro'             => '/class-wcpbc-pricing-zone-pro.php',
			'WCPBC_Pro_Coupons'                  => '/class-wcpbc-pro-coupons.php',
			'WCPBC_Pro_Frontend_Pricing_Filters' => '/class-wcpbc-pro-frontend-pricing-filters.php',
			'WCPBC_Currency_Switcher_Block'      => '/block-types/class-wcpbc-currency-switcher-block.php',
			'WCPBC_Order_Base_Currency'          => '/class-wcpbc-order-base-currency.php',
			'WCPBC_Order_Base_Currency_Item_Tax' => '/class-wcpbc-order-base-currency-item-tax.php',
			'WCPBC_License_Settings'             => '/admin/class-wcpbc-license-settings.php',
			'WC_Plugin_API_Wrapper'              => '/admin/class-wc-plugin-api-wrapper.php',
			'WCPBC_Plugin_Updater'               => '/admin/class-wcpbc-plugin-updater.php',
			'WCPBC_PRO_Install'                  => '/admin/class-wcpbc-pro-install.php',
			'WCPBC_Admin_Pro'                    => '/admin/class-wcpbc-admin-pro.php',
			'WCPBC_Admin_Product_CSV'            => '/admin/class-wcpbc-admin-product-csv.php',
		];

		$fullpath = isset( $classmap[ $classname ] ) ? dirname( __FILE__ ) . $classmap[ $classname ] : false;
		if ( $fullpath && is_readable( $fullpath ) ) {
			include_once $fullpath;
		}
	}

	/**
	 * Load text domain
	 *
	 * @since 2.2.7
	 */
	public static function load_textdomain() {
		load_plugin_textdomain( 'wc-price-based-country-pro', false, dirname( plugin_basename( WCPBC_PRO_PLUGIN_FILE ) ) . '/languages/' );
	}

	/**
	 * Init plugin
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	public static function init_plugin() {

		if ( ! self::check_environment() ) {
			return;
		}

		// Init plugin.
		WCPBC_Plugin_Updater::init();
		WCPBC_Update_Exchange_Rates::init();
		WCPBC_Frontend_Currency::init();
		WCPBC_Integrations_Pro::init();
		WCPBC_Shortcodes::init();
		WCPBC_Recurring_Actions::init();
		WCPBC_Pro_Coupons::init();

		if ( class_exists( 'WCPBC_Install' ) ) {
			// Admin request.
			WCPBC_PRO_Install::init();
			WCPBC_Admin_Pro::init();
			WCPBC_Admin_Product_CSV::init();
		}

		add_action( 'widgets_init', array( __CLASS__, 'register_widgets' ), 20 );
		add_action( 'parse_request', array( __CLASS__, 'rest_api_init' ), 0 );
		add_action( 'wc_price_based_country_weekly_job', array( __CLASS__, 'weekly_jobs' ) );
	}

	/**
	 * Checks the environment for compatibility problems.
	 *
	 * @return boolean
	 */
	private static function check_environment() {

		if ( ! class_exists( 'WC_Product_Price_Based_Country' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s needs the Basic version to work. Learn how to install Price Based on Country in the %3$sGetting Started Guide%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="https://www.pricebasedcountry.com/docs/getting-started/">', '</a>' );
			return false;
		}

		if ( ! defined( 'WC_VERSION' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s requires WooCommerce to be activated to work. Learn how to install Price Based on Country in the %3$sGetting Started Guide%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="https://www.pricebasedcountry.com/docs/getting-started/">', '</a>' );
			return false;
		}

		if ( version_compare( WC_VERSION, self::$min_wc_version, '<' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( 'Price Based on Country Pro - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', 'wc-price-based-country-pro' ), self::$min_wc_version, WC_VERSION );
			return false;
		}

		if ( version_compare( WCPBC()->version, self::$min_pbc_version, '<' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s - The minimum Price Based on Country (Free) version required for this plugin is %3$s. You are running %4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', self::$min_pbc_version, WCPBC()->version );
			return false;
		}

		return true;
	}

	/**
	 * Display admin notices
	 */
	public static function display_notices() {
		foreach ( self::$admin_notices as $notice ) {
			echo '<div id="message" class="error"><p>' . wp_kses_post( $notice ) . '</p></div>';
		}
	}

	/**
	 * Register Widgets
	 */
	public static function register_widgets() {
		if ( class_exists( 'WC_Widget' ) ) {
			include_once self::plugin_path() . 'includes/class-wcpbc-widget-currency-switcher.php';
			register_widget( 'WCPBC_Widget_Currency_Switcher' );
		}
	}

	/**
	 * WooCommerce REST API init.
	 */
	public static function rest_api_init() {
		if ( ! ( function_exists( 'wc' ) && is_callable( array( wc(), 'is_rest_api_request' ) ) && wc()->is_rest_api_request() && ! empty( $GLOBALS['wp']->query_vars['rest_route'] ) ) ) {
			return;
		}
		$route = untrailingslashit( $GLOBALS['wp']->query_vars['rest_route'] );

		// Support for manual order via REST API.
		if ( preg_match( '/\/wc\/v\d\/(orders|subscriptions)$/', $route ) && 'post' === strtolower( $_SERVER['REQUEST_METHOD'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$body   = is_callable( array( 'WP_REST_Server', 'get_raw_data' ) ) ? WP_REST_Server::get_raw_data() : '';
			$params = json_decode( $body );
			if ( $params ) {
				$billing_country  = isset( $params->billing->country ) ? strtoupper( $params->billing->country ) : false;
				$shipping_country = isset( $params->shipping->country ) ? strtoupper( $params->shipping->country ) : false;
				$country          = empty( $shipping_country ) || 'billing' === get_option( 'wc_price_based_country_based_on', 'billing' ) ? $billing_country : $shipping_country;

				$zone = WCPBC_Pricing_Zones::get_zone_by_country( $country );

				if ( $zone ) {
					// Load the front-end pricing for the current zone.
					wcpbc()->current_zone = $zone;
					if ( ! did_action( 'wc_price_based_country_frontend_princing_init' ) ) {
						WCPBC_Frontend_Pricing::init();
					}
				}

				// Update order metadata after order creation.
				add_action( 'woocommerce_new_order', array( 'WCPBC_Frontend', 'update_order_meta' ) );
			}
		}
	}

	/**
	 * Weekly jobs.
	 */
	public static function weekly_jobs() {

		if ( ! class_exists( 'WC_Plugin_API_Wrapper' ) ) {
			include_once self::plugin_path() . 'includes/admin/class-wc-plugin-api-wrapper.php';
		}

		// Check the license status.
		$options = WCPBC_License_Settings::instance();
		if ( $options->is_valid_key() ) {
			$options->check_license_status();
		}
	}
}


