<?php
/**
 * Handle coupons.
 *
 * @since 3.7.0
 * @package WCPBC
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCPBC_Pro_Coupons class.
 */
class WCPBC_Pro_Coupons {

	/**
	 * Init.
	 */
	public static function init() {
		wcpbc_add_pricing_filter( 'woocommerce_coupon_loaded', [ __CLASS__, 'coupon_loaded' ], 20 );
		add_action( 'admin_init', [ __CLASS__, 'admin_init' ] );
	}

	/**
	 * Set coupon amount
	 *
	 * @param WC_Coupon $coupon Coupon instance.
	 */
	public static function coupon_loaded( $coupon ) {
		$coupon_id     = $coupon->get_id();
		$discount_type = $coupon->get_discount_type();
		$pro_version   = get_post_meta( $coupon_id, '_wcpbc_pro_version', true );

		if ( empty( $pro_version ) || false !== strpos( $discount_type, 'percent' ) ) {
			return;
		}

		$amount = wcpbc_the_zone()->get_post_price( $coupon_id, 'coupon_amount', 'coupon' );

		if ( version_compare( WC_VERSION, '3.0', '<' ) ) {
			$coupon->coupon_amount = $amount;
		} else {
			$coupon->set_amount( $amount );
		}
	}

	/**
	 * Admin init
	 */
	public static function admin_init() {
		remove_action( 'woocommerce_coupon_options', [ 'WCPBC_Admin_Meta_Boxes', 'coupon_options' ] );
		add_action( 'woocommerce_coupon_options', [ __CLASS__, 'coupon_options' ] );
		add_action( 'woocommerce_coupon_options_save', [ __CLASS__, 'coupon_options_save' ] );
	}

	/**
	 * Display coupon amount options.
	 *
	 * @param int $post_id Post ID.
	 * @since 2.4.7
	 */
	public static function coupon_options( $post_id ) {
		$coupon_version = get_post_meta( $post_id, '_wcpbc_pro_version', true );
		$pricing_type   = get_post_meta( $post_id, 'zone_pricing_type', true );

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			if ( empty( $coupon_version ) ) {
				if ( ! wcpbc_is_exchange_rate( $pricing_type ) ) {
					$zone->set_postmeta( $post_id, '_price_method', 'manual' );
					$zone->set_postmeta( $post_id, 'coupon_amount', get_post_meta( $post_id, 'coupon_amount', true ) );
				}
			}

			include WC_Product_Price_Based_Country_Pro::plugin_path() . 'includes/admin/views/html-coupon-pricing.php';
		}
	}

	/**
	 * Save coupon amount options.
	 *
	 * @since 2.4.7
	 * @param int $post_id Post ID.
	 */
	public static function coupon_options_save( $post_id ) {
		$discount_type = get_post_meta( $post_id, 'discount_type', true );
		if ( 'percent' !== $discount_type ) {
			foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
				$price_method = empty( $_POST[ $zone->get_postmetakey( '_price_method' ) ] ) ? '' : wc_clean( wp_unslash( $_POST[ $zone->get_postmetakey( '_price_method' ) ] ) ); // phpcs:ignore WordPress.Security.NonceVerification
				if ( wcpbc_is_exchange_rate( $price_method ) ) {
					$coupon_amount = empty( $_POST['coupon_amount'] ) ? : $zone->get_exchange_rate_price( wc_format_decimal( wc_clean( wp_unslash( $_POST['coupon_amount'] ) ) ), false ); // phpcs:ignore WordPress.Security.NonceVerification
				} else {
					$coupon_amount = empty( $_POST[ $zone->get_postmetakey( 'coupon_amount' ) ] ) ? '' : wc_format_decimal( wc_clean( wp_unslash( $_POST[ $zone->get_postmetakey( 'coupon_amount' ) ] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification
				}

				$zone->set_postmeta( $post_id, '_price_method', $price_method );
				$zone->set_postmeta( $post_id, 'coupon_amount', $coupon_amount );
			}
		}
		update_post_meta( $post_id, 'zone_pricing_type', 'nothing' );
		update_post_meta( $post_id, '_wcpbc_pro_version', WC_Product_Price_Based_Country_Pro::VERSION );
	}
}
