<?php
/**
 * WP All import addon.
 *
 * @since 3.5.0
 * @package WCPBC
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCPBC_WPAllimport Class
 */
final class WCPBC_WPAllImport {

	/**
	 * Parse data.
	 *
	 * @var array.
	 */
	protected $parse_data;

	/**
	 * Init integration.
	 */
	public static function init() {
		add_filter( 'pmxi_addons', [ __CLASS__, 'pmxi_addons' ] );
		add_action( 'pmxi_extend_options_main', [ __CLASS__, 'pmxi_extend_options' ], 15, 2 );
		add_action( 'pmxi_reimport', [ __CLASS__, 'pmxi_extend_options' ], 15, 2 );

	}

	/**
	 * Register the addon.
	 *
	 * @param array $addons Addons.
	 * @return array
	 */
	public static function pmxi_addons( $addons ) {
		$addons              = is_array( $addons ) ? $addons : [];
		$addons[ __CLASS__ ] = 1;

		return $addons;
	}

	/**
	 * Display the addon options.
	 *
	 * @param string $entry The current post type to import.
	 * @param Array  $post  The current import post.
	 */
	public static function pmxi_extend_options( $entry, $post ) {
		if ( 'product' !== $entry ) {
			return false;
		}

		$pieces = explode( '_', current_action() );
		$view   = end( $pieces );
		include dirname( __FILE__ ) . "/views/options/{$view}.php";
	}

	/**
	 * Method returns default import options, main utility of the method is to avoid warnings when new
	 * option is introduced but already registered imports don't have it
	 *
	 * @return array
	 */
	public static function get_default_import_options() {
		static $defaults = false;
		if ( false !== $defaults ) {
			return $defaults;
		}

		$defaults = [];

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			foreach ( self::get_fields_config() as $name => $data ) {
				$defaults[ self::get_option_name( $name, $zone ) ] = isset( $data['default'] ) ? $data['default'] : '';
				if ( ! empty( $data['xpath'] ) ) {
					$defaults[ self::get_option_name( "{$name}_xpath", $zone ) ] = '';
				}
			}
		}

		// Add the import options.
		$defaults[ self::get_option_name( 'is_update_pricing_zones' ) ]    = '1';
		$defaults[ self::get_option_name( 'update_pricing_zones_logic' ) ] = 'full_update';
		$defaults[ self::get_option_name( 'pricing_zones_only' ) ]         = [];
		$defaults[ self::get_option_name( 'update_fields_logic' ) ]        = 'full_update';
		$defaults[ self::get_option_name( 'fields_only' ) ]                = [];

		return $defaults;
	}

	/**
	 * Return the option name.
	 *
	 * @param string             $name Field name.
	 * @param WCPBC_Pricing_Zone $zone Pricing zone.
	 */
	private static function get_option_name( $name, $zone = false ) {
		return __CLASS__ . '_' . ( false !== $zone ? 'pricing_' . $zone->get_id() : '' ) . $name;
	}

	/**
	 * Return post value.
	 *
	 * @param array              $post Post data.
	 * @param string             $name Field name.
	 * @param WCPBC_Pricing_Zone $zone Pricing zone.
	 */
	private static function get_post_value( $post, $name, $zone = false ) {
		$field_slug = self::get_option_name( $name, $zone );
		return isset( $post[ $field_slug ] ) ? $post[ $field_slug ] : '';
	}

	/**
	 * Returns the fields config.
	 *
	 * @return array
	 */
	private static function get_fields_config() {
		return [
			'_price_method'          => [
				'default' => 'exchange_rate',
				'xpath'   => true,
				'label'   => 'Price mode (exchange_rate or manual)',
			],
			'_sale_price_dates'      => [
				'default' => 'default',
				'xpath'   => true,
				'label'   => 'Sale price dates mode (default or manual)',
			],
			'_regular_price'         => [
				'label' => 'Regular price',
			],
			'_sale_price'            => [
				'label' => 'Sale price',
			],
			'_sale_price_dates_from' => [
				'label' => 'Sale price date from',
			],
			'_sale_price_dates_to'   => [
				'label' => 'Sale price date to',
			],
		];
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'shutdown', [ $this, 'variable_product_sync' ] );
	}

	/**
	 * Parse data.
	 *
	 * @param array $parsing_data Data to parse.
	 * @return array|bool
	 */
	public function parse( $parsing_data ) {

		$this->variable_sync = [];
		$this->parse_data    = [];

		$import  = $parsing_data['import'];
		$options = $import->options;

		if ( '1' !== self::get_post_value( $options, 'is_update_pricing_zones' ) ) {
			return false;
		}

		$zone_ids = false;
		if ( 'only' === self::get_post_value( $options, 'update_pricing_zones_logic' ) ) {
			$zone_ids = explode( ',', self::get_post_value( $options, 'pricing_zones_only' ) );
		}

		$cxpath    = $parsing_data['xpath_prefix'] . $import->xpath;
		$tmp_files = [];
		$fields    = wc_list_pluck( self::get_fields_config(), 'label' );

		if ( 'only' === self::get_post_value( $options, 'update_fields_logic' ) ) {
			$fields = array_intersect( $fields, explode( ',', self::get_post_value( $options, 'fields_only' ) ) );
		}

		foreach ( WCPBC_Pricing_Zones::get_zones( $zone_ids ) as $zone ) {

			$values = [];

			foreach ( array_keys( $fields ) as $field ) {

				$field_slug = self::get_option_name( $field, $zone );

				if ( 'xpath' === $options[ $field_slug ] ) {
					$field_slug = self::get_option_name( "{$field}_xpath", $zone );
				}

				if ( empty( $options[ $field_slug ] ) ) {
					$values[ $field ] = array_fill( 0, $parsing_data['count'], '' );
				} else {
					$values[ $field ] = XmlImportParser::factory(
						$parsing_data['xml'],
						$cxpath,
						$options[ $field_slug ],
						$file
					)->parse();
					$tmp_files[]      = $file;
				}
			}

			$this->parse_data[ $zone->get_id() ] = $values;
		}

		// Remove all temporary files created.
		foreach ( $tmp_files as $file ) {
			unlink( $file );
		}

		return $this->parse_data;
	}

	/**
	 * Import data.
	 *
	 * @param array $import_data Data to import.
	 * @return array|bool
	 */
	public function import( $import_data ) {

		if ( empty( $import_data['post_type'] ) || 'product' !== $import_data['post_type'] || empty( $this->parse_data ) ) {
			return false;
		}
		$logger  = isset( $import_data['logger'] ) && is_callable( $import_data['logger'] ) ? $import_data['logger'] : false;
		$product = wc_get_product( $import_data['pid'] );

		$logger && call_user_func( $logger, '<b>Price Based on Country ADD-ON</b>:' );

		if ( ! $product ) {
			$logger && call_user_func( $logger, '<b>SKIP</b>: ' . __( 'The product does not exist', 'wc-price-based-country-pro' ) );
			return false;
		}

		if ( in_array( $product->get_type(), WCPBC_Backward_Compatibility::wrapper_product_types(), true ) ) {
			// Translators: Product type.
			$logger && call_user_func( $logger, '<b>SKIP</b>: ' . sprintf( __( 'Product type is %s', 'wc-price-based-country-pro' ), $product->get_type() ) );
			return false;
		}

		$index = $import_data['i'];

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {

			if ( empty( $this->parse_data[ $zone->get_id() ] ) || ! is_array( $this->parse_data[ $zone->get_id() ] ) ) {
				continue;
			}

			// Translators: Pricing zone name.
			$logger && call_user_func( $logger, sprintf( __( 'Importing pricing zone `%s`', 'wc-price-based-country-pro' ), $zone->get_name() ) );

			$row = [];
			foreach ( $this->parse_data[ $zone->get_id() ] as $key => $values ) {
				$row[ $key ] = wc_clean( $values[ $index ] );
			}

			if ( empty( $row ) ) {
				continue;
			}

			if ( empty( $row['_price_method'] ) && ( ! empty( $row['_regular_price'] ) || ! empty( $row['_sale_price'] ) ) ) {
				$logger && call_user_func( $logger, '<b>WARNING</b>: ' . __( 'Price method is empty. Set it to `manual`', 'wc-price-based-country-pro' ) );
				$row['_price_method'] = 'manual';
			}

			if ( empty( $row['_price_method'] ) ) {
				$row['_price_method'] = $zone->get_postmeta( $product->get_id(), '_price_method' );
				// Translators: 1: Product name, 2: price method.
				$logger && call_user_func( $logger, sprintf( __( 'Preserve price method for `%1$s`: `%2$s`', 'wc-price-based-country-pro' ), $product->get_name(), $row['_price_method'] ) );
			}

			if ( empty( $row['_sale_price_dates'] ) && ( ! empty( $row['_sale_price_dates_from'] ) || ! empty( $row['_sale_price_dates_to'] ) ) ) {
				if ( 'manual' === $row['_price_method'] ) {
					$logger && call_user_func( $logger, '<b>WARNING</b>: ' . __( 'Sale price dates method is empty. Set it to `manual`', 'wc-price-based-country-pro' ) );
					$row['_sale_price_dates'] = 'manual';
				} else {
					$logger && call_user_func( $logger, '<b>WARNING</b>: ' . __( 'Skip sale dates because price method is `exchange rate`', 'wc-price-based-country-pro' ) );
				}
			}

			/**
			 * Update product pricing.
			 */
			wcpbc_update_product_pricing( $product->get_id(), $zone, $row );

			// Translators: %s: Updated fields.
			$logger && call_user_func( $logger, sprintf( __( 'Updated `%s`', 'wc-price-based-country-pro' ), implode( '`, `', array_keys( $row ) ) ) );

			if ( is_a( $product, 'WC_Product_Variation' ) ) {
				if ( ! isset( $this->variable_sync[ $zone->get_id() ] ) ) {
					$this->variable_sync[ $zone->get_id() ] = [];
				}

				$this->variable_sync[ $zone->get_id() ][] = $product->get_parent_id();
			}
		}

		return false;
	}

	/**
	 * Sync of a parent product with children.
	 */
	public function variable_product_sync() {
		if ( ! is_array( $this->variable_sync ) ) {
			return;
		}

		foreach ( $this->variable_sync as $zone_id => $product_ids ) {

			$_product_ids = array_unique( array_filter( array_map( 'absint', $product_ids ) ) );

			WCPBC_Backward_Compatibility::product_price_sync(
				[
					'zone'        => $zone_id,
					'product_ids' => $_product_ids,
				],
				count( $_product_ids ) + 1
			);
		}

		$this->variable_sync = false;
	}
}

WCPBC_WPAllImport::init();
