<?php
/**
 *
 * Handle exchange rates updates from API providers
 *
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WCPBC_Update_Exchange_Rates' ) ) :

	/**
	 * WCPBC_Update_Exchange_Rates Class.
	 */
	class WCPBC_Update_Exchange_Rates {

		const OPTION_NAME = 'wc_price_based_country_exchange_rates';

		/**
		 * Exchange rates data.
		 *
		 * @var array
		 */
		private static $data;

		/**
		 * Exchange provides.
		 *
		 * @var array
		 */
		private static $exchange_rates_providers = array();

		/**
		 * Init.
		 */
		public static function init() {
			self::set_defaults();
			self::read();
			self::init_hooks();
		}

		/**
		 * Set defaults properties.
		 */
		private static function set_defaults() {
			self::$data = [
				'from'      => '',
				'rates'     => [],
				'timestamp' => false,
				'source'    => '',
			];
		}

		/**
		 * Reads data from the database.
		 */
		private static function read() {
			$data = get_option( self::OPTION_NAME, false );

			if ( ( isset( $data['from'], $data['rates'], $data['timestamp'], $data['source'] ) && is_array( $data['rates'] ) ) ) {
				self::$data = [
					'from'      => $data['from'],
					'rates'     => array_map( 'floatval', $data['rates'] ),
					'timestamp' => floatval( $data['timestamp'] ),
					'source'    => $data['source'],
				];

			} elseif ( false === $data ) {
				// First install.
				self::fetch();
			}
		}

		/**
		 * Hook actions and filters
		 */
		private static function init_hooks() {
			add_action( 'woocommerce_scheduled_sales', array( __CLASS__, 'update_all_zones' ), 5 );
			add_action( 'update_option_woocommerce_currency', array( __CLASS__, 'update_all_zones' ) );
		}

		/**
		 * Gets data from API and saves them.
		 */
		private static function fetch() {
			$source = self::get_provider();

			if ( ! $source ) {

				return new WP_Error(
					'update_exchange_rate_error',
					sprintf(
						// translators: 1: Source, 2: Currency.
						__(
							'Error fetching the exchange rate. %1$s does not exits.',
							'wc-price-based-country-pro'
						),
						self::get_provider_id()
					)
				);
			}

			$from_currency = wcpbc_get_base_currency();

			$rates = $source->get_exchange_rates( $from_currency );

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				WCPBC_Debug_Logger::log_message( sprintf( 'Exchange rates from %s =>', $source->get_name() ), __METHOD__ );
				WCPBC_Debug_Logger::log_response( $rates, __METHOD__ );
			}

			if ( ! empty( $rates ) && is_array( $rates ) ) {

				self::$data = [
					'from'      => $from_currency,
					'rates'     => array_map( 'floatval', $rates ),
					'timestamp' => time(),
					'source'    => $source->get_id(),
				];

				self::$data['rates'][ $from_currency ] = 1; // Force to 1 the exchange rate of the base currency.

				update_option( self::OPTION_NAME, self::$data );

			} else {

				return new WP_Error(
					'update_exchange_rate_error',
					sprintf(
						// translators: 1: Source, 2: Currency.
						__(
							'Error fetching the exchange rate. %1$s did not return values.',
							'wc-price-based-country-pro'
						),
						$source->get_name()
					)
				);
			}

			return true;
		}

		/**
		 * Do need refresh the rates?
		 *
		 * @param int $expiration Number of seconds after the timestamp expires. Default HOUR_IN_SECONDS.
		 * @return bool
		 */
		private static function data_expired( $expiration = HOUR_IN_SECONDS ) {
			if ( empty( self::$data['from'] ) || empty( self::$data['timestamp'] ) || empty( self::$data['source'] ) || ! is_array( self::$data['rates'] ) ) {
				return true;
			}
			return ( time() > self::$data['timestamp'] + $expiration ) || ( self::get_provider_id() !== self::$data['source'] );
		}

		/**
		 * Returns a exchange rate.
		 *
		 * @param string $from From currency.
		 * @param string $to To currency.
		 */
		public static function get_rate( $from, $to ) {
			if ( $from === $to ) {
				return 1;
			}

			$rate = false;

			if ( ! isset( self::$data['from'], self::$data['rates'][ $to ] ) ) {
				return $rate;
			}

			$rate = self::$data['rates'][ $to ];

			if ( self::$data['from'] !== $from ) {

				$from_rate = isset( self::$data['rates'][ $from ] ) ? self::$data['rates'][ $from ] : false;

				if ( $from_rate ) {
					$rate = $rate / $from_rate;
				} else {
					$rate = false;
				}
			}

			return $rate;
		}

		/**
		 * Convert an amount between two currencies.
		 *
		 * @param array $args {
		 *     Optional. Array of arguments. Default empty array.
		 *
		 *     @type float $amount Amount to convert. Default 1
		 *     @type string $from From currency. Default to current currency.
		 *     @type string $to To currency. Default to shop base currency.
		 * }
		 */
		public static function convert( $args ) {
			$args = wp_parse_args(
				$args,
				[
					'amount' => 1,
					'from'   => get_woocommerce_currency(),
					'to'     => wcpbc_get_base_currency(),
				]
			);

			$amount = floatval( $args['amount'] );
			$rate   = self::get_rate( $args['from'], $args['to'] );

			if ( $rate ) {
				$amount = $amount * $rate;
			}

			return $amount;
		}

		/**
		 * Updates the exchange rate of all zones.
		 */
		public static function update_all_zones() {

			$zones = WCPBC_Pricing_Zones::get_zones();

			foreach ( $zones as $zone ) {
				try {
					self::update_zone( $zone );
				} catch ( Exception $e ) {
					WCPBC_Debug_Logger::log_error( $e->getMessage(), __METHOD__ );
				}
			}

			WCPBC_Pricing_Zones::bulk_save( $zones );
		}

		/**
		 * Updates the exchange rate of a single zone.
		 *
		 * @param WC_Pricing_Zone $zone Pricing zone instance.
		 * @throws Exception Throws exception if there is no rate for currency of the zone.
		 */
		public static function update_zone( $zone ) {
			if ( self::data_expired() ) {

				$result = self::fetch();

				if ( is_wp_error( $result ) ) {
					throw new Exception( $result->get_error_message() );
				}
			}

			$rate = self::get_rate(
				wcpbc_get_base_currency(),
				$zone->get_currency()
			);

			if ( ! $rate ) {
				// translators: 1: Source, 2: Currency.
				throw new Exception(
					sprintf(
						// translators: 1: Source, 2: Currency.
						__(
							'Error fetching the exchange rate. Rate from %1$s to %2$s not found in %3$s.',
							'wc-price-based-country-pro'
						),
						wcpbc_get_base_currency(),
						$zone->get_currency(),
						self::get_provider()->get_name()
					)
				);
			}

			if ( $zone->get_auto_exchange_rate() ) {
				$zone->set_exchange_rate( $rate * ( 1 + ( $zone->get_exchange_rate_fee() / 100 ) ) );
			}

			$zone->set_real_exchange_rate( $rate );
		}

		/**
		 * Get exchange rates providers.
		 *
		 * @return array
		 */
		public static function get_providers() {
			if ( ! empty( self::$exchange_rates_providers ) ) {
				return self::$exchange_rates_providers;
			}

			$load_files = array(
				'class-wcpbc-floatrates.php',
				'class-wcpbc-open-exchange-rates.php',
				'class-wcpbc-xrates.php',
			);

			foreach ( $load_files as $file ) {
				$provider = include dirname( __FILE__ ) . "/exchage-rates-providers/{$file}";

				self::$exchange_rates_providers[ $provider->get_id() ] = $provider;
			}

			self::$exchange_rates_providers = apply_filters( 'wc_price_based_country_exchange_providers', self::$exchange_rates_providers );

			return self::$exchange_rates_providers;
		}

		/**
		 * Returns the provider selected by the user.
		 *
		 * @return WCPBC_Exchange_Rates_Provider
		 */
		public static function get_provider() {
			$providers   = self::get_providers();
			$provider_id = self::get_provider_id();
			return isset( $providers[ $provider_id ] ) ? $providers[ $provider_id ] : false;
		}

		/**
		 * Returns the provider ID selected by the user.
		 *
		 * @return string
		 */
		public static function get_provider_id() {
			return get_option( 'wc_price_based_country_exchange_rate_api', 'floatrates' );
		}

		/**
		 * Return an array of the provider options.
		 *
		 * @return array
		 */
		public static function get_providers_fields() {
			$fields    = array();
			$providers = self::get_providers();
			foreach ( $providers as $id => $provider ) {
				$fields = array_merge( $fields, $provider->get_options_fields() );
			}
			return $fields;

		}

		/**
		 * Update the exchange rate of a zone.
		 *
		 * @param WCPBC_Pricing_Zone $zone Zone to update.
		 * @param bool               $save Save the zone after updating?.
		 * @return bool|WP_Error True is sucess. WP_Error on error.
		 */
		public static function update_exchange_rate( $zone, $save = false ) {

			$error = false;

			try {

				self::update_zone( $zone );

			} catch ( Exception $e ) {
				$error = new WP_Error( 'update_exchange_rate_error', $e->getMessage() );
			}

			if ( is_wp_error( $error ) && $zone->get_auto_exchange_rate() ) {
				return $error;
			}

			if ( $save && ! is_wp_error( $error ) ) {
				$zone->save();
			}

			return true;
		}

		/**
		 * Fixes wrong "exchange rate" of orders. Callback of an admin tool.
		 *
		 * @since 2.16.0
		 * @see WCPBC_Admin_Pro::debug_tools
		 */
		public static function fix_order_exchange_rate() {
			global $wpdb;

			$rows = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT posts.ID, _order_currency.meta_value AS order_currency
					FROM {$wpdb->posts} AS posts
						INNER JOIN {$wpdb->postmeta} AS _order_currency ON _order_currency.post_id = posts.ID AND _order_currency.meta_key = '_order_currency'
						LEFT JOIN {$wpdb->postmeta} AS _base_exchange_rate ON _base_exchange_rate.post_id = posts.ID AND _base_exchange_rate.meta_key = '_wcpbc_base_exchange_rate'
					WHERE posts.post_type IN ( 'shop_order', 'shop_order_refund' )
						AND ( trim(coalesce( _base_exchange_rate.meta_value, '')) = '' OR _base_exchange_rate.meta_value = '1' )
						AND _order_currency.meta_value <> %s",
					wcpbc_get_base_currency()
				)
			);

			if ( count( $rows ) ) {

				foreach ( $rows as $row ) {
					$base_exchange_rate = self::get_rate( $row->order_currency, wcpbc_get_base_currency() );
					if ( $base_exchange_rate ) {
						update_post_meta( $row->ID, '_wcpbc_base_exchange_rate', $base_exchange_rate );
					}
				}

				if ( is_callable( array( 'Automattic\WooCommerce\Admin\API\Reports\Cache', 'invalidate' ) ) ) {
					// Clear Analitycs cache after update.
					Automattic\WooCommerce\Admin\API\Reports\Cache::invalidate();
				}
			}

			return __( 'Orders exchange rate regenerated, and the Analytics cache have been cleared.', 'wc-price-based-country-pro' );
		}
	}
endif;
