<?php
/**
 * Handle integration with WooCommerce Product Bundles by WooCommerce.
 *
 * @since 2.22.0
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once dirname( __FILE__ ) . '/class-wcpbc-product-bundles-sync-prices.php';

/**
 * WCPBC_Product_Bundles Class
 */
class WCPBC_Product_Bundles {

	/**
	 * Check enviroment notice.
	 *
	 * @var string
	 */
	private static $notice = '';

	/**
	 * Doing sync flag.
	 *
	 * @var bool
	 */
	private static $doing_sync = false;

	/**
	 * Bundled Subscription price HTML flag.
	 *
	 * @var bool
	 */
	private static $doing_sub_price_html = false;

	/**
	 * Hook actions and filters
	 */
	public static function init() {
		add_action( 'woocommerce_product_options_general_product_data', array( __CLASS__, 'options_general_product_data' ) );
		add_action( 'woocommerce_process_product_meta_bundle', array( __CLASS__, 'process_product_meta' ) );
		add_action( 'wc_price_based_country_manual_order_before_line_save', array( __CLASS__, 'manual_order_before_line_save' ), 10, 2 );
		add_action( 'wc_price_based_country_frontend_princing_init', array( __CLASS__, 'frontend_princing_init' ) );
		add_filter( 'wc_price_based_country_ajax_geolocation_product_data', array( __CLASS__, 'ajax_geolocation_product_data' ), 10, 3 );
		add_filter( 'wpml_config_array', array( __CLASS__, 'add_custom_fields' ) );

		/**
		 * AJAX geo. Don't wrapper the subscription price HTML inside the bundle price HTML.
		 */
		if ( WCPBC_Ajax_Geolocation::is_enabled() ) {
			add_filter( 'woocommerce_get_bundle_price_html', array( __CLASS__, 'before_subcription_price_html' ), 10, 2 );
			add_filter( 'woocommerce_get_price_html', array( __CLASS__, 'after_subcription_price_html' ), -10, 2 );
		}
	}

	/**
	 * Set the correct values to the regular and sale price.
	 */
	public static function options_general_product_data() {
		global $product_object;
		if ( is_callable( array( $product_object, 'get_type' ) ) && 'bundle' !== $product_object->get_type() ) {
			return;
		}

		$params = array();
		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			foreach ( array( '_regular_price', '_sale_price' ) as $prop ) {
				$key            = $zone->get_postmetakey( $prop );
				$params[ $key ] = $zone->get_postmeta( $product_object->get_id(), '_wc_pb_base' . $prop );
			}
		}
		wc_enqueue_js(
			sprintf(
				"var _wcpbc_product_bundles_prices = %s;
				for (const key in _wcpbc_product_bundles_prices ) {
					$('#' + key).val( _wcpbc_product_bundles_prices[key]);
				}
				",
				wp_json_encode( $params )
			)
		);
	}

	/**
	 * Save product metadata
	 *
	 * @param int $post_id Post ID.
	 */
	public static function process_product_meta( $post_id ) {

		WCPBC_Admin_Meta_Boxes::process_product_meta( $post_id );

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			if ( ! $zone->is_exchange_rate_price( $post_id ) ) {
				self::copy_metadata( $post_id, $zone );
			} else {
				self::set_by_exchange_rate_metadata( $post_id, $zone );
			}
		}
	}

	/**
	 * Copy product price to the bundle metadata.
	 *
	 * @param int                $post_id Post ID.
	 * @param WCPBC_Pricing_Zone $zone Pricing zone instance..
	 */
	public static function copy_metadata( $post_id, $zone ) {
		$regular_price = $zone->get_postmeta( $post_id, '_regular_price' );
		$sale_price    = $zone->get_postmeta( $post_id, '_sale_price' );
		$price         = $zone->get_postmeta( $post_id, '_price' );

		$zone->set_postmeta( $post_id, '_wc_pb_base_regular_price', $regular_price );
		$zone->set_postmeta( $post_id, '_wc_pb_base_sale_price', $sale_price );
		$zone->set_postmeta( $post_id, '_wc_pb_base_price', $price );
	}

	/**
	 * Set product price to the bundle metadata.
	 *
	 * @param int                $post_id Post ID.
	 * @param WCPBC_Pricing_Zone $zone Pricing zone instance..
	 */
	public static function set_by_exchange_rate_metadata( $post_id, $zone ) {
		$regular_price = $zone->get_exchange_rate_price_by_post( $post_id, '_wc_pb_base_regular_price' );
		$sale_price    = $zone->get_exchange_rate_price_by_post( $post_id, '_wc_pb_base_sale_price' );
		$price         = $zone->get_exchange_rate_price_by_post( $post_id, '_wc_pb_base_price' );

		$zone->set_postmeta( $post_id, '_wc_pb_base_regular_price', $regular_price );
		$zone->set_postmeta( $post_id, '_wc_pb_base_sale_price', $sale_price );
		$zone->set_postmeta( $post_id, '_wc_pb_base_price', $price );
	}

	/**
	 * Update the order line item subtotal before save.
	 *
	 * @since 2.5.2
	 * @param WC_Order_Line_Item $line_item Order line item instance.
	 * @param WC_Product         $product Product instance, of the line item.
	 */
	public static function manual_order_before_line_save( $line_item, $product ) {
		if ( ! ( function_exists( 'wc_pb_is_bundled_order_item' ) && function_exists( 'wc_pb_get_bundled_order_item_container' ) && wc_pb_is_bundled_order_item( $line_item ) ) ) {
			return;
		}
		$container_line_item = wc_pb_get_bundled_order_item_container( $line_item );
		$container_product   = $container_line_item->get_product();
		$bundled_item_id     = $line_item->get_meta( '_bundled_item_id' );
		$bundled_item        = $container_product->get_bundled_item( $bundled_item_id );
		if ( $bundled_item->is_priced_individually() ) {
			$bundled_item_discount = $bundled_item->get_discount();
			if ( $bundled_item_discount ) {
				$subtotal = wc_get_price_excluding_tax( $product, array( 'qty' => $bundled_item->get_quantity() * $container_line_item->get_quantity() ) ) * ( 1 - (float) $bundled_item_discount / 100 );
				$line_item->set_total( $subtotal );
				$line_item->set_subtotal( $subtotal );
			}
		} else {
			$line_item->set_total( 0 );
			$line_item->set_subtotal( 0 );
		}
	}

	/**
	 * Support for Ajax geolocation.
	 *
	 * @param array      $data Array of product data.
	 * @param WC_Product $product Product instance.
	 * @param bool       $is_single Is single page?.
	 * @return array
	 */
	public static function ajax_geolocation_product_data( $data, $product, $is_single ) {

		if ( ! $is_single ) {
			return $data;
		}

		$bundle_sell_ids = WC_PB_BS_Product::get_bundle_sell_ids( $product );
		if ( ! empty( $bundle_sell_ids ) ) {
			$bundle = WC_PB_BS_Product::get_bundle( $bundle_sell_ids, $product );
		} else {
			$bundle = $product;
		}

		if ( 'bundle' === $bundle->get_type() ) {

			$data['bundle_price_data'] = is_callable( array( $bundle, 'get_bundle_form_data' ) ) ? $bundle->get_bundle_form_data() : $bundle->get_bundle_price_data();
			$data['bundled_items']     = array();

			foreach ( $bundle->get_bundled_items() as $bundled_item ) {
				$bundled_item->add_price_filters();

				$data['bundled_items'][ $bundled_item->product->get_id() ] = array(
					'id'         => $bundled_item->product->get_id(),
					'price_html' => $bundled_item->product->get_price_html(),
					'variations' => '',
				);

				if ( $bundled_item->product->get_type() === 'variable' || $bundled_item->product->get_type() === 'variable-subscription' ) {
					$data['bundled_items'][ $bundled_item->product->get_id() ]['variations'] = $bundled_item->get_product_variations();
				}

				$bundled_item->remove_price_filters();
			}
		}
		return $data;
	}

	/**
	 * WPML integration. Custom fields.
	 *
	 * @since 2.9.5
	 * @param array $wpml_config WPML config array.
	 */
	public static function add_custom_fields( $wpml_config ) {
		$meta_keys = array( '_wc_pb_base_regular_price', '_wc_pb_base_sale_price', '_wc_pb_base_price' );

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			foreach ( $meta_keys as $field ) {
				$wpml_config['wpml-config']['custom-fields']['custom-field'][] = array(
					'value' => $zone->get_postmetakey( $field ),
					'attr'  => array(
						'action' => 'copy',
					),
				);
			}
		}

		return $wpml_config;
	}

	/**
	 * Init frontend hooks.
	 */
	public static function frontend_princing_init() {
		add_filter( 'woocommerce_bundles_synced_contents_data', array( __CLASS__, 'before_synced_bundle' ) );
		add_action( 'woocommerce_bundles_synced_bundle', array( __CLASS__, 'synced_bundle' ) );
		add_filter( 'woocommerce_product_get_price', array( __CLASS__, 'get_product_price_property' ), 5, 2 );
		add_filter( 'woocommerce_product_get_regular_price', array( __CLASS__, 'get_product_price_property' ), 5, 2 );
		add_filter( 'woocommerce_product_get_sale_price', array( __CLASS__, 'get_product_price_property' ), 5, 2 );
		add_filter( 'woocommerce_bundled_item_hash', array( 'WCPBC_Frontend_Pricing', 'get_variation_prices_hash' ) );
		add_filter( 'woocommerce_bundled_items', array( __CLASS__, 'bundled_items' ), 10, 2 );

		/**
		 * Product Bundles: Variation Bundles compatiblity
		 */
		if ( is_callable( array( 'WC_PB_Variable_Bundles', 'maybe_get_variation_bundle' ) ) ) {
			add_filter( 'wc_price_based_country_should_filter_property', array( __CLASS__, 'should_filter_property' ), 10, 2 );
			add_filter( 'woocommerce_product_variation_get_price', array( __CLASS__, 'get_variation_bundle_price' ), 5, 2 );
			add_filter( 'woocommerce_product_variation_get_regular_price', array( __CLASS__, 'get_variation_bundle_price' ), 5, 2 );
			add_filter( 'woocommerce_product_variation_get_sale_price', array( __CLASS__, 'get_variation_bundle_price' ), 5, 2 );
			add_filter( 'woocommerce_variation_prices', array( __CLASS__, 'sync_variation_bundle_prices' ), -9999, 3 );
		}
	}

	/**
	 * Remove filters before Product Bundles syncs the raw prices.
	 *
	 * @param mixed $value Raw value. No use.
	 */
	public static function before_synced_bundle( $value = false ) {
		if ( class_exists( 'WCPBC_Subscriptions' ) ) {
			remove_filter( 'woocommerce_product_get__subscription_sign_up_fee', array( 'WCPBC_Subscriptions', 'get_product_subscription_prop' ), 5, 2 );
		}

		remove_filter( 'woocommerce_get_variation_prices_hash', array( 'WCPBC_Frontend_Pricing', 'get_variation_prices_hash' ) );
		remove_filter( 'woocommerce_variation_prices_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );
		remove_filter( 'woocommerce_variation_prices_regular_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );
		remove_filter( 'woocommerce_variation_prices_sale_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );

		// Clear PB cache.
		if ( is_callable( array( 'WC_PB_Helpers', 'cache_invalidate' ) ) ) {
			WC_PB_Helpers::cache_invalidate();
		}

		// Doing sync flag.
		self::$doing_sync = true;

		return $value;
	}

	/**
	 * Add filters after sync raw prices
	 */
	private static function add_price_variation_filters() {
		if ( class_exists( 'WCPBC_Subscriptions' ) ) {
			add_filter( 'woocommerce_product_get__subscription_sign_up_fee', array( 'WCPBC_Subscriptions', 'get_product_subscription_prop' ), 5, 2 );
		}

		add_filter( 'woocommerce_get_variation_prices_hash', array( 'WCPBC_Frontend_Pricing', 'get_variation_prices_hash' ) );
		add_filter( 'woocommerce_variation_prices_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );
		add_filter( 'woocommerce_variation_prices_regular_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );
		add_filter( 'woocommerce_variation_prices_sale_price', array( 'WCPBC_Frontend_Pricing', 'get_product_price_property' ), 5, 2 );
	}

	/**
	 * Sync the current zone raw prices of the bundle.
	 *
	 * @param WC_Product_Bundle $bundle Product bundle.
	 */
	public static function synced_bundle( $bundle ) {

		// Add the price filters after sync.
		if ( self::$doing_sync ) {
			self::add_price_variation_filters();
		}

		// Sync raw prices.
		$sync_prices = new WCPBC_Product_Bundles_Sync_Prices( $bundle, wcpbc_the_zone() );
		$sync_prices->sync_raw_prices();

		foreach ( $sync_prices->get_raw_prices() as $raw_price => $value ) {

			$setter = "set_{$raw_price}";
			if ( is_callable( array( $bundle, $setter ) ) ) {
				$bundle->{$setter}( $value );
			}
		}

		// Doing sync flag.
		self::$doing_sync = false;

	}

	/**
	 * Retruns a product price property.
	 *
	 * @param mixed      $value Property value.
	 * @param WC_Product $product Product instance.
	 * @return mixed
	 */
	public static function get_product_price_property( $value, $product ) {
		$prop = str_replace( 'woocommerce_product_get_', '', current_filter() );

		if ( 'bundle' === $product->get_type() && ! array_key_exists( $prop, $product->get_changes() ) ) {
			$meta_key = '_wc_pb_base_' . $prop;
			return wcpbc_the_zone()->get_price_prop( $product, $value, $meta_key );
		}
		return $value;
	}

	/**
	 * Fixes the recurring price of the bundle items.
	 *
	 * @param array             $bundled_items Array of WC_Bundled_Item objects.
	 * @param WC_Product_Bundle $product Product instance.
	 */
	public static function bundled_items( $bundled_items, $product ) {

		static $avoid_recurring = false;

		if ( self::$doing_sync || $avoid_recurring || ! is_array( $bundled_items ) ) {
			return $bundled_items;
		}

		$avoid_recurring = true;

		$sync_prices = new WCPBC_Product_Bundles_Sync_Prices( $product, wcpbc_the_zone() );

		foreach ( $bundled_items as $bundled_item ) {
			if ( ! ( $bundled_item->is_subscription() && $bundled_item->is_priced_individually() && $bundled_item->is_purchasable() ) ) {
				continue;
			}

			if ( isset( $bundled_item->_wcpbc_processed ) ) {
				continue;
			}

			$item_prices = $sync_prices->sync_bundle_item_prices( $bundled_item );

			foreach ( array( 'min_price_product', 'min_regular_recurring_price', 'max_regular_recurring_price', 'min_recurring_price', 'max_recurring_price' ) as $prop ) {
				if ( isset( $item_prices->{$prop} ) ) {
					$bundled_item->{$prop} = $item_prices->{$prop};
				}
			}

			$bundled_item->_wcpbc_processed = true;
		}

		$avoid_recurring = false;

		return $bundled_items;
	}

	/**
	 * Before generating the bundle subscription price HTML.
	 *
	 * @param string            $price_html Price HTML.
	 * @param WC_Product_Bundle $product Product instance.
	 */
	public static function before_subcription_price_html( $price_html, $product ) {
		if ( is_a( $product, 'WC_Product_Bundle' ) && class_exists( 'WCPBC_Subscriptions' ) ) {
			self::$doing_sub_price_html = $product->get_id();

			remove_filter( 'woocommerce_subscriptions_product_price_string', array( 'WCPBC_Subscriptions', 'product_price_string' ), 10, 2 );
		}
		return $price_html;
	}

	/**
	 * After generating the bundle subscription price HTML.
	 *
	 * @param string            $price_html Price HTML.
	 * @param WC_Product_Bundle $product Product instance.
	 */
	public static function after_subcription_price_html( $price_html, $product ) {
		if ( is_a( $product, 'WC_Product_Bundle' ) && self::$doing_sub_price_html === $product->get_id() ) {
			self::$doing_sub_price_html = false;

			add_filter( 'woocommerce_subscriptions_product_price_string', array( 'WCPBC_Subscriptions', 'product_price_string' ), 10, 2 );
		}
		return $price_html;
	}

	/**
	 * Do not filter the price if the product is a variation bundle.
	 *
	 * @param bool       $should_filter True/False.
	 * @param WC_Product $product Product instance.
	 */
	public static function should_filter_property( $should_filter, $product ) {
		if ( $product instanceof WC_Product_Variation && WC_PB_Variable_Bundles::maybe_get_variation_bundle( $product ) ) {
			$should_filter = false;
		}
		return $should_filter;
	}

	/**
	 * Retrun a variation bundle price property.
	 *
	 * @since 1.9.0
	 * @param mixed      $value Property value.
	 * @param WC_Product $product Product instance.
	 * @return mixed
	 */
	public static function get_variation_bundle_price( $value, $product ) {
		if ( $product instanceof WC_Product_Variation ) {
			$bundle = WC_PB_Variable_Bundles::maybe_get_variation_bundle( $product );
			if ( $bundle ) {
				$prop     = str_replace( 'woocommerce_product_variation_get_', '', current_filter() );
				$meta_key = 'regular_price' === $prop ? '_regular_price' : '_price';
				$value    = wcpbc_the_zone()->get_postmeta( $bundle->get_id(), $meta_key );
			}
		}
		return $value;
	}

	/**
	 * Sync variation bundle prices.
	 *
	 * @param array $prices_array Array of prices.
	 */
	public static function sync_variation_bundle_prices( $prices_array ) {

		$cache = array();

		foreach ( array_keys( $prices_array['price'] ) as $variation_id ) {

			$variation_bundle = WC_PB_Variable_Bundles::maybe_get_variation_bundle( $variation_id );

			if ( $variation_bundle ) {

				self::synced_bundle( $variation_bundle );

				$cache[ $variation_id ] = $variation_bundle;
			}
		}

		foreach ( $cache as $variation_id => $variation_bundle ) {
			// Stores in the cache for the Variation Bundles plugin get the synced product. Do it after sync to preven cache invalidate on sync.
			WC_PB_Helpers::cache_set( $variation_id, $variation_bundle, 'variation_bundles' );
		}

		return $prices_array;
	}

	/**
	 * Checks the environment for compatibility problems.
	 *
	 * @return boolean
	 */
	public static function check_environment() {
		$plugin_version = isset( $GLOBALS['woocommerce_bundles']->version ) ? $GLOBALS['woocommerce_bundles']->version : 'unknown';
		$min_version    = '6.5.2';

		if ( 'unknown' === $plugin_version || version_compare( $plugin_version, $min_version, '<' ) ) {
			// translators: 1: HTML tag, 2: HTML tag, 3: Product Bundles version.
			self::$notice = sprintf( __( '%1$sPrice Based on Country Pro & WooCommerce Product Bundles%2$s compatibility requires WooCommerce Product Bundles +%4$s. You are running Product Bundles %3$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', $plugin_version, $min_version );
			add_action( 'admin_notices', array( __CLASS__, 'min_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Display admin minimun version required
	 */
	public static function min_version_notice() {
		echo '<div id="message" class="error"><p>' . wp_kses_post( self::$notice ) . '</p></div>';
	}
}

if ( WCPBC_Product_Bundles::check_environment() ) {
	WCPBC_Product_Bundles::init();
}
