<?php
/**
 * Pricing zone importer class file
 *
 * @since 3.0.0
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Importer' ) ) {
	return;
}

/**
 * WCPBC_Pricingzone_Importer class.
 */
class WCPBC_Pricingzone_Importer {

	/**
	 * Importer parameters.
	 *
	 * @var array
	 */
	protected $params;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->params = array(
			'file'      => '',
			'delimiter' => ',',
			'step'      => empty( $_GET['step'] ) ? 0 : absint( $_GET['step'] ), // phpcs:ignore WordPress.Security.NonceVerification
		);
	}

	/**
	 * Registered callback function for the WordPress Importer.
	 *
	 * Manages the three separate stages of the CSV import process.
	 */
	public function dispatch() {

		$this->header();

		switch ( $this->params['step'] ) {

			case 0:
				$this->greet();
				break;

			case 1:
				$this->import();
				break;
		}

		$this->footer();
	}

	/**
	 * Output information about the uploading process.
	 */
	protected function greet() {

		echo '<div class="narrow">';
		echo '<p>' . esc_html__( 'Hi there! Upload a CSV file containing pricing zones to import the contents into your shop. Choose a .csv file to upload, then click "Upload file and import".', 'wc-price-based-country-pro' ) . '</p>';

		/* translators: 1: Link to tax rates sample file 2: Closing link. */
		echo '<p>' . sprintf( esc_html__( 'Your CSV needs to include columns with specific column headers. %1$sClick here to download a sample%2$s.', 'wc-price-based-country-pro' ), '<a href="' . esc_url( WC_Product_Price_Based_Country_Pro::plugin_url() ) . 'sample-data/sample_pricing_zones.csv">', '</a>' ) . '</p>';

		$action = 'admin.php?import=wcpbc_pricing_zone_csv&step=1';

		$bytes      = apply_filters( 'import_upload_size_limit', wp_max_upload_size() ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
		$size       = size_format( $bytes );
		$upload_dir = wp_upload_dir();
		if ( ! empty( $upload_dir['error'] ) ) :
			?>
			<div class="error">
				<p><?php esc_html_e( 'Before you can upload your import file, you will need to fix the following error:', 'wc-price-based-country-pro' ); ?></p>
				<p><strong><?php echo esc_html( $upload_dir['error'] ); ?></strong></p>
			</div>
		<?php else : ?>
			<form enctype="multipart/form-data" id="import-upload-form" method="post" action="<?php echo esc_attr( wp_nonce_url( $action, 'import-upload' ) ); ?>">
				<table class="form-table">
					<tbody>
						<tr>
							<th>
								<label for="upload"><?php esc_html_e( 'Choose a file from your computer:', 'wc-price-based-country-pro' ); ?></label>
							</th>
							<td>
								<input type="file" id="upload" name="import" size="25" />
								<input type="hidden" name="action" value="save" />
								<input type="hidden" name="max_file_size" value="<?php echo absint( $bytes ); ?>" />
								<small>
									<?php
									printf(
										/* translators: %s: maximum upload size */
										esc_html__( 'Maximum size: %s', 'wc-price-based-country-pro' ),
										esc_attr( $size )
									);
									?>
								</small>
							</td>
						</tr>
						<tr>
							<th>
								<label for="file_url"><?php esc_html_e( 'OR enter path to file:', 'woocommerce' ); ?></label>
							</th>
							<td>
								<?php echo ' ' . esc_html( ABSPATH ) . ' '; ?><input type="text" id="file_url" name="file_url" size="25" />
							</td>
						</tr>
						<tr>
							<th><label><?php esc_html_e( 'Delimiter', 'woocommerce' ); ?></label><br/></th>
							<td><input type="text" name="delimiter" placeholder="," size="2" /></td>
						</tr>
					</tbody>
				</table>
				<p class="submit">
					<button type="submit" class="button" value="<?php esc_attr_e( 'Upload file and import', 'woocommerce' ); ?>">
						<?php esc_html_e( 'Upload file and import', 'woocommerce' ); ?>
					</button>
				</p>
			</form>
			<?php
		endif;

		echo '</div>';

		wc_enqueue_js(
			"$('#import-upload-form').on('submit', function(){
				return confirm('" . esc_html__( 'The import process will replace the current pricing zone settings. Do you want to continue?', 'wc-price-based-country-pro' ) . "');
			});"
		);
	}

	/**
	 * Import is starting.
	 */
	protected function import_start() {
		if ( function_exists( 'gc_enable' ) ) {
			gc_enable(); // phpcs:ignore PHPCompatibility.FunctionUse.NewFunctions.gc_enableFound
		}

		add_filter( 'http_request_timeout', array( $this, 'bump_request_timeout' ) );

		wc_set_time_limit( 0 );
		@ob_flush();
		@flush();
	}

	/**
	 * Added to http_request_timeout filter to force timeout at 60 seconds during import.
	 *
	 * @param  int $val Value.
	 * @return int 60
	 */
	public function bump_request_timeout( $val ) {
		return 60;
	}

	/**
	 * Import the file if it exists and is valid.
	 */
	protected function import() {

		$this->import_start();

		$errors = $this->handle_upload();

		if ( is_wp_error( $errors ) ) {
			$this->show_errors( $errors );
			return;
		}

		$handle = fopen( $this->params['file'], 'r' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen

		if ( false === $handle ) {
			return;
		}

		$header = fgetcsv( $handle, 0, $this->params['delimiter'] );

		if ( is_array( $header ) && count( $header ) ) {

			$loop      = 0;
			$zones     = array();
			$countries = array();

			$row = fgetcsv( $handle, 0, $this->params['delimiter'] );

			while ( false !== $row ) {

				$loop++;
				$data = array();

				foreach ( $header as $index => $field ) {
					$data[ $field ] = $row[ $index ];
				}

				$zone   = WCPBC_Pricing_Zones::create();
				$errors = WCPBC_Pricing_Zones::populate( $zone, $data );

				if ( ! is_wp_error( $errors ) && count( array_intersect( $countries, $zone->get_countries() ) ) ) {

					$errors = new WP_Error(
						'wc_price_based_country_import_duplicate_country',
						// translators: Line number.
						sprintf( __( 'The country(s) "%s" is already in another zone.', 'wc-price-based-country-pro' ), implode( ', ', array_intersect( $countries, $zone->get_countries() ) ) )
					);
				}

				if ( ! is_wp_error( $errors ) ) {
					try {
						WCPBC_Update_Exchange_Rates::update_zone( $zone );
					} catch ( Exception $e ) {
						if ( $zone->get_auto_exchange_rate() ) {
							$errors = new WP_Error( 'update_exchange_rate_error', $e->getMessage() );
						}
					}
				}

				if ( is_wp_error( $errors ) ) {
					// translators: Line number.
					$errors->add( 'wc_price_based_country_import_error_in_row', sprintf( __( 'Error found on row %s.', 'wc-price-based-country-pro' ), $loop + 1 ) );
					// Exit.
					$row = false;

				} else {

					$countries += $zone->get_countries();

					// Add zone to save.
					$zone->set_order( $loop );
					$zones[] = $zone;

					// Next row.
					$row = fgetcsv( $handle, 0, $this->params['delimiter'] );
				}
			}
		}

		fclose( $handle ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose

		if ( is_wp_error( $errors ) ) {

			$this->show_errors( $errors );

		} else {

			// Save the zones.
			WCPBC_Pricing_Zones::bulk_save( $zones, true );

			// Show Result.
			$this->show_result( count( $zones ) );
		}
	}

	/**
	 * Handles the CSV upload and initial parsing of the file to prepare for.
	 * displaying author import options.
	 *
	 * @return bool False if error uploading or invalid file, true otherwise
	 */
	protected function handle_upload() {
		check_admin_referer( 'import-upload' );

		$this->params['delimiter'] = empty( $_POST['delimiter'] ) ? ',' : (string) wc_clean( wp_unslash( $_POST['delimiter'] ) );

		$file_url = isset( $_POST['file_url'] ) ? wc_clean( wp_unslash( $_POST['file_url'] ) ) : '';

		if ( empty( $file_url ) ) {
			if ( ! isset( $_FILES['import'] ) ) {
				return new WP_Error( 'woocommerce_product_csv_importer_upload_file_empty', __( 'File is empty. Please upload something more substantial. This error could also be caused by uploads being disabled in your php.ini or by post_max_size being defined as smaller than upload_max_filesize in php.ini.', 'woocommerce' ) );
			}

			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
			if ( ! wc_is_file_valid_csv( wc_clean( wp_unslash( $_FILES['import']['name'] ) ), false ) ) {
				return new WP_Error( 'woocommerce_product_csv_importer_upload_file_invalid', __( 'Invalid file type. The importer supports CSV and TXT file formats.', 'woocommerce' ) );
			}

			$overrides = array(
				'test_form' => false,
				'mimes'     => array(
					'csv' => 'text/csv',
					'txt' => 'text/plain',
				),
			);
			$import    = $_FILES['import']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$upload    = wp_handle_upload( $import, $overrides );

			if ( isset( $upload['error'] ) ) {
				return new WP_Error( 'woocommerce_product_csv_importer_upload_error', $upload['error'] );
			}

			// Construct the object array.
			$object = array(
				'post_title'     => basename( $upload['file'] ),
				'post_content'   => $upload['url'],
				'post_mime_type' => $upload['type'],
				'guid'           => $upload['url'],
				'context'        => 'import',
				'post_status'    => 'private',
			);

			// Save the data.
			$id = wp_insert_attachment( $object, $upload['file'] );

			/*
			 * Schedule a cleanup for one day from now in case of failed
			 * import or missing wp_import_cleanup() call.
			 */
			wp_schedule_single_event( time() + DAY_IN_SECONDS, 'importer_scheduled_cleanup', array( $id ) );

			$this->params['file'] = $upload['file'];
			return true;

		} elseif (
			( 0 === stripos( realpath( ABSPATH . $file_url ), ABSPATH ) ) &&
			file_exists( ABSPATH . $file_url )
		) {
			if ( ! wc_is_file_valid_csv( ABSPATH . $file_url ) ) {
				return new WP_Error( 'woocommerce_product_csv_importer_upload_file_invalid', __( 'Invalid file type. The importer supports CSV and TXT file formats.', 'woocommerce' ) );
			}

			$this->params['file'] = ABSPATH . $file_url;
			return true;
		}
		// phpcs:enable

		return new WP_Error( 'woocommerce_product_csv_importer_upload_invalid_file', __( 'Please upload or provide the link to a valid CSV file.', 'woocommerce' ) );
	}


	/**
	 * Performs post-import cleanup of files and the cache.
	 */
	public function import_end() {
		echo '<p>' . esc_html__( 'All done!', 'woocommerce' ) . ' <a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=tax' ) ) . '">' . esc_html__( 'View tax rates', 'woocommerce' ) . '</a></p>';
	}

	/**
	 * Output header html.
	 */
	protected function header() {
		echo '<div class="wrap">';
		echo '<h1>' . esc_html__( 'Import pricing zones', 'woocommerce' ) . '</h1>';
	}

	/**
	 * Output footer html.
	 */
	protected function footer() {
		echo '</div>';
	}

	/**
	 * Show import errors.
	 *
	 * @param  WP_Error $errors Errors.
	 */
	protected function show_errors( $errors ) {
		echo '<div class="notice notice-error">';
		foreach ( $errors->get_error_messages() as $message ) {
			printf( '<p>%s</p>', esc_html( $message ) );
		}
		printf( '<p><a href="%s">%s</a></p>', esc_url( admin_url( '/admin.php?import=wcpbc_pricing_zone_csv' ) ), esc_html__( 'Try again', 'wc-price-based-country-pro' ) );
		echo '</div>';
	}

	/**
	 * Show result.
	 *
	 * @param int $imported Total rows imported.
	 */
	protected function show_result( $imported ) {
		$results   = array();
		$results[] = __( 'Import complete!', 'wc-price-based-country-pro' );
		$results[] = sprintf(
			/* translators: %d: products count */
			_n( '%s pricing zone imported.', '%s pricing zones imported.', $imported, 'wc-price-based-country-pro' ),
			'<strong>' . number_format_i18n( $imported ) . '</strong>'
		);
		$results[] = sprintf(
			/* translators: %s: File name */
			__( 'File uploaded: %s', 'wc-price-based-country-pro' ),
			'<strong>' . $this->params['file'] . '</strong>'
		);
		$results[] = sprintf( '<a href="%s">%s</a>', esc_url( admin_url( 'admin.php?page=wc-settings&tab=price-based-country' ) ), __( 'View pricing zones', 'wc-price-based-country-pro' ) );

		echo '<div class="notice notice-success">';
		echo '<p>' . wp_kses_post( implode( '</p><p>', $results ) ) . '</p>';
		echo '</div>';
	}
}
