<?php
/**
 * Handle integration with Affiliate WP.
 *
 * @version 2.9.0
 * @package WCPBC/Integrations
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WCPBC_Affiliate_WP' ) ) :

	/**
	 * WCPBC_Affiliate_WP Class
	 */
	class WCPBC_Affiliate_WP {

		/**
		 * Hook actions and filters
		 */
		public static function init() {
			add_filter( 'affwp_woocommerce_add_pending_referral_amount', array( __CLASS__, 'add_pending_referral_amount' ), 999, 2 );

			/**
			 * Compatibility with AffiliateWP - Recurring Referrals
			 *
			 * @see https://affiliatewp.com/addons/recurring-referrals/
			 */
			if ( class_exists( 'AffiliateWP_Recurring_Referrals' ) ) {
				add_filter( 'affwp_insert_pending_recurring_referral', array( __CLASS__, 'insert_pending_referral' ), 999 );
			}

			/**
			 * Compatibility with AffiliateWP MLM by Christian Freeman
			 *
			 * @see http://theperfectplugin.com/downloads/affiliatewp-mlm
			 */
			if ( class_exists( 'AffiliateWP_Multi_Level_Marketing' ) ) {
				add_filter( 'affwp_mlm_insert_pending_referral', array( __CLASS__, 'insert_pending_referral' ) );
				add_filter( 'affwp_mlm_indirect_referral_created', array( __CLASS__, 'mlm_indirect_referral_created' ), 0, 2 );
			}
		}

		/**
		 * Convert the amount to the AffilateWP currency using the exchange rate.
		 *
		 * @param float $amount   Calculated referral amount.
		 * @param int   $order_id Order ID (reference).
		 */
		public static function add_pending_referral_amount( $amount, $order_id ) {
			$order = wc_get_order( $order_id );
			if ( ! $order ) {
				return $amount;
			}
			$order_currency = $order->get_currency();
			$affwp_currency = affwp_get_currency();
			$base_currency  = wcpbc_get_base_currency();

			if ( $order_currency !== $affwp_currency ) {

				$base_amount        = WCPBC_Update_Exchange_Rates::convert(
					[
						'amount' => $amount,
						'from'   => $order_currency,
						'to'     => $affwp_currency,
					]
				);
				$format_amount      = affwp_format_amount( $amount ) . '&nbsp;' . $order_currency;
				$format_base_amount = affwp_format_amount( $base_amount ) . '&nbsp;' . $affwp_currency;
				$exchange_rate      = "1&nbsp;{$order_currency} = " . wc_format_localized_decimal( round( $base_amount / $amount, 6 ), false ) . '&nbsp;' . $affwp_currency;

				// Convert the amount to the base currency.
				$decimals = affwp_get_decimal_count();
				$amount   = round( $base_amount, $decimals );

				// Add a order note.
				$order->add_order_note(
					sprintf(
						// translators: 1:original amount, 2: converted amount, 3: exchange rate.
						__(
							'Price Based on Country: Referral amount converted from %1$s to %2$s (%3$s).',
							'wc-price-based-country-pro'
						),
						$format_amount,
						$format_base_amount,
						$exchange_rate
					)
				);
			}
			return $amount;
		}

		/**
		 * Convert the amount of the indirect referral/recurring referral of "AffiliateWP Multi Level Marketing" and "AffiliateWP Recurring Referrals".
		 *
		 * @param array $data Referral data.
		 */
		public static function insert_pending_referral( $data ) {
			$order_id = isset( $data['reference'] ) ? $data['reference'] : false;
			$amount   = isset( $data['amount'] ) ? $data['amount'] : false;
			if ( $order_id && $amount && is_callable( [ 'WCPBC_Affiliate_WP', 'add_pending_referral_amount' ] ) ) {
				$data['amount'] = self::add_pending_referral_amount( $amount, $order_id );
			}
			return $data;
		}

		/**
		 * Save the indirect referral of AffiliateWP Multi Level Marketing order comment with the correct amount.
		 *
		 * @param int   $referral_id Referral ID.
		 * @param array $data Referral data.
		 */
		public static function mlm_indirect_referral_created( $referral_id, $data ) {
			if ( ! ( isset( $data['context'] ) && 'woocommerce' === $data['context'] ) ) {
				return;
			}
			$referral = affiliate_wp()->referrals->get_by( 'referral_id', $referral_id );
			$amount   = affwp_currency_filter( affwp_format_amount( floatval( $referral->amount ) ) );
			$name     = affiliate_wp()->affiliates->get_affiliate_name( $data['affiliate_id'] );

			$last_comments = wc_get_order_notes(
				[
					'post_id'  => $data['reference'],
					'order_by' => 'comment_ID',
					'order'    => 'DESC',
					'number'   => 1,
				]
			);

			foreach ( $last_comments as $comment ) {
				wc_delete_order_note( $comment->id );
			}

			wc_create_order_note(
				$data['reference'],
				// translators: %d: referral ID. %s affiliate name.
				sprintf( __( 'Indirect Referral #%1$d for %2$s recorded for %3$s', 'affiliatewp-multi-level-marketing' ), $referral_id, $amount, $name ) // phpcs:ignore WordPress
			);
		}

		/**
		 * Checks the environment for compatibility problems.
		 *
		 * @return boolean
		 */
		public static function check_environment() {
			if ( defined( 'AFFILIATEWP_VERSION' ) && version_compare( AFFILIATEWP_VERSION, '2.4.4', '<' ) ) {
				add_action( 'admin_notices', array( __CLASS__, 'min_version_notice' ) );
				return false;
			}

			return true;
		}

		/**
		 * Display admin minimun version required
		 */
		public static function min_version_notice() {
			$affwp_version = defined( 'AFFILIATEWP_VERSION' ) ? AFFILIATEWP_VERSION : 'undefined';
			// translators: 1: HTML tag, 2: HTML tag, 3: AFFILIATEWP version.
			$notice = sprintf( __( '%1$sPrice Based on Country Pro & AffiliateWP%2$s compatibility requires AffiliateWP +2.4.4. You are running AffiliateWP %3$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', $affwp_version );
			echo '<div id="message" class="error"><p>' . wp_kses_post( $notice ) . '</p></div>';
		}
	}

	if ( WCPBC_Affiliate_WP::check_environment() ) {
		WCPBC_Affiliate_WP::init();
	}

endif;
