<?php
/**
 * WooCommerce Plugin API Wrapper Class.
 *
 * Handle WooCommerce Plugin API calls.
 *
 * @author oscargare
 * @version 1.1.2
 * @package WCPBC/Updater
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WC_Plugin_API_Wrapper' ) ) :

	/**
	 * WC_Plugin_API_Wrapper Class
	 */
	class WC_Plugin_API_Wrapper {

		/**
		 * Returns the API endpoint.
		 */
		private static function get_api_url() {

			if ( true !== apply_filters( 'wc_price_based_country_license_proxy_endpoint', wc_string_to_bool( WCPBC_Helper_Options::get( 'license_proxy_endpoint' ) ) ) ) {
				return 'https://www.pricebasedcountry.com/wc-api/plugin-apiv2/';
			} else {
				return 'https://apiproxy.pricebasedcountry.com/';
			}

		}

		/**
		 * Do a request to API.
		 *
		 * @param string $action API request action.
		 * @param array  $params Request parameters.
		 * @return array
		 */
		private static function api_request( $action, $params ) {

			$data     = false;
			$url      = add_query_arg( 'request', $action, self::get_api_url() );
			$response = wp_safe_remote_post(
				$url,
				[
					'body'    => $params,
					'timeout' => 20,
				]
			);

			if ( is_wp_error( $response ) ) {

				$data = new WP_Error(
					'-wcpbc-unexpected',
					$response->get_error_code() . ': ' . $response->get_error_message(),
					[ 'status' => 'error' ]
				);

			} else {

				$response = self::parse_response( $response );

				if ( self::is_captcha( $response ) && false === wc_string_to_bool( WCPBC_Helper_Options::get( 'license_proxy_endpoint', 'no' ) ) ) {

					// Antibot block. Try again using proxy endpoint.
					WCPBC_Helper_Options::update( 'license_proxy_endpoint', 'true' );

					return self::api_request( $action, $params );

				} elseif ( 200 === absint( $response->code ) ) {

					// Success.
					$data = self::json_decode( $response->body );

					if ( empty( $data ) ) {
						// Did the firewall block the request?
						$data = self::build_error( $response, $action );

					} elseif ( isset( $data->errors ) ) {
						try {
							// Unserialize as WP_Error.
							$data = maybe_unserialize( 'O:8:"WP_Error"' . substr( maybe_serialize( json_decode( $response->body, true ) ), 1 ) );
						} catch ( Exception $e ) {
							$data = new WP_Error( '-wcpbc-unexpected', $e->getMessage(), [ 'status' => 'error' ] );
						}
					}
				} else {
					// Unknow error.
					$data = self::build_error( $response, $action );
				}
			}

			if ( is_wp_error( $data ) ) {
				WCPBC_Debug_Logger::log_error( $data->get_error_message(), __METHOD__ );
			}

			return $data;
		}

		/**
		 * Parse the wp_safe_remote_post response.
		 *
		 * @param array $response Request response.
		 */
		private static function parse_response( $response ) {
			return (object) [
				'code' => isset( $response['response']['code'] ) ? absint( $response['response']['code'] ) : '-',
				'body' => isset( $response['body'] ) ? $response['body'] : '',
			];
		}

		/**
		 * Has the SG anti-bot blocked the request?
		 *
		 * @param array $response Request response.
		 */
		private static function is_captcha( $response ) {
			$pattern = '/<html.+<meta .+content="0;\/.well-known\/sgcaptcha\/.+meta>.+html>/';
			return in_array( $response->code, [ 200, 202 ], true ) && preg_match( $pattern, $response->body );
		}

		/**
		 * Decodes a JSON string. json_decode wrapper.
		 *
		 * @param string $json The json string being decoded.
		 * @param bool   $associative When TRUE, returned objects will be converted into associative arrays.
		 */
		private static function json_decode( $json, $associative = false ) {
			if ( is_numeric( $json ) ) {
				// json_decode function decodes number.
				return false;
			}
			return json_decode( $json, $associative );
		}

		/**
		 * Returns a WP_Error object from the response.
		 *
		 * @param array  $response Request response data.
		 * @param string $request Request name.
		 * @return WP_Error.
		 */
		private static function build_error( $response, $request ) {

			$code = $response->code;
			$body = $response->body;
			$body = 500 < absint( $code ) ? wp_strip_all_tags( $body ) : esc_html( $body );
			$ip   = is_callable( [ 'WC_Geolocation', 'get_external_ip_address' ] ) ? WC_Geolocation::get_external_ip_address() : '-';

			$data = new WP_Error(
				'-wcpbc-unexpected',
				sprintf( 'Unexpected response for "%1$s" request. Code: %2$s Body: %3$s External IP: %4$s', $request, $code, $body, $ip ),
				[ 'status' => 'error' ]
			);

			return $data;
		}

		/**
		 * Activate a license key
		 *
		 * @param sting $license_key License key.
		 * @return array
		 */
		public static function activate_license( $license_key ) {
			return self::api_request(
				'activate',
				array(
					'license'  => $license_key,
					'home_url' => get_bloginfo( 'url' ),
				)
			);
		}

		/**
		 * Deactivate a license key
		 *
		 * @param sting $license_key License key.
		 * @param sting $api_key API Key.
		 * @return array
		 */
		public static function deactivate_license( $license_key, $api_key ) {
			return self::api_request(
				'deactivate',
				array(
					'license' => $license_key,
					'api-key' => $api_key,
				)
			);
		}

		/**
		 * Check status of activation
		 *
		 * @param sting $license_key License key.
		 * @return array
		 */
		public static function status_check( $license_key ) {
			return self::api_request(
				'status_check',
				array(
					'license' => $license_key,
				)
			);
		}

		/**
		 * Update check
		 *
		 * @param int   $plugin_id Plugin ID.
		 * @param sting $api_key API Key.
		 * @param sting $license_key License key.
		 * @param sting $version The current plugin version.
		 * @return array
		 * @access public
		 */
		public static function update_check( $plugin_id, $api_key, $license_key, $version ) {
			return self::api_request(
				'update_check',
				array(
					'id'      => $plugin_id,
					'api-key' => $api_key,
					'license' => $license_key,
					'version' => $version,
				)
			);
		}

		/**
		 * Plugin information
		 *
		 * @param int   $plugin_id Plugin ID.
		 * @param sting $api_key API Key.
		 * @param sting $license_key License key.
		 * @return array
		 */
		public static function plugin_information( $plugin_id, $api_key, $license_key ) {
			return self::api_request(
				'plugin_information',
				array(
					'id'      => $plugin_id,
					'api-key' => $api_key,
					'license' => $license_key,
				)
			);
		}
	}

endif;
