<?php
/**
 * Handle license settings.
 *
 * @package WC_Plugin_API
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WC_Settings_API' ) ) {
	require_once dirname( WC_PLUGIN_FILE ) . '/includes/abstracts/abstract-wc-settings-api.php';
}

if ( class_exists( 'WCPBC_License_Settings' ) || ! class_exists( 'WC_Settings_API' ) ) {
	return;
}

/**
 * WCPBC_License_Settings Class.
 */
class WCPBC_License_Settings extends WC_Settings_API {

	/**
	 * The single instance of the class.
	 *
	 * @var WCPBC_License_Settings
	 */
	protected static $_instance = null; // phpcs:ignore

	/**
	 * License data.
	 *
	 * @var array
	 */
	protected $license_data;

	/**
	 * License status.
	 *
	 * @var string
	 */
	protected $status = '';

	/**
	 * License key.
	 *
	 * @var string
	 */
	protected $license_key = '';

	/**
	 * API key.
	 *
	 * @var string
	 */
	protected $api_key = '';

	/**
	 * Signature.
	 *
	 * @var string
	 */
	protected $signature = '';

	/**
	 * Constructor.
	 */
	public function __construct() {

		// The plugin ID. Used for option names.
		$this->plugin_id = 'wc_price_based_country_';

		// ID of the class extending the settings API. Used in option names.
		$this->id = 'license';

		// Init form fields.
		$this->init_form_fields();

		// Define user set variables.
		$this->license_key  = $this->get_option( 'license_key' );
		$this->api_key      = $this->get_option( 'api_key' );
		$this->signature    = $this->get_option( 'signature' );
		$this->license_data = wp_parse_args(
			$this->get_option( 'license_data' ),
			array(
				'status'                   => '',
				'expires'                  => '',
				'sites_active'             => 0,
				'sites_max'                => 0,
				'product_id'               => '',
				'renewal_period'           => '',
				'renewal_url'              => '',
				'renewal_percent_discount' => '',
				'timeout'                  => '0',
				'error_message'            => '',
			)
		);
	}

	/**
	 * Save the settings.
	 */
	public function save_settings() {
		$this->settings['license_key']  = $this->license_key;
		$this->settings['api_key']      = $this->api_key;
		$this->settings['signature']    = $this->signature;
		$this->settings['license_data'] = $this->license_data;

		return update_option( $this->get_option_key(), $this->settings );
	}

	/**
	 * Set license data.
	 *
	 * @param array $data license data to set.
	 */
	protected function set_license_data( $data ) {
		if ( ! is_array( $data ) ) {
			$data = (array) $data;
		}

		$data = wp_parse_args(
			$data,
			array(
				'status'                   => '',
				'expires'                  => '',
				'sites_active'             => 0,
				'sites_max'                => 0,
				'product_id'               => '',
				'renewal_period'           => '',
				'renewal_url'              => '',
				'renewal_percent_discount' => '',
				'timeout'                  => '0',
				'error_message'            => '',
			)
		);
		foreach ( $data as $prop => $value ) {
			if ( isset( $this->license_data[ $prop ] ) ) {
				$this->license_data[ $prop ] = wc_clean( $value );
			}
		}

		$this->license_data['timeout'] = empty( $this->license_data['status'] ) ? 0 : strtotime( '+3 hours', time() );
	}

	/**
	 * Retuns the signature for the current site.
	 *
	 * @param bool $skip_wpml Skip WPML filters or not.
	 */
	protected function get_signature( $skip_wpml = true ) {

		global $wpml_url_filters;
		static $cache = [];

		if ( empty( $this->api_key ) || empty( $this->license_key ) ) {
			return '';
		}

		$wpml_option = defined( 'ICL_SITEPRESS_VERSION' ) ? get_option( 'icl_sitepress_settings', [] ) : [];
		$has_wpml    = $skip_wpml && $wpml_url_filters && is_object( $wpml_url_filters ) && method_exists( $wpml_url_filters, 'home_url_filter' ) && isset( $wpml_option['language_negotiation_type'] ) && 2 === (int) $wpml_option['language_negotiation_type'];
		$index       = $has_wpml ? 1 : 0;

		if ( ! isset( $cache[ $index ] ) ) {

			if ( $has_wpml ) {
				remove_filter( 'home_url', [ $wpml_url_filters, 'home_url_filter' ], -10 );
			}

			$cache[ $index ] = get_bloginfo( 'url' );

			if ( $has_wpml ) {
				add_filter( 'home_url', [ $wpml_url_filters, 'home_url_filter' ], -10, 4 );
			}
		}

		return hash( 'crc32', $cache[ $index ] . $this->api_key . $this->license_key );
	}

	/**
	 * Validate License and API keys.
	 *
	 * @param bool $skip_wpml Skip WPML filters or not.
	 * @return boolean
	 */
	protected function validate( $skip_wpml = true ) {
		if ( 'active' !== $this->license_data['status'] || empty( $this->api_key ) || empty( $this->signature ) || empty( $this->license_key ) ) {
			return false;
		}

		return hash_equals( $this->signature, $this->get_signature( $skip_wpml ) );
	}

	/**
	 * Activate the license.
	 *
	 * @param string $license_key License key.
	 */
	protected function activate_license( $license_key ) {
		if ( empty( $license_key ) ) {
			return false;
		}

		$this->license_key = $license_key;
		$this->api_key     = '';
		$this->signature   = '';
		$this->set_license_data( array() );

		$response = WC_Plugin_API_Wrapper::activate_license( $license_key );

		if ( is_wp_error( $response ) ) {

			$this->set_license_data(
				array_merge(
					$response->get_error_data(),
					[ 'error_message' => $response->get_error_message() ]
				)
			);

			WC_Admin_Settings::add_error( __( 'Unable to activate the license.', 'wc-price-based-country-pro' ) );

		} else {

			// Activate the license.
			if ( ! empty( $response->api_key ) ) {

				$this->api_key   = $response->api_key;
				$this->signature = $this->get_signature();

				// Clear updater cache.
				WCPBC_Helper_Options::update( 'plugin_updater_cache', [] );

			} else {
				WC_Admin_Settings::add_error( __( 'Unable to activate the license. Check the status log file.', 'wc-price-based-country-pro' ) );
			}

			$license_data = isset( $response->license ) ? $response->license : array();

			$this->set_license_data( $license_data );
		}

		return $this->save_settings();
	}

	/**
	 * Deactivate the license.
	 */
	protected function deactivate_license() {
		$result   = false;
		$response = WC_Plugin_API_Wrapper::deactivate_license( $this->license_key, $this->api_key );

		$this->api_key   = '';
		$this->signature = '';

		if ( $response && ! is_wp_error( $response ) && isset( $response->data ) ) {
			$this->set_license_data( $response->data );
		}

		$result = $this->save_settings();

		return $result;
	}

	/**
	 * Check the license status.
	 */
	public function check_license_status() {

		if ( empty( $this->license_key ) || 'expired' === $this->license_data['status'] || time() < $this->license_data['timeout'] ) {
			return;
		}

		$response = WC_Plugin_API_Wrapper::status_check( $this->license_key );

		if ( is_wp_error( $response ) ) {

			if ( '404' === strval( $response->get_error_code() ) && count( $response->get_error_data() ) ) {
				// No license.
				$this->set_license_data( $response->get_error_data() );

			} else {
				// Inform user.
				$this->license_data['error_message'] = $response->get_error_message();

			}
		} elseif ( $response ) {
			// Ok.
			$this->set_license_data( $response );

			if ( isset( $response->is_connected ) && ! wc_string_to_bool( $response->is_connected ) ) {
				$this->api_key   = '';
				$this->signature = '';
			}
		}

		if ( 'active' !== $this->license_data['status'] ) {
			$this->api_key   = '';
			$this->signature = '';
		}

		$this->save_settings();
	}

	/**
	 * Unset the renewal period.
	 */
	public function unset_renewal_period() {
		$this->license_data['renewal_period'] = 'no';
		$this->save_settings();
	}

	/**
	 * License key getter.
	 *
	 * @return string
	 */
	public function get_license_key() {
		return $this->license_key;
	}

	/**
	 * API key getter.
	 *
	 * @return string
	 */
	public function get_api_key() {
		return $this->api_key;
	}

	/**
	 * License data getter.
	 *
	 * @return string
	 */
	public function get_license_data() {
		return $this->license_data;
	}

	/**
	 * License has been activated.
	 *
	 * @deprecated 3.4.10
	 * @return boolean
	 */
	public function is_license_active() {
		return $this->validate();
	}

	/**
	 * API key is valid for the current URL.
	 *
	 * @return boolean
	 */
	public function is_valid_key() {
		return $this->validate( false );
	}

	/**
	 * Initialize settings form fields.
	 */
	public function init_form_fields() {

		$this->form_fields = array(
			'status'      => array(
				'title' => __( 'License status', 'wc-price-based-country-pro' ),
				'type'  => 'status_info',
			),
			'toggle'      => array(
				'title' => __( 'Toggle activation', 'wc-price-based-country-pro' ),
				'type'  => 'toggle_activation',
			),
			'license_key' => array(
				'title'             => __( 'License Key', 'wc-price-based-country-pro' ),
				'type'              => 'license_key',
				'description'       => __( 'Enter your Price Based on Country Pro license key', 'wc-price-based-country-pro' ),
				'desc_tip'          => true,
				'default'           => '',
				'custom_attributes' => array( 'autocomplete' => 'off' ),
			),
		);
	}

	/**
	 * Processes and saves options.
	 *
	 * @return bool
	 */
	public function process_admin_options() {
		$this->init_settings();

		$fields      = $this->get_form_fields();
		$post_data   = $this->get_post_data();
		$license_key = $this->get_field_value( 'license_key', $fields['license_key'] );

		if ( ! $this->is_license_active() ) {
			// Activate the license.
			$result = $this->activate_license( $license_key );

		} elseif ( $this->is_license_active() && ! empty( $post_data['save'] ) && 'deactivate' === $post_data['save'] ) {
			// Deactivate the license.
			$result = $this->deactivate_license();

		}

		return $result;
	}

	/**
	 * Output the admin options table.
	 */
	public function admin_options() {

		$this->check_license_status();

		if ( $this->is_license_active() ) {
			$GLOBALS['hide_save_button'] = true; // phpcs:ignore
		}
		$this->display_errors();
		parent::admin_options();
	}

	/**
	 * Generate Status Info HTML.
	 *
	 * @param  mixed $key The field key.
	 * @param  mixed $data Field data.
	 * @since  1.0.0
	 * @return string
	 */
	public function generate_status_info_html( $key, $data ) {

		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title' => '',
		);

		$data  = wp_parse_args( $data, $defaults );
		$style = 'color:white; padding: 3px 6px; background:' . ( $this->is_license_active() ? 'green' : 'red' ) . ';';
		// translators: License status.
		$text = sprintf( __( 'you are %s receiving updates', 'wc-price-based-country-pro' ), $this->is_license_active() ? '' : sprintf( __( '%1$snot%2$s', 'wc-price-based-country-pro' ), '<strong>', '</strong>' ) );

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
					<span style="color:white; padding: 3px 6px; background:<?php echo ( $this->is_license_active() ? 'green' : 'red' ); ?>">
						<?php echo esc_html( $this->is_license_active() ? 'Active' : 'Inactive' ); ?>
					</span>
					&nbsp; - &nbsp; <?php echo wp_kses_post( $text ); ?>
				</fieldset>
			</td>
		</tr>
		<?php

		return ob_get_clean();
	}

	/**
	 * Generate Toggle Activation HTML.
	 *
	 * @param  mixed $key The field key.
	 * @param  mixed $data Field data.
	 * @return string
	 */
	public function generate_toggle_activation_html( $key, $data ) {
		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title' => '',
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();

		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
				<?php
				if ( $this->is_license_active() ) {
					echo '<button type="submit" class="button" name="save" value="deactivate">' . esc_html__( 'Deactivate License', 'wc-price-based-country-pro' ) . '</button>';
					echo '<p class="description">' . esc_html__( 'Deactivate your license so you can activate it  on another WooCommerce site', 'wc-price-based-country-pro' ) . '</p>';
				} else {
					esc_html_e( 'First add your Price Based on Country license key.', 'wc-price-based-country-pro' );
				}
				?>
				</fieldset>
			</td>
		</tr>
		<?php

		return ob_get_clean();
	}

	/**
	 * Generate License key input box
	 *
	 * @param  mixed $key The field key.
	 * @param  mixed $data Field data.
	 * @return string
	 */
	public function generate_license_key_html( $key, $data ) {

		$data['type'] = 'text';
		if ( $this->is_license_active() ) {
			$data['disabled']       = true;
			$this->settings[ $key ] = str_repeat( '*', 24 ) . substr( $this->settings[ $key ], -6 );
		}
		$text_html = $this->generate_text_html( $key, $data );

		if ( $this->is_license_active() && ! empty( $this->license_data['expires'] ) ) {
			// translators: Expire date.
			$text_html .= '<tr valign="top"><th colspan="2">' . sprintf( __( 'Your Price Based on Country Pro license will expire on %s', 'wc-price-based-country-pro' ), date_i18n( wc_date_format(), strtotime( $this->license_data['expires'] ) ) ) . '</td></tr>';
		}
		return $text_html;
	}

	/**
	 * Singelton implementation
	 *
	 * @return WCPBC_License_Settings
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Output option fields
	 */
	public static function output_fields() {
		self::instance()->admin_options();
	}

	/**
	 * Save option fields
	 */
	public static function save_fields() {
		self::instance()->process_admin_options();
	}

	/**
	 * Is license connected?
	 */
	public static function is_license_connected() {
		return self::instance()->is_license_active();
	}

	/**
	 * Upgrade license to version 3.5.0
	 */
	public function upgrade_350() {

		if ( version_compare( get_option( 'wc_price_based_country_pro_version', '1.0' ), '3.5.0', '>=' )
			|| ! empty( $this->signature )
			|| empty( $this->license_key )
			|| empty( $this->api_key )
		) {
			return;
		}

		$updater = new WCPBC_Plugin_Updater(
			WCPBC_PRO_PLUGIN_FILE,
			get_option( 'wc_price_based_country_pro_version', '1.0' ),
			array(
				'item_id'     => 1450,
				'license_key' => $this->license_key,
				'api_key'     => $this->api_key,
			)
		);

		remove_filter( 'pre_set_site_transient_update_plugins', array( $updater, 'transient_update_plugins' ), 25 );
		remove_filter( 'plugins_api', array( $updater, 'plugin_information' ), 10, 3 );

		$data = $updater->transient_update_plugins(
			(object) [
				'response'  => [],
				'no_update' => [],
			]
		);

		$plugin = plugin_basename( WCPBC_PRO_PLUGIN_FILE );

		if ( isset( $data->response[ $plugin ] ) || isset( $data->no_update[ $plugin ] ) ) {
			$this->signature = $this->get_signature();
			$this->save_settings();
		}
	}
}
