<?php
/**
 * Shortcodes.
 *
 * @version 2.4.10
 * @package WCPBC/Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WCPBC_Shortcodes' ) ) :

	/**
	 * Shortcodes Class
	 */
	class WCPBC_Shortcodes {

		/**
		 * Returns the shortcodes.
		 */
		private static function get_shortcodes() {
			return [
				'currency_switcher',
				'product_price',
				'content',
				'content_default',
				'convert',
				'country',
			];
		}

		/**
		 * Init shortcodes.
		 */
		public static function init() {

			foreach ( self::get_shortcodes() as $shortcode ) {
				add_shortcode( "wcpbc_{$shortcode}", array( __CLASS__, $shortcode ), 20 );
			}

			// AJAX refresh support.
			add_filter( 'wc_price_based_country_ajax_geolocation_convert_amount_content', array( __CLASS__, 'ajax_convert' ), 10, 2 );
			add_filter( 'wc_price_based_country_ajax_geolocation_country_name_content', array( __CLASS__, 'country' ) );
		}

		/**
		 * Output Currency Switcher
		 *
		 * @param array $atts Shortcode attributes.
		 * @return string
		 */
		public static function currency_switcher( $atts ) {

			$atts = shortcode_atts(
				array(
					'currency_display_style' => '[name] ([symbol])',
					'title'                  => '',
					'flag'                   => 0,
				),
				$atts,
				'wcpbc_currency_switcher'
			);

			$atts['currency_display_style'] = str_replace( array( '{', '}' ), array( '[', ']' ), $atts['currency_display_style'] );

			ob_start();

			the_widget( 'WCPBC_Widget_Currency_Switcher', $atts );

			return ob_get_clean();
		}

		/**
		 * Output price html for a pair product and zone
		 *
		 * @param array $atts Shortcode attributes.
		 */
		public static function product_price( $atts ) {
			$atts = shortcode_atts(
				array(
					'id'         => '',
					'zone'       => '',
					'product_id' => '',
					'zone_slug'  => '',
					'raw'        => '',
				),
				$atts,
				'wcpbc_product_price'
			);

			$price_html = '';
			$product_id = empty( $atts['id'] ) ? $atts['product_id'] : $atts['id'];
			$zone_slug  = empty( $atts['zone'] ) ? $atts['zone_slug'] : $atts['zone'];
			$_product   = wc_get_product( $product_id );

			if ( $_product ) {

				if ( ! $zone_slug ) {

					$price_html = $_product->get_price_html();

				} else {

					$zone = WCPBC_Pricing_Zones::get_zone( $zone_slug );
					if ( $zone && $zone->get_enabled() ) {

						$zone_slug = $zone->get_id();

						foreach ( [ '_regular_price', '_sale_price', '_price' ] as $key ) {
							$value = $zone->get_post_price( $_product->get_id(), $key );
							$_product->{"set{$key}"}( $value );
						}

						if ( empty( $atts['raw'] ) ) {

							WCPBC_Frontend_Currency::set_zone( $zone );

							add_filter( 'wc_price_based_country_ajax_geo_skip_wrapper', array( __CLASS__, 'return_true' ), 9999 );

							$price_html = $_product->get_price_html();

							remove_filter( 'wc_price_based_country_ajax_geo_skip_wrapper', array( __CLASS__, 'return_true' ), 9999 );

							WCPBC_Frontend_Currency::unset_zone();

						} else {
							$price_html = $_product->get_price();
						}
					} else {
						return '<span>' . __( 'Pricing zone not found.', 'wc-price-based-country-pro' ) . '<span>';
					}
				}
			} else {
				return '<span>' . __( 'Product not found.', 'wc-price-based-country-pro' ) . '<span>';
			}

			return $price_html;
		}

		/**
		 * Custom return true function. Make sure we remove our callback.
		 *
		 * @return bool
		 */
		public static function return_true() {
			return true;
		}


		/**
		 * Is the Geolcation AJAX enabled.
		 *
		 * @return bool
		 */
		private static function is_geolocation_ajax() {
			if ( is_cart() || is_account_page() || is_checkout() || is_customize_preview() ) {
				return false;
			}
			return WCPBC_Ajax_Geolocation::is_enabled();
		}

		/**
		 * Output the content filtered by zone
		 *
		 * @since 2.4.10
		 * @param array  $atts Shortcode attributes.
		 * @param string $content HTML content that comes between the shortcode tags.
		 * @return string HTML
		 */
		public static function content( $atts, $content = '' ) {
			$atts = shortcode_atts(
				array(
					'zone'        => '',
					'zone_slug'   => '',
					'wrapper_tag' => 'span',
				),
				$atts,
				'wcpbc_content'
			);

			$atts['zone'] = empty( $atts['zone'] ) ? $atts['zone_slug'] : $atts['zone']; // Backward compatibility.

			$zones  = array_map( 'trim', ( explode( ',', $atts['zone'] ) ) );
			$return = '';
			foreach ( $zones as $zone_slug ) {
				$return .= self::wrapper_content( $content, $zone_slug, $atts['wrapper_tag'] );
			}
			return $return;
		}

		/**
		 * Output the content for default zone
		 *
		 * @since 2.4.10
		 * @param array  $atts Shortcode attributes.
		 * @param string $content HTML content that comes between the shortcode tags.
		 * @return string HTML
		 */
		public static function content_default( $atts = array(), $content = '' ) {
			$atts = shortcode_atts(
				array(
					'wrapper_tag' => 'span',
				),
				$atts,
				'wcpbc_content'
			);

			return self::wrapper_content( $content, false, $atts['wrapper_tag'] );
		}

		/**
		 * Add wrapper to the content shortcode.
		 *
		 * @param string $content Shortcode content.
		 * @param string $zone_id Pricing zone ID.
		 * @param string $tag Wrapper HTML tag.
		 */
		private static function wrapper_content( $content, $zone_id, $tag ) {

			foreach ( self::get_shortcodes() as $shortcode ) {
				if ( has_shortcode( $content, "wcpbc_{$shortcode}" ) ) {
					// Translators: 1 shortcode name, 2, 3 HTML tags.
					$message = sprintf( __( 'Please do not use %1$s inside the %2$swcpbc_content%3$s shortcode!', 'wc-price-based-country-pro' ), "<code>[wcpbc_{$shortcode}]</code>", '<code>[', ']</code>' );
					return sprintf( '<span>%s</span>', wp_kses_post( $message ) );
				}
			}

			$return          = '';
			$current_zone_id = wcpbc_the_zone() ? wcpbc_the_zone()->get_id() : false;
			$html_attributes = array();

			if ( self::is_geolocation_ajax() ) {
				$return          = $content;
				$html_attributes = [
					'data-zone_id' => $zone_id,
					'class'        => 'wcpbc-content ' . ( $zone_id ? 'content-' . $zone_id : 'no-zone' ),
				];

			} elseif ( $current_zone_id === $zone_id ) {
				$return = $content;
			}

			if ( ! empty( $return ) ) {
				$return        = do_shortcode( self::sanitize_html( $return ) );
				$tag           = ! self::has_block_level_elements( "</{$tag}>" ) && self::has_block_level_elements( $return ) ? 'div' : $tag;
				$content_start = sprintf( '<%s %s>', esc_attr( $tag ), wc_implode_html_attributes( $html_attributes ) );
				$content_end   = sprintf( '</%s>', esc_attr( $tag ) );

				$return = $content_start . $return . $content_end;
			}

			return $return;
		}

		/**
		 * Sanitize HTML. When shortcode's content includes HTML, The shortcode gutenberg block wrappers the content with '</p><p>'.
		 *
		 * @param string $html String to sanitize.
		 */
		private static function sanitize_html( $html ) {
			if ( '</p>' === substr( $html, 0, 4 ) ) {
				$html = substr( $html, 4 );
			}

			if ( '<p>' === substr( $html, -3 ) ) {
				$html = substr( $html, 0, -3 );
			}
			return trim( $html );
		}

		/**
		 * Returns true if the string has block-level elements.
		 *
		 * @param string $html HTML to check.
		 * @return boolean
		 */
		private static function has_block_level_elements( $html ) {
			$block_level_elements = [
				'</address>',
				'</article>',
				'</aside>',
				'</blockquote>',
				'</canvas>',
				'</dd>',
				'</div>',
				'</dl>',
				'</dt>',
				'</fieldset>',
				'</figure>',
				'</footer>',
				'</form>',
				'</header>',
				'</h1>',
				'</h2>',
				'</h3>',
				'</h4>',
				'</h5>',
				'</h6>',
				'<hr>',
				'</li>',
				'</main>',
				'</nav>',
				'</ol>',
				'</p>',
				'</pre>',
				'</section>',
				'</table>',
				'</ul>',
				'</video>',
			];

			return str_replace( $block_level_elements, '', $html ) !== $html;
		}

		/**
		 * Return an amount convert by exchange rate.
		 *
		 * @since 2.5.0
		 * @param array $atts Shortcode attributes.
		 * @return string HTML
		 */
		public static function convert( $atts ) {
			$atts = shortcode_atts(
				array(
					'amount' => '',
				),
				$atts,
				'wcpbc_convert'
			);

			$output         = '';
			$amount         = wc_format_decimal( wp_kses_post( $atts['amount'] ) );
			$convert_amount = false;
			if ( $amount ) {
				$convert_amount = wcpbc_the_zone() ? wcpbc_the_zone()->get_exchange_rate_price( $amount ) : $amount;
				$output         = wc_price( $convert_amount );

				if ( self::is_geolocation_ajax() ) {
					$options       = wp_json_encode( array( 'amount' => $amount ) );
					$id            = md5( $options );
					$loading_class = defined( 'DOING_AJAX' ) && DOING_AJAX ? '' : ' wcpbc-price loading';
					$output        = '<span class="wc-price-based-country-refresh-area' . $loading_class . '" data-area="convert_amount" data-id="' . esc_attr( $id ) . '" data-options="' . esc_attr( $options ) . '">' . $output . '</span>';
				}
			}

			return $output;
		}

		/**
		 * Return the convert shortcode output. Use by AJAX geolocation.
		 *
		 * @param string $output The output.
		 * @param array  $options Refresh area options.
		 * @return string
		 */
		public static function ajax_convert( $output, $options ) {
			return self::convert( $options );
		}

		/**
		 * Returns the customer's country.
		 *
		 * @since 3.4.0
		 * @param string $output Only useful for add_filter callback.
		 * @return string HTML
		 */
		public static function country( $output = '' ) {
			$country      = 'Unknown';
			$country_code = wcpbc_get_woocommerce_country();
			$countries    = isset( wc()->countries ) && is_callable( [ wc()->countries, 'get_countries' ] ) ? wc()->countries->get_countries() : [];

			if ( isset( $countries[ $country_code ] ) ) {
				$country = $countries[ $country_code ];
			}

			if ( self::is_geolocation_ajax() ) {
				if ( ! doing_filter( 'wc_price_based_country_ajax_geolocation_country_name_content' ) ) {
					$country = sprintf( '<span style="display:none;">%s</span>', $country );
				}

				$country = sprintf( '<span class="wc-price-based-country-refresh-area" data-area="country_name" data-id="%s" data-options="">%s</span>', md5( __METHOD__ ), $country );
			}

			return $country;
		}
	}

endif;
