<?php
/**
 * Handle integration with WooCommerce Gift Cards by WooCommerce.
 *
 * @since 3.1.0
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WCPBC_Gift_Cards Class
 */
class WCPBC_Gift_Cards {

	/**
	 * Check enviroment notice.
	 *
	 * @var string
	 */
	private static $notice = '';

	/**
	 * Hook actions and filters
	 */
	public static function init() {
		add_action( 'init', [ __CLASS__, 'maybe_install' ], 100 );
		add_action( 'load-marketing_page_gc_giftcards', [ __CLASS__, 'admin_giftcard_page' ], 0 );
		add_action( 'load-marketing_page_gc_activity', [ __CLASS__, 'admin_activity_page' ], 0 );
		add_filter( 'manage_gc_giftcards_columns', [ __CLASS__, 'giftcards_columns' ] );
		add_filter( 'manage_gc_giftcards_activity_columns', [ __CLASS__, 'giftcards_activity_columns' ] );
		// Frontend actions.
		add_action( 'init', [ __CLASS__, 'convert_account_giftcards' ], 1 );
		add_filter( 'woocommerce_gc_create_order_giftcard_args', [ __CLASS__, 'create_order_giftcard_args' ], 10, 4 );
		add_action( 'woocommerce_after_calculate_totals', [ __CLASS__, 'after_calculate_totals' ], 0 );
		add_action( 'woocommerce_checkout_create_order', [ __CLASS__, 'checkout_create_order' ], 0 );
		add_action( 'woocommerce_store_api_checkout_update_order_meta', [ __CLASS__, 'rest_checkout_create_order' ], 0 );
		add_action( 'woocommerce_before_order_item_object_save', [ __CLASS__, 'save_issued_activity_meta' ] );
		add_action( 'woocommerce_gc_gift_card_debited', [ __CLASS__, 'update_activity_meta' ], 10, 2 );
		add_action( 'woocommerce_gc_gift_card_redeemed', [ __CLASS__, 'update_activity_meta' ], 10, 2 );
		add_action( 'woocommerce_account_giftcards_endpoint', [ __CLASS__, 'account_giftcards_endpoint' ] );
		// Email suppport.
		add_filter( 'woocommerce_gc_email_template_container_args', [ __CLASS__, 'email_template_container_args' ], 9999, 2 );

	}

	/**
	 * Check version and run the installer if necessary.
	 */
	public static function maybe_install() {

		$must_install = WCPBC_Helper_Options::get( 'wcpbc_gift_cards_version', false ) === false;
		$can_install  = ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX ) && ! defined( 'IFRAME_REQUEST' );
		if ( $must_install && $can_install ) {
			self::install();
		}
	}

	/**
	 * Install module.
	 */
	private static function install() {
		global $wpdb;
		$wpdb->hide_errors();
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		$schema = "CREATE TABLE {$wpdb->prefix}woocommerce_gc_pbc_activity_meta (
			`activity_id` BIGINT NOT NULL,
			`currency` VARCHAR(5) NOT NULL,
			`base_rate` double NOT NULL,
			PRIMARY KEY  (`activity_id`)
		) $collate;";

		dbDelta( $schema );

		// Update version.
		WCPBC_Helper_Options::update( 'wcpbc_gift_cards_version', WC_Product_Price_Based_Country_Pro::$version );
	}

	/**
	 * Run on admin Gift card page load.
	 */
	public static function admin_giftcard_page() {
		// Select section.
		$section = isset( $_GET['section'] ) ? wc_clean( wp_unslash( $_GET['section'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

		switch ( $section ) {
			case 'edit':
				self::admin_edit_giftcard_page();
				break;
			default:
				self::admin_list_table_giftcard_page();
				break;
		}
	}

	/**
	 * Run on admin edit gift card page.
	 */
	private static function admin_edit_giftcard_page() {

		$giftcard_id   = isset( $_GET['giftcard'] ) ? absint( $_GET['giftcard'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification
		$giftcard_meta = self::get_giftcard_meta( $giftcard_id );

		if ( ! $giftcard_meta ) {
			return;
		}

		$giftcard_currency   = $giftcard_meta['currency'];
		$balance_conversions = $giftcard_meta['conversions'];

		if ( $giftcard_currency && wcpbc_get_base_currency() !== $giftcard_currency ) {

			/**
			 * Set default currency to Giftcard currency.
			 */
			add_filter(
				'woocommerce_currency',
				function( $value ) use ( $giftcard_currency ) {
					return $giftcard_currency;
				},
				999
			);
		}

		/**
		 * Outputs the wcpbc_amount column.
		 */
		add_action(
			'manage_gc_giftcards_activity_custom_column',
			function( $column_name, $item ) use ( $giftcard_id ) {
				if ( 'wcpbc_amount' !== $column_name ) {
					return;
				}

				static $meta = false;

				if ( false === $meta ) {
					// Read meta.
					$meta = [];
					$rows = self::get_activity_meta( [ 'gc_id' => $giftcard_id ] );

					if ( $rows ) {
						$meta = wc_list_pluck( $rows, 'currency', 'activity_id' );
					}
				}

				// Output the currency.
				echo wp_kses_post(
					wc_price(
						floatval( $item['amount'] ),
						[
							'currency' => isset( $meta[ $item['id'] ] ) ? $meta[ $item['id'] ] : wcpbc_get_base_currency(),
						]
					)
				);

			},
			10,
			2
		);

		/**
		 * Outputs the currency conversions log.
		 */
		add_action(
			'marketing_page_gc_giftcards',
			function() use ( $balance_conversions ) {

				$rows = is_array( $balance_conversions ) && count( $balance_conversions ) ? $balance_conversions : false;

				if ( ! $rows ) {
					return;
				}

				include_once dirname( __FILE__ ) . '/class-wcpbc-gift-cards-list-table-conversion.php';

				echo '<div id="wcpbc-giftcards-currency-conversions-table" style="display:none;">';
				$list_table = new WCPBC_Gift_Cards_List_Table_Conversion( $rows );
				$list_table->display();
				echo '</div>';
			},
			20
		);

		add_action( 'admin_head', [ __CLASS__, 'admin_styles' ] );

		/**
		 * Enqueue JavaScript.
		 */
		wc_enqueue_js(
			"
			$('td.wcpbc_amount.column-wcpbc_amount').each(
				function(){
					$(this).closest('tr').find('td.amount.column-amount').html( $(this).html() );
				}
			);
			$('td.column-wcpbc_amount,th.column-wcpbc_amount').hide();
			$('#wcpbc-giftcards-currency-conversions-table').appendTo( $('#gift-card-data').closest('.postbox-container') ).show();
			"
		);
	}

	/**
	 * Run on admin list table gift card page.
	 */
	private static function admin_list_table_giftcard_page() {

		add_action(
			'manage_gc_giftcards_custom_column',
			function( $column_name, $item ) {
				if ( 'wcpbc_balance' !== $column_name ) {
					return;
				}

				$giftcard_meta = self::get_giftcard_meta( $item['id'] );

				// Output the currency.
				echo wp_kses_post(
					wc_price(
						floatval( $item['remaining'] ),
						[
							'currency' => $giftcard_meta['currency'],
						]
					)
				);
			},
			10,
			2
		);

		add_action( 'admin_head', [ __CLASS__, 'admin_styles' ] );

		/**
		 * Enqueue JavaScript.
		 */
		wc_enqueue_js(
			"
			$('td.wcpbc_balance.column-wcpbc_balance').each(
				function(){
					$(this).closest('tr').find('td.balance.column-balance').html( $(this).html() );
				}
			);
			"
		);
	}

	/**
	 * Run on admin Activity page load.
	 */
	public static function admin_activity_page() {

		/**
		 * Outputs the wcpbc_amount column.
		 */
		add_action(
			'manage_gc_giftcards_activity_custom_column',
			function( $column_name, $item ) {

				if ( 'wcpbc_amount' !== $column_name ) {
					return;
				}

				$rows = self::get_activity_meta( [ 'activity_id' => $item['id'] ] );

				if ( $rows ) {
					$meta = wc_list_pluck( $rows, 'currency', 'activity_id' );
				}

				// Output the currency.
				echo wp_kses_post(
					wc_price(
						floatval( $item['amount'] ),
						[
							'currency' => isset( $meta[ $item['id'] ] ) ? $meta[ $item['id'] ] : wcpbc_get_base_currency(),
						]
					)
				);

			},
			10,
			2
		);

		add_action( 'admin_head', [ __CLASS__, 'admin_styles' ] );

		/**
		 * Enqueue JavaScript.
		 */
		wc_enqueue_js(
			"
			$('td.wcpbc_amount.column-wcpbc_amount').each(
				function(){
					$(this).closest('tr').find('td.amount.column-amount').html( $(this).html() );
				}
			);
			"
		);
	}

	/**
	 * Add new colum to display the balance in the correct currency.
	 *
	 * @param array $columns Activity table columns.
	 */
	public static function giftcards_columns( $columns ) {
		$columns['wcpbc_balance'] = $columns['balance'];
		return $columns;
	}

	/**
	 * Add new colum to display the amount in the correct currency.
	 *
	 * @param array $columns Activity table columns.
	 */
	public static function giftcards_activity_columns( $columns ) {
		$columns['wcpbc_amount'] = $columns['amount'];
		return $columns;
	}

	/**
	 * Output styles for admin pages.
	 */
	public static function admin_styles() {
		?>
		<style>
			td.column-wcpbc_amount, th.column-wcpbc_amount, td.column-wcpbc_balance, th.column-wcpbc_balance {
				display: none;
			}
		</style>
		<?php
	}

	/*
	|--------------------------------------------------------------------------
	| Front-end actions.
	|--------------------------------------------------------------------------
	*/

	/**
	 * Adds the gift card metadata before add it to the DB.
	 *
	 * @param array         $args Key/value pair array.
	 * @param WC_Product    $product Gift card product.
	 * @param WC_Order_Item $order_item Order Item instance.
	 * @param WC_Order      $order Order instance.
	 */
	public static function create_order_giftcard_args( $args, $product, $order_item, $order ) {

		$args['meta_data']                      = isset( $args['meta_data'] ) && is_array( $args['meta_data'] ) ? $args['meta_data'] : [];
		$args['meta_data']['wcpbc_currency']    = $order->get_currency();
		$args['meta_data']['wcpbc_conversions'] = [];
		return $args;
	}

	/**
	 * Runs on before order item is save. Do into this hooks because the issue is added after order_giftcard_created.
	 *
	 * @param WC_Order_Item $order_item Order Item instance.
	 */
	public static function save_issued_activity_meta( $order_item ) {

		if ( ! did_action( 'woocommerce_gc_order_giftcard_created' ) ) {
			return;
		}

		$currency     = $order_item->get_order()->get_currency();
		$giftcard_ids = $order_item->get_meta( 'wc_gc_giftcards' );

		if ( empty( $giftcard_ids ) || ! is_array( $giftcard_ids ) ) {
			return;
		}

		foreach ( $giftcard_ids as $giftcard_id ) {

			$activities = WC_GC()->db->activity->query(
				[
					'return' => 'ids',
					'gc_id'  => $giftcard_id,
				]
			);

			if ( is_array( $activities ) ) {
				foreach ( $activities as $activity_id ) {
					self::save_activity_meta(
						[
							'activity_id' => $activity_id,
							'currency'    => $currency,
							'base_rate'   => WCPBC_Update_Exchange_Rates::get_rate( $currency, wcpbc_get_base_currency() ),
						]
					);
				}
			}
		}
	}

	/**
	 * Runs after session init. Converts the amount of account giftcards to current currency.
	 */
	public static function convert_account_giftcards() {
		if ( ! ( WC()->session && get_option( 'wc_gc_version', false ) ) ) {
			// Run only on frontend and GC plugin is installed.
			return;
		}

		$active_giftcards = WC_GC()->account->get_active_giftcards();

		foreach ( $active_giftcards as $giftcard_data ) {
			self::update_giftcard_balance( $giftcard_data );
		}

		$cache_key   = WC_Cache_Helper::get_transient_version( 'account_giftcards' ) . '_wc_gc_active_giftcards';
		$cached_data = [
			'active_giftcards' => $active_giftcards,
			'timestamp'        => time(),
		];
		WC()->session->set( $cache_key, $cached_data );
	}

	/**
	 * Runs after calculate totals. Updates gift cards balance to current currency.
	 *
	 * @param WC_Cart $cart Cart instance.
	 */
	public static function after_calculate_totals( $cart ) {
		// Updates applied giftcards.
		$cache_key         = WC_Cache_Helper::get_transient_version( 'applied_giftcards' ) . '_wc_gc_applied_giftcards';
		$applied_giftcards = WC()->session->get( $cache_key );

		if ( is_array( $applied_giftcards ) ) {

			foreach ( $applied_giftcards as $giftcard_data ) {
				self::update_giftcard_balance( $giftcard_data );
			}

			WC()->session->set( $cache_key, $applied_giftcards );
		}

	}

	/**
	 * Checkout block create order. Update gift cards amount.
	 *
	 * @param WC_Order $order The order instance.
	 */
	public static function rest_checkout_create_order( $order ) {
		if ( ! ( is_callable( [ 'WC_GC_Core_Compatibility', 'is_store_api_request' ] ) && WC_GC_Core_Compatibility::is_store_api_request( 'checkout', 'POST' ) ) ) {
			return;
		}

		self::checkout_create_order( $order );
	}

	/**
	 * Updates all gift cards amount before WC Gift cards add the order items.
	 *
	 * @param WC_Order $order The order instance.
	 */
	public static function checkout_create_order( $order ) {

		$giftcards = WC_GC()->giftcards->get();

		if ( empty( $giftcards ) || ! is_array( $giftcards ) ) {
			return;
		}

		foreach ( $giftcards as $giftcard_info ) {
			if ( ! isset( $giftcard_info['giftcard'] ) ) {
				continue;
			}

			self::update_giftcard_balance( $giftcard_info['giftcard'], true );
		}
	}

	/**
	 * Runs after gift card debit/credit. Updates the activity meta data.
	 *
	 * @param float           $amount Giftcard amount.
	 * @param WC_GC_Gift_Card $giftcard Giftcard.
	 */
	public static function update_activity_meta( $amount, $giftcard ) {

		$currency = $giftcard->data->get_meta( 'wcpbc_currency' );

		if ( ! $currency ) {
			return;
		}

		$activities = WC_GC()->db->activity->query(
			[
				'return'   => 'ids',
				'gc_id'    => $giftcard->get_id(),
				'order_by' => array( 'id' => 'DESC' ),
				'limit'    => 1,
			]
		);

		if ( is_array( $activities ) ) {
			foreach ( $activities as $activity_id ) {
				self::save_activity_meta(
					[
						'activity_id' => $activity_id,
						'currency'    => $currency,
						'base_rate'   => WCPBC_Update_Exchange_Rates::get_rate( $currency, wcpbc_get_base_currency() ),
					]
				);
			}
		}
	}

	/**
	 * Run on my account > giftcard page. Output the JavaScript to converts the activity amount to the current currency.
	 */
	public static function account_giftcards_endpoint() {
		// Get all redeemed gift cards by current user.
		$query_args   = [
			'return'      => 'ids',
			'redeemed_by' => get_current_user_id(),
		];
		$giftcard_ids = WC_GC()->db->giftcards->query( $query_args );
		if ( empty( $giftcard_ids ) ) {
			return;
		}

		$current_page = ! empty( $current_page ) ? absint( $current_page ) : 1;
		$per_page     = 5;
		$query_args   = [
			'return'   => 'objects',
			'type'     => array( 'redeemed', 'used', 'refunded', 'manually_refunded', 'refund_reversed' ),
			'gc_id'    => $giftcard_ids,
			'order_by' => [
				'date' => 'desc',
			],
			'limit'    => $per_page,
			'offset'   => ( $current_page - 1 ) * $per_page,
		];
		$activities   = WC_GC()->db->activity->query( $query_args );

		if ( empty( $activities ) || ! is_array( $activities ) ) {
			return;
		}

		$data = [];
		foreach ( $activities as $activity ) {
			$rows = self::get_activity_meta( [ 'activity_id' => $activity->get_id() ] );
			if ( $rows ) {
				$meta = wc_list_pluck( $rows, 'currency', 'activity_id' );
			}

			$data[] = [
				'activity_id' => $activity->get_id(),
				'amount'      => wc_price(
					floatval( $activity->get_amount() ),
					[
						'currency' => isset( $meta[ $activity->get_id() ] ) ? $meta[ $activity->get_id() ] : wcpbc_get_base_currency(),
					]
				),
			];

		}

		$json_var = wp_json_encode( $data );
		wc_enqueue_js(
			"
			var activity_data={$json_var};
			$('table.woocommerce-giftcards-activity-table>tbody>tr').each(function( index ) {
				$(this).find('td').eq(2).html(activity_data[index].amount);
			});
			"
		);
	}

	/**
	 * Run before get the email template. Update the currency.
	 *
	 * @param array           $template_args Template args.
	 * @param WC_GC_Gift_Card $giftcard WC_GC_Gift_Card instance.
	 * @return array
	 */
	public static function email_template_container_args( $template_args, $giftcard ) {
		$args = array(
			'order' => $giftcard->get_order_id(),
		);
		WCPBC_Frontend_Currency::email_order_zone_data( $args );

		return $template_args;
	}

	/**
	 * Checks the environment for compatibility problems.
	 *
	 * @return boolean
	 */
	public static function check_environment() {
		$plugin_version = function_exists( 'WC_GC' ) && is_callable( [ WC_GC(), 'get_plugin_version' ] ) ? WC_GC()->get_plugin_version() : 'unknown';
		$min_version    = '1.15.1';

		if ( 'unknown' === $plugin_version || version_compare( $plugin_version, $min_version, '<' ) ) {
			// translators: 1: HTML tag, 2: HTML tag, 3: Product Bundles version.
			self::$notice = sprintf( __( '%1$sPrice Based on Country Pro & WooCommerce Gift Cards%2$s compatibility requires WooCommerce Gift Cards +%4$s. You are running "WooCommerce Gift Cards" %3$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', $plugin_version, $min_version );
			add_action( 'admin_notices', array( __CLASS__, 'min_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Display admin minimun version required
	 */
	public static function min_version_notice() {
		echo '<div id="message" class="error"><p>' . wp_kses_post( self::$notice ) . '</p></div>';
	}

	/*
	|--------------------------------------------------------------------------
	| Utils
	|--------------------------------------------------------------------------
	*/

	/**
	 * Returns the giftcard meta data. Includes the current balance amount.
	 *
	 * @param int $gc_id The gift card ID.
	 */
	private static function get_giftcard_meta( $gc_id ) {
		$data     = false;
		$giftcard = WC_GC()->db->giftcards->get( $gc_id );

		if ( ! $giftcard ) {
			return $data;
		}

		$data = [
			'balance'         => $giftcard->get_balance(),
			'initial_balance' => $giftcard->get_initial_balance(),
			'currency'        => wcpbc_get_base_currency(),
			'conversions'     => [],
		];

		if ( $giftcard->get_meta( 'wcpbc_currency' ) ) {
			$data['currency']    = $giftcard->get_meta( 'wcpbc_currency' );
			$data['conversions'] = $giftcard->get_meta( 'wcpbc_conversions' );
		} else {
			// Before plugin install. Get currency from the order.
			$order = wc_get_order( $giftcard->get_order_id() );
			if ( $order ) {
				$data['currency'] = $order->get_currency();
			}
		}

		return $data;
	}

	/**
	 * Updates the amount of a WC_GC_Gift_Card_Data.
	 *
	 * @param WC_GC_Gift_Card_Data $giftcard_data Gift card data object.
	 * @param bool                 $persist Persits data on database.
	 */
	private static function update_giftcard_balance( &$giftcard_data, $persist = false ) {

		if ( ! is_a( $giftcard_data, 'WC_GC_Gift_Card_Data' ) ) {
			return;
		}

		$data = self::get_giftcard_meta( $giftcard_data->get_id() );

		if ( ! $data ) {
			return;
		}

		$changed         = false;
		$balance         = $data['balance'];
		$initial_balance = $data['initial_balance'];

		if ( get_woocommerce_currency() !== $data['currency'] ) {

			$balance = WCPBC_Update_Exchange_Rates::convert(
				[
					'amount' => $balance,
					'from'   => $data['currency'],
					'to'     => get_woocommerce_currency(),
				]
			);

			$initial_balance = WCPBC_Update_Exchange_Rates::convert(
				[
					'amount' => $initial_balance,
					'from'   => $data['currency'],
					'to'     => get_woocommerce_currency(),
				]
			);

			$changed = true;
		}

		$giftcard_data->set_balance( $balance );
		$giftcard_data->set_initial_balance( $initial_balance );

		if ( $persist && $changed ) {

			$data['conversions']   = is_array( $data['conversions'] ) ? $data['conversions'] : [];
			$data['conversions'][] = [
				'from' => [
					'amount'   => $data['balance'],
					'currency' => $data['currency'],
				],
				'to'   => [
					'amount'   => $balance,
					'currency' => get_woocommerce_currency(),
				],
				'date' => time(),
			];
			$data['currency']      = get_woocommerce_currency();

			$giftcard_data->update_meta( 'wcpbc_currency', $data['currency'] );
			$giftcard_data->update_meta( 'wcpbc_conversions', $data['conversions'] );

			$giftcard_data->save();
		}
	}

	/**
	 * Returns the activity metadata from DB.
	 *
	 * @param array $args Array of arguments.
	 */
	private static function get_activity_meta( $args ) {
		global $wpdb;

		$data = false;
		$args = wp_parse_args(
			$args,
			[
				'gc_id'       => 0,
				'activity_id' => 0,
			]
		);

		$where  = [];
		$params = [];

		if ( $args['gc_id'] ) {
			$where[]  = 'activity.gc_id = %d';
			$params[] = absint( $args['gc_id'] );
		}

		if ( $args['activity_id'] ) {
			$where[]  = 'activity.id = %d';
			$params[] = absint( $args['activity_id'] );
		}

		if ( count( $where ) ) {
			$data = $wpdb->get_results(
				$wpdb->prepare(
					'SELECT activity.id AS activity_id, activity.object_id, activity_meta.base_rate, activity_meta.currency ' .
					"FROM {$wpdb->prefix}woocommerce_gc_activity activity LEFT JOIN {$wpdb->prefix}woocommerce_gc_pbc_activity_meta AS activity_meta ON  activity_meta.activity_id = activity.id " .
					'WHERE ' . implode( ' AND ', $where ), // @phpcs:ignore
					$params
				)
			);

			foreach ( $data as $row ) {
				if ( $row->currency ) {
					continue;
				}
				// Get currency from order.
				$order         = wc_get_order( $row->object_id );
				$row->currency = $order ? $order->get_currency() : wcpbc_get_base_currency();
			}
		}

		return $data;
	}

	/**
	 * Saves activity meta.
	 *
	 * @param array $args Key/value pair array.
	 */
	private static function save_activity_meta( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			[
				'activity_id' => 0,
				'currency'    => '',
				'base_rate'   => 1,
			]
		);

		if ( ! absint( $args['activity_id'] ) > 0 ) {
			return;
		}

		$row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}woocommerce_gc_pbc_activity_meta WHERE activity_id = %d LIMIT 1;", absint( $args['activity_id'] ) ) );

		if ( ! $row ) {

			// Insert.
			$wpdb->insert(
				$wpdb->prefix . 'woocommerce_gc_pbc_activity_meta',
				[
					'activity_id' => absint( $args['activity_id'] ),
					'currency'    => $args['currency'],
					'base_rate'   => floatval( $args['base_rate'] ),
				],
				[ '%d', '%s', '%s' ]
			);
		}
	}
}

if ( WCPBC_Gift_Cards::check_environment() ) {
	WCPBC_Gift_Cards::init();
}
