<?php
/**
 * Currency conversions admin table.
 *
 * @since 3.1.0
 * @package WCPBC
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * WCPBC_Gift_Cards_List_Table_Conversion Class
 */
class WCPBC_Gift_Cards_List_Table_Conversion extends WP_List_Table {

	/**
	 * Constructor
	 *
	 * @param array $rows Array that contains the currency conversions data. Fecth from order item meta data.
	 */
	public function __construct( $rows ) {
		$this->prepare_items( $rows );
		parent::__construct(
			[
				'singular' => 'currency conversion',
				'plural'   => 'currency conversions',
			]
		);
	}

	/**
	 * Overrides the tablenav.
	 *
	 * @since 3.1.0
	 * @param string $which Top or bottom.
	 */
	protected function display_tablenav( $which ) {
	}

	/**
	 * Prepare items.
	 *
	 * @param array $rows Array of currency conversions.
	 */
	public function prepare_items( $rows = [] ) {
		$this->_column_headers = [ $this->get_columns(), [], [] ];
		$this->items           = [];

		if ( ! is_array( $rows ) ) {
			return;
		}

		foreach ( $rows as $row ) {
			if ( ! isset( $row['from'], $row['to'], $row['date'] ) ) {
				continue;
			}
			$this->items[] = $row;
		}

		usort(
			$this->items,
			function( $a, $b ) {
				return $a['date'] < $b['date'] ? 1 : -1;
			}
		);
	}

	/**
	 * Get a list of columns. The format is: 'internal-name' => 'Title'
	 */
	public function get_columns() {
		return [
			'date'   => _x( 'Date', 'Gift card currency conversions table', 'wc-price-based-country-pro' ),
			'amount' => _x( 'Balance conversion', 'Gift card currency conversions table', 'wc-price-based-country-pro' ),
		];
	}

	/**
	 * Outputs date column.
	 *
	 * @param array $item Key/value pair array.
	 */
	public function column_date( $item ) {

		$t_time    = WC_GC_Core_Compatibility::wp_date( _x( 'Y/m/d g:i:s a', 'list table date hover format', 'wc-price-based-country-pro' ), $item['date'] );
		$time_diff = time() - $item['date'];

		if ( $time_diff > 0 && $time_diff < DAY_IN_SECONDS ) {
			/* translators: %s: human time diff */
			$h_time = sprintf( esc_html__( '%s ago', 'wc-price-based-country-pro' ), human_time_diff( $item['date'] ) );
		} else {
			$h_time = WC_GC_Core_Compatibility::wp_date( _x( 'Y/m/d', 'list table date format', 'wc-price-based-country-pro' ), $item['date'] );
		}

		echo '<span title="' . esc_attr( $t_time ) . '">' . esc_html( $h_time ) . '</span>';
	}

	/**
	 * Outputs amount column.
	 *
	 * @param array $item Key/value pair array.
	 */
	public function column_amount( $item ) {
		// Output the currency.
		echo wp_kses_post(
			wc_price(
				floatval( $item['from']['amount'] ),
				[
					'currency' => $item['from']['currency'],
				]
			)
		);

		echo '<span>&nbsp;→&nbsp;</span>';

		echo wp_kses_post(
			wc_price(
				floatval( $item['to']['amount'] ),
				[
					'currency' => $item['to']['currency'],
				]
			)
		);
	}

}
