<?php
/**
 * Product Bundles sync prices.
 *
 * @since 2.22.0
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WCPBC_Product_Bundles_Sync_Prices Class
 */
class WCPBC_Product_Bundles_Sync_Prices {

	/**
	 * Product bundle.
	 *
	 * @var WC_Product_Bundle
	 */
	protected $bundle;

	/**
	 * Pricing zone.
	 *
	 * @var WCPBC_Pricing_Zone
	 */
	protected $zone;

	/**
	 * Price properties.
	 *
	 * @var array
	 */
	protected $prices;

	/**
	 * Constructor
	 *
	 * @param WC_Product_Bundle  $bundle Product bundle.
	 * @param WCPBC_Pricing_Zone $zone Pricing zone.
	 */
	public function __construct( $bundle, $zone ) {
		$this->bundle = $bundle;
		$this->zone   = $zone;
		$this->prices = array();
	}

	/**
	 * Returns the raw prices.
	 */
	public function get_raw_prices() {
		return $this->prices;
	}

	/**
	 * Sync product bundle raw price meta.
	 */
	public function sync_raw_prices() {

		$min_raw_price         = $this->zone->get_post_price( $this->bundle->get_id(), '_wc_pb_base_price' );
		$min_raw_regular_price = $this->zone->get_post_price( $this->bundle->get_id(), '_wc_pb_base_regular_price' );

		if ( $this->bundle->contains( 'priced_individually' ) ) {
			$min_raw_price         = floatval( $min_raw_price );
			$min_raw_regular_price = floatval( $min_raw_regular_price );
		}

		$max_raw_price         = $min_raw_price;
		$max_raw_regular_price = $min_raw_regular_price;

		if ( $this->bundle->is_nyp() ) {
			$max_raw_price         = INF;
			$max_raw_regular_price = INF;
		}

		$bundled_items = $this->get_bundled_items();

		foreach ( $bundled_items as $bundled_item ) {

			$min_raw_price         += $bundled_item->min_quantity * floatval( $bundled_item->min_price );
			$min_raw_regular_price += $bundled_item->min_quantity * floatval( $bundled_item->min_regular_price );

			if ( '' === $bundled_item->max_quantity ) {
				$max_raw_price         = INF;
				$max_raw_regular_price = INF;
			}

			$item_max_raw_price         = INF !== $bundled_item->max_price ? floatval( $bundled_item->max_price ) : INF;
			$item_max_raw_regular_price = INF !== $bundled_item->max_regular_price ? floatval( $bundled_item->max_regular_price ) : INF;

			if ( INF !== $max_raw_price ) {
				if ( INF !== $item_max_raw_price ) {
					$max_raw_price         += $bundled_item->max_quantity * $item_max_raw_price;
					$max_raw_regular_price += $bundled_item->max_quantity * $item_max_raw_regular_price;
				} else {
					$max_raw_price         = INF;
					$max_raw_regular_price = INF;
				}
			}
		}

		// Calculate the min bundled item price and use it when the active group mode requires a child selection.
		if ( false === WC_Product_Bundle::group_mode_has( $this->bundle->get_group_mode( 'edit' ), 'parent_item' ) && ! $this->bundle->contains( 'mandatory' ) ) {

			$min_item_price = null;

			foreach ( $bundled_items as $bundled_item ) {

				$min_quantity = max( $bundled_item->quantity, 1 );

				if ( is_null( $min_item_price ) || $min_quantity * floatval( $bundled_item->min_price ) < $min_item_price ) {
					$min_item_price = $min_quantity * floatval( $bundled_item->min_price );
				}
			}

			if ( $min_item_price > 0 ) {
				$min_raw_price = $min_item_price;
			}
		}

		$this->prices = array(
			'min_raw_price'         => $min_raw_price,
			'min_raw_regular_price' => $min_raw_regular_price,
			'max_raw_price'         => $max_raw_price,
			'max_raw_regular_price' => $max_raw_regular_price,
		);

		$this->save_price_props();
	}

	/**
	 * Save the price properties.
	 */
	protected function save_price_props() {

		if ( 'bundle' !== $this->bundle->get_data_store_type() ) {
			// don't save properties if the product is a virtual bundle.
			return;
		}

		/*
		 * Properties to metakey.
		 */
		$props_to_metakeys = array(
			'min_raw_price'         => '_price',
			'min_raw_regular_price' => '_regular_price',
			'max_raw_price'         => '_wc_sw_max_price',
			'max_raw_regular_price' => '_wc_sw_max_regular_price',
		);

		foreach ( $props_to_metakeys as $prop => $metakey ) {
			$this->zone->set_postmeta(
				$this->bundle->get_id(),
				$metakey,
				wc_format_decimal( $this->prices[ $prop ] )
			);
		}

		// Update on sale price.
		if ( $this->bundle->contains( 'discounted_mandatory' ) && $this->prices['min_raw_regular_price'] > 0 ) {

			$this->zone->set_postmeta(
				$this->bundle->get_id(),
				'_sale_price',
				wc_format_decimal( $this->prices['min_raw_price'] )
			);
		} else {

			$this->zone->set_postmeta(
				$this->bundle->get_id(),
				'_sale_price',
				''
			);
		}
	}

	/**
	 * Returns the bundles items with the correct prices.
	 */
	protected function get_bundled_items() {

		$bundled_items = array();

		foreach ( $this->bundle->get_bundled_items( 'edit' ) as $_bundled_item ) {

			if ( ! $_bundled_item->is_priced_individually() ) {
				continue;
			}

			$item = $this->sync_bundle_item_prices( $_bundled_item );

			if ( $item ) {

				// Clear the PB cache.
				if ( is_callable( array( 'WC_PB_Helpers', 'cache_delete' ) ) ) {
					WC_PB_Helpers::cache_delete( 'min_price_quantities_' . $this->bundle->get_id() );
					WC_PB_Helpers::cache_delete( 'max_price_quantities_' . $this->bundle->get_id() );
				}

				$item->min_quantity = $_bundled_item->get_quantity(
					'min',
					array(
						'context'        => 'price',
						'check_optional' => true,
					)
				);
				$item->max_quantity = $_bundled_item->get_quantity(
					'max',
					array(
						'context' => 'price',
					)
				);

				$item->quantity = $_bundled_item->get_quantity( 'min' );

				$bundled_items[ $_bundled_item->get_id() ] = $item;
			}
		}

		return $bundled_items;
	}

	/**
	 * Sync price data.
	 *
	 * @param WC_Bundled_Item $bundled_item Bundled item.
	 * @return stdClass $item A object with the price properties for the current pricing zone.
	 */
	public function sync_bundle_item_prices( &$bundled_item ) {

		$item = false;

		$bundled_product = isset( $bundled_item->product ) ? $bundled_item->product : false;

		if ( ! is_callable( array( $bundled_product, 'get_type' ) ) ) {
			return;
		}

		if ( in_array( $bundled_product->get_type(), array( 'simple', 'subscription' ), true ) ) {

			$bundled_product = clone $bundled_item->product;

			$this->adjust_product_price( $bundled_product );
		}

		if ( 'subscription' === $bundled_product->get_type() ) {

			/**
			 * Simple subscriptions.
			 */
			$item = $this->sync_bundle_item_subscription( $bundled_item, $bundled_product );

		} elseif ( 'simple' === $bundled_product->get_type() ) {

			/**
			 * Simple products.
			 */
			$item = $this->sync_bundle_item_simple( $bundled_item, $bundled_product );

			// Name your price support.
			if ( $bundled_item->is_priced_individually() && $this->is_nyp( $bundled_product ) ) {
				$max_nyp_price           = WC_Name_Your_Price_Helpers::get_maximum_price( $bundled_product );
				$item->max_regular_price = $max_nyp_price ? $max_nyp_price : INF;
				$item->max_price         = $max_nyp_price ? $max_nyp_price : INF;
			}
		} elseif ( in_array( $bundled_product->get_type(), array( 'variable', 'variable-subscription' ), true ) ) {

			/**
			 * Variable products.
			 */

			$min_variation = false;
			$max_variation = false;

			/*
			 * Find the the variations with the min & max price.
			 */

			$variations_prices      = $this->get_variation_prices( $bundled_product );
			$variation_prices_array = $variations_prices['prices'];
			$discount               = $bundled_item->get_discount( 'sync' );

			if ( ! empty( $discount ) && false === $bundled_item->is_discount_allowed_on_sale_price() ) {
				$variation_prices = $variation_prices_array['regular_price'];
			} else {
				$variation_prices = $variation_prices_array['price'];
			}

			// Clean filtered-out variations.
			if ( $bundled_item->has_filtered_variations() ) {
				$variation_prices = array_intersect_key( $variation_prices, array_flip( $bundled_item->get_filtered_variations() ) );
			}

			asort( $variation_prices, SORT_NUMERIC );

			$variation_price_ids = array_keys( $variation_prices );

			$min_variation_price_id = current( $variation_price_ids );
			$max_variation_price_id = end( $variation_price_ids );

			$min_variation = $variations_prices['variations'][ $min_variation_price_id ];
			$max_variation = $variations_prices['variations'][ $max_variation_price_id ];

			if ( is_a( $min_variation, 'WC_Product' ) && is_a( $max_variation, 'WC_Product' ) ) {

				if ( 'variable-subscription' === $bundled_product->get_type() ) {

					$item = $this->sync_bundle_item_subscription( $bundled_item, $min_variation );

				} else {

					$item = $this->sync_bundle_item_simple( $bundled_item, $min_variation, $max_variation );

					// Name your price support.
					if ( $bundled_item->is_priced_individually() && $this->is_nyp( $bundled_product ) ) {
						// There is no performant way to search for the max NYP price of a variation. NYP does not filter lookup table data.
						$item->max_regular_price = INF;
						$item->max_price         = INF;
					}
				}

				// Set the min price product.
				$item->min_price_product = $min_variation;
			}
		}

		return $item;
	}

	/**
	 * Set the product price to the pricing zone price.
	 *
	 * @param WC_Product $product Product instance.
	 */
	protected function adjust_product_price( &$product ) {
		foreach ( array( '_price', '_regular_price', '_sale_price' ) as $meta_key ) {
			$setter = 'set' . $meta_key;
			$value  = $this->zone->get_post_price( $product->get_id(), $meta_key );

			// Force change on the prices properties updating it with a ridiculous value.
			$product->{$setter}( -9999 );

			// Set the real price.
			$product->{$setter}( $value );
		}
	}

	/**
	 * Sync the item with the simple product prices.
	 *
	 * @param WC_Bundled_Item $bundled_item Bundled item.
	 * @param WC_Product      $min_price_product Product object.
	 * @param WC_Product      $max_price_product Product object.
	 */
	protected function sync_bundle_item_simple( $bundled_item, $min_price_product, $max_price_product = false ) {
		$item = new stdClass();

		$item->min_price   = $bundled_item->get_raw_price( $min_price_product, 'sync' );
		$min_regular_price = $bundled_item->get_raw_regular_price( $min_price_product );

		if ( $max_price_product ) {
			$item->max_price   = $bundled_item->get_raw_price( $max_price_product, 'sync' );
			$max_regular_price = $bundled_item->get_raw_regular_price( $max_price_product );
		} else {
			$item->max_price   = $item->min_price;
			$max_regular_price = $min_regular_price;
		}

		$item->min_regular_price = min( $min_regular_price, $max_regular_price );
		$item->max_regular_price = max( $min_regular_price, $max_regular_price );

		return $item;
	}

	/**
	 * Sync the item with the subscription prices.
	 *
	 * @param WC_Bundled_Item $bundled_item Bundled item.
	 * @param WC_Product      $bundled_product Product object.
	 */
	protected function sync_bundle_item_subscription( $bundled_item, $bundled_product ) {
		$item = new stdClass();

		$regular_recurring_fee = $bundled_item->get_raw_regular_price( $bundled_product );
		$recurring_fee         = $bundled_item->get_raw_price( $bundled_product, 'sync' );

		// Recurring prices.
		$item->min_regular_recurring_price = $regular_recurring_fee;
		$item->max_regular_recurring_price = $regular_recurring_fee;
		$item->min_recurring_price         = $recurring_fee;
		$item->max_recurring_price         = $recurring_fee;

		// Sign up price.
		$signup_fee   = $this->zone->get_post_price( $bundled_product->get_id(), '_subscription_sign_up_fee' );
		$trial_length = WC_Subscriptions_Product::get_trial_length( $bundled_product );

		// Up-front price.
		$up_front_fee         = $trial_length > 0 ? $signup_fee : floatval( $signup_fee ) + floatval( $recurring_fee );
		$regular_up_front_fee = $trial_length > 0 ? $signup_fee : floatval( $signup_fee ) + floatval( $regular_recurring_fee );

		$item->min_regular_price = $regular_up_front_fee;
		$item->max_regular_price = $regular_up_front_fee;
		$item->min_price         = $up_front_fee;
		$item->max_price         = $up_front_fee;

		return $item;
	}

	/**
	 * Check if the product is Name Your Price.
	 *
	 * @param WC_Product $product Product instance.
	 */
	protected function is_nyp( $product ) {
		$is_nyp = false;

		if ( is_callable( array( 'WC_Name_Your_Price_Helpers', 'is_nyp' ) )
			&& is_callable( array( 'WC_Name_Your_Price_Helpers', 'has_nyp' ) )
			&& is_callable( array( 'WC_Name_Your_Price_Helpers', 'get_maximum_price' ) )
		) {

			$is_nyp = WC_Name_Your_Price_Helpers::is_nyp( $product ) || WC_Name_Your_Price_Helpers::has_nyp( $product );
		}

		return $is_nyp;
	}

	/**
	 * Return the variation prices array.
	 *
	 * @param WC_Product_Variable $product Product instance.
	 */
	protected function get_variation_prices( $product ) {
		$prices_array  = array(
			'price'         => array(),
			'regular_price' => array(),
			'sale_price'    => array(),
		);
		$variation_ids = $product->get_visible_children();
		$variations    = array();

		foreach ( $variation_ids as $variation_id ) {

			$variation = wc_get_product( $variation_id );

			if ( $variation ) {
				$price         = $this->zone->get_post_price( $variation_id, '_price' );
				$regular_price = $this->zone->get_post_price( $variation_id, '_regular_price' );
				$sale_price    = $this->zone->get_post_price( $variation_id, '_sale_price' );

				// Skip empty prices.
				if ( '' === $price ) {
					continue;
				}

				// If sale price does not equal price, the product is not yet on sale.
				if ( $sale_price === $regular_price || $sale_price !== $price ) {
					$sale_price = $regular_price;
				}

				$prices_array['price'][ $variation_id ]         = wc_format_decimal( $price, wc_get_price_decimals() );
				$prices_array['regular_price'][ $variation_id ] = wc_format_decimal( $regular_price, wc_get_price_decimals() );
				$prices_array['sale_price'][ $variation_id ]    = wc_format_decimal( $sale_price, wc_get_price_decimals() );

				$prices_array = apply_filters( 'woocommerce_variation_prices_array', $prices_array, $variation, false ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals

				$variation->set_price( $price );
				$variation->set_regular_price( $regular_price );
				$variation->set_sale_price( $sale_price );

				$variations[ $variation_id ] = $variation;
			}
		}

		return array(
			'prices'     => $prices_array,
			'variations' => $variations,
		);
	}
}
