<?php
/**
 * WooCommerce Price Based on Country Pro Add-on setup
 *
 * @package WCPBC
 * @version 2.6.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_Product_Price_Based_Country_Pro Class
 */
class WC_Product_Price_Based_Country_Pro {

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	public static $version = '3.6.3';

	/**
	 * Admin notices.
	 *
	 * @var array
	 */
	private static $admin_notices;

	/**
	 * Min WC required version.
	 *
	 * @var string
	 */
	private static $min_wc_version = '4.0';

	/**
	 * Min PBC required version.
	 *
	 * @var string
	 */
	private static $min_pbc_version = '3.4.13';

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public static function plugin_url() {
		return plugin_dir_url( WCPBC_PRO_PLUGIN_FILE );
	}

	/**
	 * Get the plugin path.
	 *
	 * @return string
	 */
	public static function plugin_path() {
		return plugin_dir_path( WCPBC_PRO_PLUGIN_FILE );
	}

	/**
	 * Init plugin, Hook actions and filters
	 *
	 * @since 1.0
	 */
	public static function init() {

		self::$admin_notices = array();

		self::includes();

		register_activation_hook( WCPBC_PRO_PLUGIN_FILE, array( __CLASS__, 'install' ) );

		add_action( 'plugins_loaded', array( __CLASS__, 'init_plugin' ), 20 );
		add_action( 'init', array( __CLASS__, 'load_textdomain' ) );
		add_action( 'admin_init', array( __CLASS__, 'update_plugin_version' ) );
		add_action( 'admin_notices', array( __CLASS__, 'display_notices' ) );
	}

	/**
	 * Include required files
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	private static function includes() {
		include_once self::plugin_path() . 'includes/abstracts/class-wcpbc-exchange-rates-provider.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-backward-compatibility.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-recurring-actions.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-update-exchange-rates.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-frontend-currency.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-integrations-pro.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-shortcodes.php';
		include_once self::plugin_path() . 'includes/admin/class-wc-plugin-api-wrapper.php';
		include_once self::plugin_path() . 'includes/admin/class-wcpbc-plugin-updater.php';

		if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
			include_once self::plugin_path() . 'includes/admin/class-wcpbc-admin-pro.php';
			include_once self::plugin_path() . 'includes/admin/class-wcpbc-admin-product-csv.php';
		}
	}

	/**
	 * Load text domain
	 *
	 * @since 2.2.7
	 */
	public static function load_textdomain() {
		load_plugin_textdomain( 'wc-price-based-country-pro', false, dirname( plugin_basename( WCPBC_PRO_PLUGIN_FILE ) ) . '/languages/' );
	}

	/**
	 * Install
	 *
	 * @since 1.1
	 * @return void
	 */
	public static function install() {

		// Update currency format option.
		$currency_format = get_option( 'wc_price_based_currency_format', false );

		if ( ! $currency_format ) {
			$currency_pos = get_option( 'woocommerce_currency_pos' );
			$format       = '[symbol-alt][price]';

			switch ( $currency_pos ) {
				case 'left':
					$format = '[symbol-alt][price]';
					break;
				case 'right':
					$format = '[price][symbol-alt]';
					break;
				case 'left_space':
					$format = '[symbol-alt]&nbsp;[price]';
					break;
				case 'right_space':
					$format = '[price]&nbsp;[symbol-alt]';
					break;
			}

			update_option( 'wc_price_based_currency_format', $format );
		}

		// WooCommerce subscriptions supports.
		delete_transient( 'wc_report_subscription_events_by_date' );
		delete_transient( 'wc_report_upcoming_recurring_revenue' );

		// Trigger action.
		do_action( 'wc_price_based_country_pro_installed' );

		update_option( 'wc_price_based_country_pro_version', self::$version );

		// Increments the transient version to invalidate cache.
		WC_Cache_Helper::get_transient_version( 'product', true );
	}

	/**
	 * Update plugin version
	 *
	 * @since 2.1.9
	 * @return void
	 */
	public static function update_plugin_version() {

		$current_version = get_option( 'wc_price_based_country_pro_version', '1.0' );

		if ( $current_version === self::$version || ( defined( 'IFRAME_REQUEST' ) && IFRAME_REQUEST ) ) {
			return;
		}

		if ( version_compare( $current_version, '2.1.9', '<' ) ) {
			wp_clear_scheduled_hook( 'wc_price_based_country_pro_cron' );
		}

		if ( version_compare( $current_version, '3.5.0', '<' ) ) {

			wp_clear_scheduled_hook( 'wc_price_based_country_weekly_scheduled_events' );

			if ( class_exists( 'WCPBC_License_Settings' ) ) {
				// Upgrade license version.
				WCPBC_License_Settings::instance()->upgrade_350();
			}
		}

		if ( version_compare( $current_version, '3.6.3', '<' ) ) {
			// WooCommerce subscriptions supports.
			delete_transient( 'wcs_report_subscription_events_by_date' );
			delete_transient( 'wcs_report_upcoming_recurring_revenue' );
		}

		$exchange_rate_api = get_option( 'wc_price_based_country_exchange_rate_api' );
		if ( in_array( $exchange_rate_api, [ 'fixerio', 'yahoofinance' ], true ) ) {
			update_option( 'wc_price_based_country_exchange_rate_api', 'floatrates' );
		}

		// Increments the transient version to invalidate cache.
		WC_Cache_Helper::get_transient_version( 'product', true );

		// Update plugin version.
		update_option( 'wc_price_based_country_pro_version', self::$version );
	}

	/**
	 * Init plugin
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	public static function init_plugin() {

		if ( ! self::check_environment() ) {
			return;
		}

		// Include dependent files.
		include_once self::plugin_path() . 'includes/class-wcpbc-pricing-zone-pro.php';
		include_once self::plugin_path() . 'includes/block-types/class-wcpbc-currency-switcher-block.php';
		include_once self::plugin_path() . 'includes/admin/class-wcpbc-license-settings.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-order-base-currency.php';
		include_once self::plugin_path() . 'includes/class-wcpbc-order-base-currency-item-tax.php';

		// Init plugin.
		WCPBC_Update_Exchange_Rates::init();
		WCPBC_Frontend_Currency::init();
		WCPBC_Integrations_Pro::init();
		WCPBC_Shortcodes::init();
		WCPBC_Recurring_Actions::init();

		if ( class_exists( 'WCPBC_Admin_Pro' ) ) {
			// Admin request.
			WCPBC_Admin_Pro::init();
			WCPBC_Admin_Product_CSV::init();
		}

		// Run the plugin updater.
		self::updater();

		add_action( 'widgets_init', array( __CLASS__, 'register_widgets' ), 20 );
		add_action( 'wc_price_based_country_frontend_princing_init', array( __CLASS__, 'frontend_princing_init' ) );
		add_action( 'parse_request', array( __CLASS__, 'rest_api_init' ), 0 );
		add_action( 'wc_price_based_country_weekly_job', array( __CLASS__, 'weekly_jobs' ) );
	}

	/**
	 * Checks the environment for compatibility problems.
	 *
	 * @return boolean
	 */
	private static function check_environment() {

		if ( ! class_exists( 'WC_Product_Price_Based_Country' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s needs the Basic version to work. Learn how to install Price Based on Country in the %3$sGetting Started Guide%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="https://www.pricebasedcountry.com/docs/getting-started/">', '</a>' );
			return false;
		}

		if ( ! defined( 'WC_VERSION' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s requires WooCommerce to be activated to work. Learn how to install Price Based on Country in the %3$sGetting Started Guide%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="https://www.pricebasedcountry.com/docs/getting-started/">', '</a>' );
			return false;
		}

		if ( version_compare( WC_VERSION, self::$min_wc_version, '<' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( 'Price Based on Country Pro - The minimum WooCommerce version required for this plugin is %1$s. You are running %2$s.', 'wc-price-based-country-pro' ), self::$min_wc_version, WC_VERSION );
			return false;
		}

		if ( version_compare( WCPBC()->version, self::$min_pbc_version, '<' ) ) {
			// translators: HTML Tags.
			self::$admin_notices[] = sprintf( __( '%1$sPrice Based on Country Pro%2$s - The minimum Price Based on Country (Free) version required for this plugin is %3$s. You are running %4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', self::$min_pbc_version, WCPBC()->version );
			return false;
		}

		return true;
	}

	/**
	 * Init plugin updater
	 */
	private static function updater() {
		if ( ! class_exists( 'WCPBC_License_Settings' ) ) {
			return;
		}

		$options = WCPBC_License_Settings::instance();

		if ( $options->is_valid_key() ) {

			$updater = new WCPBC_Plugin_Updater(
				WCPBC_PRO_PLUGIN_FILE,
				self::$version,
				array(
					'item_id'     => 1450,
					'license_key' => $options->get_license_key(),
					'api_key'     => $options->get_api_key(),
				)
			);
		}

		if ( ! $options->is_license_active() ) {
			add_action( 'after_plugin_row_' . plugin_basename( WCPBC_PRO_PLUGIN_FILE ), array( __CLASS__, 'plugin_update_row' ), 20 );
			add_filter( 'wp_get_update_data', array( __CLASS__, 'update_data' ) );
		}
	}

	/**
	 * Display admin notices
	 */
	public static function display_notices() {
		foreach ( self::$admin_notices as $notice ) {
			echo '<div id="message" class="error"><p>' . wp_kses_post( $notice ) . '</p></div>';
		}
	}

	/**
	 * Display a warning message after plugin row if the license is not set.
	 */
	public static function plugin_update_row() {
		$wp_list_table = _get_list_table( 'WP_Plugins_List_Table' );
		$slug          = basename( WCPBC_PRO_PLUGIN_FILE, '.php' );
		// translators: HTML Tags.
		$message = sprintf( __( '%1$sWarning!%2$s You didn\'t set your license key, which means you\'re missing out on updates and support! Enter your %3$slicense key%4$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=price-based-country&section=license' ) ) . '">', '</a>' );
		?>
		<tr class="plugin-update-tr active" id="<?php echo esc_attr( $slug . '-nolicense' ); ?> "><td colspan="<?php echo esc_attr( $wp_list_table->get_column_count() ); ?>" class="plugin-update colspanchange"><div class="update-message notice inline notice-error notice-alt">
			<p>
				<?php echo wp_kses_post( $message ); ?>
			</p>
		</div></td></tr>
		<script>
		jQuery(document).ready(function($){
			$('tr[data-plugin="woocommerce-price-based-country-pro-addon/woocommerce-price-based-country-pro-addon.php"]').addClass('update');
		});
		</script>
		<?php
	}

	/**
	 * Increments the plugins count to display an alert on the plugin menu item.
	 *
	 * @param array $update_data Update data array.
	 * @return array
	 */
	public static function update_data( $update_data ) {
		if ( current_user_can( 'update_plugins' ) ) {
			$update_data['counts']['plugins']++;
		}
		return $update_data;
	}

	/**
	 * Register Widgets
	 */
	public static function register_widgets() {
		if ( class_exists( 'WC_Widget' ) ) {
			include_once self::plugin_path() . 'includes/class-wcpbc-widget-currency-switcher.php';
			register_widget( 'WCPBC_Widget_Currency_Switcher' );
		}
	}

	/**
	 * Front-end pricing init
	 */
	public static function frontend_princing_init() {
		add_action( 'woocommerce_coupon_loaded', array( __CLASS__, 'coupon_loaded' ), 20 );
	}

	/**
	 * Set coupon amount
	 *
	 * @param WC_Coupon $coupon Coupon instance.
	 */
	public static function coupon_loaded( $coupon ) {
		$coupon_id     = $coupon->get_id();
		$discount_type = $coupon->get_discount_type();
		$pro_version   = get_post_meta( $coupon_id, '_wcpbc_pro_version', true );

		if ( false !== strpos( $discount_type, 'percent' ) || empty( $pro_version ) ) {
			return;
		}

		$amount = wcpbc_the_zone()->get_post_price( $coupon_id, 'coupon_amount', 'coupon' );

		if ( version_compare( WC_VERSION, '3.0', '<' ) ) {
			$coupon->coupon_amount = $amount;
		} else {
			$coupon->set_amount( $amount );
		}
	}

	/**
	 * WooCommerce REST API init.
	 */
	public static function rest_api_init() {
		if ( ! ( function_exists( 'wc' ) && is_callable( array( wc(), 'is_rest_api_request' ) ) && wc()->is_rest_api_request() && ! empty( $GLOBALS['wp']->query_vars['rest_route'] ) ) ) {
			return;
		}
		$route = untrailingslashit( $GLOBALS['wp']->query_vars['rest_route'] );

		// Support for manual order via REST API.
		if ( preg_match( '/\/wc\/v\d\/(orders|subscriptions)$/', $route ) && 'post' === strtolower( $_SERVER['REQUEST_METHOD'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			$body   = is_callable( array( 'WP_REST_Server', 'get_raw_data' ) ) ? WP_REST_Server::get_raw_data() : '';
			$params = json_decode( $body );
			if ( $params ) {
				$billing_country  = isset( $params->billing->country ) ? strtoupper( $params->billing->country ) : false;
				$shipping_country = isset( $params->shipping->country ) ? strtoupper( $params->shipping->country ) : false;
				$country          = empty( $shipping_country ) || 'billing' === get_option( 'wc_price_based_country_based_on', 'billing' ) ? $billing_country : $shipping_country;

				$zone = WCPBC_Pricing_Zones::get_zone_by_country( $country );

				if ( $zone ) {
					// Load the front-end pricing for the current zone.
					wcpbc()->current_zone = $zone;
					if ( ! did_action( 'wc_price_based_country_frontend_princing_init' ) ) {
						WCPBC_Frontend_Pricing::init();
					}
				}

				// Update order metadata after order creation.
				add_action( 'woocommerce_new_order', array( 'WCPBC_Frontend', 'update_order_meta' ) );
			}
		}
	}

	/**
	 * Weekly jobs.
	 */
	public static function weekly_jobs() {

		if ( ! class_exists( 'WC_Plugin_API_Wrapper' ) ) {
			include_once self::plugin_path() . 'includes/admin/class-wc-plugin-api-wrapper.php';
		}

		// Check the license status.
		$options = WCPBC_License_Settings::instance();
		if ( $options->is_valid_key() ) {
			$options->check_license_status();
		}
	}
}


