<?php
/**
 * Currency Switcher Widget.
 *
 * @version 2.5.0
 * @package WCPBC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WCPBC_Widget_Currency_Switcher class.
 */
class WCPBC_Widget_Currency_Switcher extends WC_Widget {

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->widget_description = __( 'Display a currency switcher using a dropdown. It groups the countries by currency. When the user changes the currency, he also changes his country.', 'wc-price-based-country-pro' );
		$this->widget_id          = 'wcpbc_currency_switcher';
		$this->widget_name        = __( 'WooCommerce Currency Switcher', 'wc-price-based-country-pro' );
		$this->settings           = array(
			'title'                  => array(
				'type'  => 'text',
				'std'   => __( 'Currency', 'wc-price-based-country-pro' ),
				'label' => __( 'Title', 'wc-price-based-country-pro' ),
			),

			'flag'                   => array(
				'type'  => 'checkbox',
				'std'   => 0,
				'label' => __( 'Display flags in supported devices', 'wc-price-based-country-pro' ),
			),

			'currency_display_style' => array(
				'type'  => 'display_style',
				'std'   => '',
				'label' => __( 'Currency display style', 'wc-price-based-country-pro' ),
			),
		);

		add_action( 'woocommerce_widget_field_display_style', array( $this, 'field_display_style' ), 10, 4 );

		parent::__construct();
	}

	/**
	 * Output currency display style field.
	 *
	 * @param string $key Field key.
	 * @param string $value Field value.
	 * @param array  $setting Widget settings.
	 * @param object $instance Widget instance.
	 */
	public function field_display_style( $key, $value, $setting, $instance ) {
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>" type="text" value="<?php echo esc_attr( $value ); ?>" placeholder="[name] ([symbol])" />
		</p>
		<p class="description"><?php esc_html_e( 'Supports the following placeholders: [name] = currency name, [code] = currency code and [symbol] = currency symbol', 'wc-price-based-country-pro' ); ?></p>
		<?php
	}

	/**
	 * Widget function.
	 *
	 * @see WP_Widget
	 *
	 * @param array $args Args.
	 * @param array $instance Object instance.
	 */
	public function widget( $args, $instance ) {

		$options       = array();
		$display_style = empty( $instance['currency_display_style'] ) ? '[name] ([symbol])' : $instance['currency_display_style'];
		$data          = self::get_data();

		foreach ( $data['data'] as $currency ) {

			$options['options'][ $currency->country_code ]    = apply_filters(
				'wc_price_based_country_currency_widget_text',
				str_replace( [ '[code]', '[symbol]', '[name]', ' ' ], [ $currency->code, $currency->symbol, $currency->name, '&nbsp;' ], $display_style ),
				$currency->code
			);
			$options['currencies'][ $currency->country_code ] = $currency->code;

			if ( $currency->code === $data['selected'] ) {
				$options['selected_country'] = $currency->country_code;
			}
		}
		$options['show_flags'] = ! empty( $instance['flag'] );

		wcpbc_maybe_asort_locale( $options['options'] );

		$widget_data = wp_json_encode(
			array(
				'instance' => $instance,
				'id'       => $this->widget_id,
			)
		);

		$this->widget_start( $args, $instance );

		printf(
			'<div class="wc-price-based-country wc-price-based-country-refresh-area%1$s" data-area="widget" data-id="%2$s" data-options="%3$s">',
			esc_attr( empty( $instance['className'] ) ? '' : " {$instance['className']}" ),
			esc_attr( md5( $widget_data ) ),
			esc_attr( $widget_data )
		);

		wc_get_template(
			'currency-switcher.php',
			$options,
			'woocommerce-product-price-based-on-countries/',
			WC_Product_Price_Based_Country_Pro::plugin_path() . 'templates/'
		);
		echo '</div>';

		$this->widget_end( $args );
	}

	/**
	 * Returns widget data.
	 *
	 * @return array
	 */
	public static function get_data() {

		if ( ! class_exists( 'WCPBC_Country_Flags' ) ) {
			require_once dirname( WCPBC_PLUGIN_FILE ) . '/includes/class-wcpbc-country-flags.php';
		}

		$currencies = array();
		$countries  = array();

		$base_country  = wc_get_base_location();
		$base_country  = $base_country['country'];
		$base_currency = wcpbc_get_base_currency();

		$selected_currency = $base_currency;
		$customer_country  = wcpbc_get_woocommerce_country();

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {

			if ( ! count( $zone->get_countries() ) || ! $zone->get_enabled() ) {
				continue;
			}

			if ( count( $zone->get_countries() ) > 1 && in_array( $base_country, $zone->get_countries(), true ) ) {
				// Get first country that isn't base country.
				foreach ( $zone->get_countries() as $region_country ) {
					if ( $region_country !== $base_country ) {
						$_country = $region_country;
						break;
					}
				}
			} else {
				$_country = $zone->get_countries();
				$_country = reset( $_country );
			}

			// Check selected currency.
			if ( in_array( $customer_country, $zone->get_countries(), true ) ) {
				$selected_currency = $zone->get_currency();
				$_country          = $customer_country;
			}

			// Add currency to array.
			$currencies[ $_country ] = $zone->get_currency();

			// Add zone countries to array.
			$countries = array_merge( $countries, $zone->get_countries() );
		}

		// Add base country.
		if ( ! in_array( $base_country, $countries, true ) ) {
			$currencies[ $base_country ] = $base_currency;
		}

		// Add others countries currency.
		if ( ! in_array( $base_currency, $currencies, true ) ) {
			$country_diff            = array_diff( array_keys( WC()->countries->countries ), $countries );
			$_country                = current( $country_diff );
			$currencies[ $_country ] = $base_currency;
		}

		$data           = [];
		$all_currencies = get_woocommerce_currencies();
		foreach ( array_unique( $currencies ) as $country_code => $currency_code ) {
			$data[] = (object) [
				'country_code' => $country_code,
				'code'         => $currency_code,
				'symbol'       => get_woocommerce_currency_symbol( $currency_code ),
				'name'         => isset( $all_currencies[ $currency_code ] ) ? $all_currencies[ $currency_code ] : '',
				'emoji_flag'   => WCPBC_Country_Flags::get_by_currency( $currency_code ),
			];
		}

		return [
			'data'     => $data,
			'selected' => $selected_currency,
		];
	}
}
?>
