<?php
/**
 * Handle integration with AutomateWoo by WooCommerce.
 *
 * @since 3.2.0
 * @package WCPBC
 * @see https://automatewoo.com
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WCPBC_Automatewoo Class
 */
class WCPBC_Automatewoo {

	/**
	 * Check enviroment notice.
	 *
	 * @var string
	 */
	private static $notice = '';

	/**
	 * Hook actions and filters
	 */
	public static function init() {
		add_filter(
			'automatewoo/actions',
			function( $includes ) {
				if (
					isset( $includes['subscription_add_product'], $includes['subscription_update_product'] ) &&
					class_exists( '\AutomateWoo\Action_Subscription_Add_Product' ) &&
					class_exists( '\AutomateWoo\Actions\Subscriptions\UpdateProduct' )
				) {

					if ( ! class_exists( 'WCPBC_Automatewoo_Action_Subscription_Add_Product' ) ) {
						include_once dirname( __FILE__ ) . '/Actions/wcpbc-automatewoo-action-trait.php';
						include_once dirname( __FILE__ ) . '/Actions/class-wcpbc-automatewoo-action-subscription-add-product.php';
						include_once dirname( __FILE__ ) . '/Actions/class-wcpbc-automatewoo-action-subscription-update-product.php';
					}

					$includes['wcpbc_subscription_add_product']    = WCPBC_Automatewoo_Action_Subscription_Add_Product::class;
					$includes['wcpbc_subscription_update_product'] = WCPBC_Automatewoo_Action_Subscription_Update_Product::class;
				}
				return $includes;
			}
		);

		add_filter(
			'automatewoo/cart/price_args',
			function( $args, $cart ) {
				if ( is_callable( [ $cart, 'get_currency' ] ) ) {
					$args['currency'] = $cart->get_currency();
				}
				return $args;
			},
			10,
			2
		);

		add_filter(
			'automatewoo/variables',
			function( $variables ) {
				if (
					isset( $variables['cart']['items'] ) &&
					isset( $variables['cart']['total'] ) &&
					class_exists( '\AutomateWoo\Variable_Cart_Items' ) &&
					class_exists( '\AutomateWoo\Variable_Cart_Total' )
				) {
					if ( ! class_exists( 'WCPBC_Automatewoo_Variable_Cart_Items' ) ) {
						include_once dirname( __FILE__ ) . '/Variables/class-wcpbc-automatewoo-variable-cart-items.php';
						include_once dirname( __FILE__ ) . '/Variables/class-wcpbc-automatewoo-variable-cart-total.php';
					}

					$variables['cart']['items'] = WCPBC_Automatewoo_Variable_Cart_Items::class;
					$variables['cart']['total'] = WCPBC_Automatewoo_Variable_Cart_Total::class;
				}

				return $variables;
			}
		);

		add_action( 'automatewoo/stored_cart/updated_via_frontend', [ __CLASS__, 'update_stored_customer_cart' ] );
		add_action( 'automatewoo/stored_cart/created_via_frontend', [ __CLASS__, 'update_stored_customer_cart' ] );
	}

	/**
	 * Updates the customer stored cart with the product prices.
	 *
	 * @since 3.4.6
	 * @param AutomateWoo\Cart $aw_cart Cart instance.
	 */
	public static function update_stored_customer_cart( $aw_cart ) {

		$cart         = WC()->cart->get_cart();
		$cart_session = array();

		foreach ( $cart as $key => $values ) {
			$prices = [];
			foreach ( [ 'price', 'regular_price' ] as $prop ) {
				$getter          = 'get_' . $prop;
				$prices[ $prop ] = $values['data']->{$getter}();
			}

			if ( is_callable( [ 'WC_Subscriptions_Product', 'is_subscription' ] ) && WC_Subscriptions_Product::is_subscription( $values['data'] ) ) {
				$prices['subscription_sign_up_fee'] = $values['data']->get_meta( '_subscription_sign_up_fee' );
			}

			$cart_session[ $key ]                   = $values;
			$cart_session[ $key ]['product_title']  = $cart_session[ $key ]['data']->get_title();
			$cart_session[ $key ]['product_prices'] = $prices;

			unset( $cart_session[ $key ]['data'] ); // Unset product object.
		}

		$aw_cart->set_items( $cart_session );
		$aw_cart->save();
	}

	/**
	 * Checks the environment for compatibility problems.
	 *
	 * @return boolean
	 */
	public static function check_environment() {
		$plugin_version = defined( 'AUTOMATEWOO_VERSION' ) ? AUTOMATEWOO_VERSION : 'unknown';
		$min_version    = '5.6.8';

		if ( 'unknown' === $plugin_version || version_compare( $plugin_version, $min_version, '<' ) ) {
			// translators: 1: HTML tag, 2: HTML tag, 3: Product Bundles version.
			self::$notice = sprintf( __( '%1$sPrice Based on Country Pro & AutomateWoo%2$s compatibility requires AutomateWoo +%4$s. You are running "AutomateWoo" %3$s.', 'wc-price-based-country-pro' ), '<strong>', '</strong>', $plugin_version, $min_version );
			add_action( 'admin_notices', array( __CLASS__, 'min_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Display admin minimun version required
	 */
	public static function min_version_notice() {
		echo '<div id="message" class="error"><p>' . wp_kses_post( self::$notice ) . '</p></div>';
	}
}

if ( WCPBC_Automatewoo::check_environment() ) {
	WCPBC_Automatewoo::init();
}
