<?php
/**
 * Legacy Subscription Reports
 *
 * @package WCPBC
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WCPBC_Subscription_Reports' ) ) :

	/**
	 * WCPBC_Subscription_Reports class.
	 */
	class WCPBC_Subscription_Reports {

		/**
		 * Hook actions and filter.
		 */
		public static function init() {

			add_filter( 'wcs_reports_subscription_events_args', [ __CLASS__, 'add_query_filter' ] );
			add_filter( 'wcs_reports_subscription_events_sign_up_data', array( __CLASS__, 'remove_query_filter' ), 10, 2 );

			add_filter( 'wcs_reports_upcoming_recurring_revenue_args', [ __CLASS__, 'add_query_filter' ] );
			add_filter( 'wcs_reports_upcoming_recurring_revenue_data', array( __CLASS__, 'remove_query_filter' ) );

			add_filter( 'wcs_reports_product_query', array( __CLASS__, 'reports_product_query' ) );
			add_filter( 'wcs_reports_product_lifetime_value_query', array( __CLASS__, 'reports_product_lifetime_value_query' ) );
			add_filter( 'wcs_reports_current_customer_query', array( __CLASS__, 'current_customer_query' ) );
			add_filter( 'wcs_reports_current_customer_renewal_switch_total_query', array( __CLASS__, 'current_customer_renewal_switch_total_query' ) );
			add_filter( 'wcs_reports_customer_total_query', array( __CLASS__, 'customer_total_query' ) );
			add_filter( 'wcs_reports_customer_total_renewal_switch_query', array( __CLASS__, 'customer_total_renewal_switch' ) );

			add_filter( 'wc_price_based_country_tabs_report_notice', array( __CLASS__, 'tabs_report_notice' ) );
		}

		/**
		 * Adds the query filter to convert amounts.
		 *
		 * @param array $args Report args.
		 * @return array
		 */
		public static function add_query_filter( $args ) {
			add_filter( 'query', [ __CLASS__, 'query_filter' ] );
			return $args;
		}

		/**
		 * Adds the exchange rate conversion to query.
		 *
		 * @param string $query Report query.
		 * @return string
		 */
		public static function query_filter( $query ) {
			if ( false !== strpos( $query, 'SUM(mo.meta_value) as recurring_total,' ) ) {
				$query = self::add_exchange_rate_to_query( $query, 'mo.meta_value', 'WHERE', 'p', 'LEFT' );

			} elseif ( false !== strpos( $query, 'SUM(order_total_post_meta.meta_value) as signup_totals' ) ) {
				$query = self::add_exchange_rate_to_query( $query, 'order_total_post_meta.meta_value', 'WHERE', 'order_posts', 'LEFT' );
			}
			return $query;
		}

		/**
		 * Removes the query filter.
		 *
		 * @param array $data Report data.
		 * @return array
		 */
		public static function remove_query_filter( $data ) {
			remove_filter( 'query', [ __CLASS__, 'query_filter' ] );
			return $data;
		}

		/**
		 * Return product report query - WC_Report_Subscription_By_Product
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function reports_product_query( $query ) {
			return self::add_exchange_rate_to_query( $query, 'subscription_line_items.product_total', 'WHERE', 'subscriptions', 'LEFT' );
		}

		/**
		 * Return product lifetime value report query - WC_Report_Subscription_By_Product
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function reports_product_lifetime_value_query( $query ) {
			return self::add_exchange_rate_to_query( $query, 'wcoimeta2.meta_value', 'WHERE', 'wcorders', 'INNER' );
		}

		/**
		 * Return current customer query - WC_Report_Subscription_By_Customer
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function current_customer_query( $query ) {
			return self::add_exchange_rate_to_query( $query, 'parent_total.meta_value', 'WHERE', 'parent_order', 'LEFT' );
		}

		/**
		 * Return customer renewal switch total query - WC_Report_Subscription_By_Customer
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function current_customer_renewal_switch_total_query( $query ) {
			return self::add_exchange_rate_to_query( $query, 'renewal_switch_totals.meta_value', 'WHERE', 'renewal_order_posts', 'LEFT' );
		}

		/**
		 * Return customer total query - WC_Report_Subscription_By_Customer
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function customer_total_query( $query ) {
			return self::add_exchange_rate_to_query( $query, 'parent_total.meta_value', 'WHERE', 'parent_order', 'LEFT' );
		}

		/**
		 * Return customer total renewal switch query - WC_Report_Subscription_By_Customer
		 *
		 * @param string $query Query string.
		 * @return array
		 */
		public static function customer_total_renewal_switch( $query ) {
			return self::add_exchange_rate_to_query( $query, 'renewal_switch_totals.meta_value', 'INNER JOIN', 'subscription_posts', 'INNER' );
		}

		/**
		 * Return query with currency exchange rates
		 *
		 * @param string $query Query string.
		 * @param string $line_item_field Line item field.
		 * @param string $search Search field to insert the query.
		 * @param string $post_alias Post table alias.
		 * @param string $join_type Join Type.
		 * @return string
		 */
		private static function add_exchange_rate_to_query( $query, $line_item_field, $search, $post_alias, $join_type ) {

			$currency_rates = WCPBC_Pricing_Zones::get_currency_rates();
			$pos            = $currency_rates ? strripos( $query, $search ) : false;

			if ( $currency_rates && $pos ) {

				$caseex = WCPBC_Backward_Compatibility::built_query_case( $line_item_field, $currency_rates );
				$query  = substr( $query, 0, $pos ) . ' ' . WCPBC_Backward_Compatibility::built_join_meta_currency( $post_alias, $join_type ) . ' ' . substr( $query, $pos );
				$query  = str_replace( $line_item_field, $caseex, $query );
			}

			return $query;
		}

		/**
		 * Return subscription report screen id
		 *
		 * @param array $tabs Tabs.
		 * @return array
		 */
		public static function tabs_report_notice( $tabs ) {
			$tabs[] = 'subscriptions';
			return $tabs;
		}
	}

endif;
WCPBC_Subscription_Reports::init();
