<?php
/**
 * Run plugin class
 *
 * @since             1.0.0
 * @package           TInvWishlist
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
	die;
}

/**
 * Run plugin class
 */
class TInvWL
{

	/**
	 * Plugin name
	 *
	 * @var string
	 */
	private $_name;
	/**
	 * Plugin version
	 *
	 * @var string
	 */
	private $_version;
	/**
	 * Admin class
	 *
	 * @var TInvWL_Admin_TInvWL
	 */
	public $object_admin;
	/**
	 * Public class
	 *
	 * @var TInvWL_Public_TInvWL
	 */
	public $object_public;
	/**
	 * Array of deprecated hook handlers.
	 *
	 * @var array of WC_Deprecated_Hooks
	 */
	public $deprecated_hook_handlers = array();

	/**
	 * Constructor
	 * Created admin and public class
	 */
	function __construct()
	{
		$this->_name = TINVWL_PREFIX;
		$this->_version = TINVWL_VERSION;

		$this->set_locale();
		TIUpdater($this->_name, $this->_version);
		$this->maybe_update();
		$this->load_function();
		$this->define_hooks();
		$this->object_admin = new TInvWL_Admin_TInvWL($this->_name, $this->_version);

		// Allow to disable wishlist for frontend conditionally. Must be hooked on 'plugins_loaded' action.
		if (apply_filters('tinvwl_load_frontend', true)) {
			$this->object_public = TInvWL_Public_TInvWL::instance($this->_name, $this->_version);
		}
	}

	/**
	 * Run plugin
	 */
	function run()
	{
		TInvWL_View::_init($this->_name, $this->_version);
		TInvWL_Form::_init($this->_name);

		if (is_admin()) {
			new TInvWL_WizardSetup($this->_name, $this->_version);
			new TInvWL_Export($this->_name, $this->_version);
			$this->object_admin->load_function();
		} else {
			// Allow to disable wishlist for frontend conditionally. Must be hooked on 'plugins_loaded' action.
			if (apply_filters('tinvwl_load_frontend', true)) {
				$this->object_public->load_function();
			}
		}

		$this->deprecated_hook_handlers['actions'] = new TInvWL_Deprecated_Actions();
		$this->deprecated_hook_handlers['filters'] = new TInvWL_Deprecated_Filters();
		$this->rest_api = TInvWL_API::init();
	}

	/**
	 * Set localization
	 */
	private function set_locale()
	{
		if (function_exists('determine_locale')) {
			$locale = determine_locale();
		} else {
			$locale = is_admin() && function_exists('get_user_locale') ? get_user_locale() : get_locale();
		}

		$locale = apply_filters('plugin_locale', $locale, TINVWL_DOMAIN);

		$mofile = sprintf('%1$s-%2$s.mo', TINVWL_DOMAIN, $locale);
		$mofiles = array();

		$mofiles[] = WP_LANG_DIR . DIRECTORY_SEPARATOR . basename(TINVWL_PATH) . DIRECTORY_SEPARATOR . $mofile;
		$mofiles[] = WP_LANG_DIR . DIRECTORY_SEPARATOR . 'plugins' . DIRECTORY_SEPARATOR . $mofile;
		$mofiles[] = TINVWL_PATH . 'languages' . DIRECTORY_SEPARATOR . $mofile;
		foreach ($mofiles as $mofile) {
			if (file_exists($mofile) && load_textdomain(TINVWL_DOMAIN, $mofile)) {
				return;
			}
		}

		load_plugin_textdomain(TINVWL_DOMAIN, false, basename(TINVWL_PATH) . DIRECTORY_SEPARATOR . 'languages');
	}

	/**
	 * Define hooks
	 */
	function define_hooks()
	{

	}

	/**
	 * Load function
	 */
	function load_function()
	{
	}

	/**
	 * Testing for the ability to update the functional
	 */
	function maybe_update()
	{
		$current = 'p.' . $this->_version;
		$prev = $this->get_current_version($current);
		$prev_p = get_option($this->_name . '_verp');
		if (false === $prev_p) {
			add_option($this->_name . '_verp', $this->_version);
		}
		if (version_compare($current, $prev, 'gt')) {
			new TInvWL_Update($current, $prev);
			TInvWL_Activator::activate();
			update_option($this->_name . '_verp', $this->_version);
			do_action('tinvwl_updated', $current, $prev);
		}
	}

	/**
	 * Get current version
	 *
	 * @param string $version Default Version.
	 *
	 * @return string
	 */
	function get_current_version($version = false)
	{
		$_version = get_option($this->_name . '_verp');
		if (false !== $_version) {
			return 'p.' . $_version;
		}
		$_version = get_option($this->_name . '_ver');
		if (false !== $_version) {
			return 'f.' . $_version;
		}

		return $version;
	}
}
