<?php

/**
 * The Theme My Login Profiles Extension
 *
 * @package Theme_My_Login_Profiles
 */

/*
Plugin Name: Theme My Login Profiles
Plugin URI: https://thememylogin.com/extensions/profiles
Description: Adds profile support to Theme My Login.
Author: Theme My Login
Author URI: https://thememylogin.com
Version: 1.0.7
Text Domain: theme-my-login-profiles
Network: true
*/

// Bail if TML is not active
if ( ! class_exists( 'Theme_My_Login_Extension' ) ) {
	return;
}

/**
 * The class used to implement the Profiles extension.
 */
class Theme_My_Login_Profiles extends Theme_My_Login_Extension {

	/**
	 * The extension name.
	 *
	 * @var string
	 */
	protected $name = 'theme-my-login-profiles';

	/**
	 * The extension version.
	 *
	 * @var string
	 */
	protected $version = '1.0.7';

	/**
	 * The extension's documentation URL.
	 *
	 * @var string
	 */
	protected $documentation_url = 'https://docs.thememylogin.com/category/26-profiles';

	/**
	 * The extension's support URL.
	 *
	 * @var string
	 */
	protected $support_url = 'https://thememylogin.com/support';

	/**
	 * The extension's store URL.
	 *
	 * @var string
	 */
	protected $store_url = 'https://thememylogin.com';

	/**
	 * The extension's item ID.
	 *
	 * @var int
	 */
	protected $item_id = 46;

	/**
	 * The option name used to store the license key.
	 *
	 * @var string
	 */
	protected $license_key_option = 'tml_profiles_license_key';

	/**
	 * The option name used to store the license status.
	 *
	 * @var string
	 */
	protected $license_status_option = 'tml_profiles_license_status';

	/**
	 * Set class properties.
	 *
	 * @since 1.0
	 */
	protected function set_properties() {
		$this->title = __( 'Profiles', 'theme-my-login-profiles' );
	}

	/**
	 * Include extension files.
	 *
	 * @since 1.0
	 */
	protected function include_files() {
		require $this->path . 'functions.php';

		if ( is_admin() ) {
			require $this->path . 'admin.php';
		}
	}

	/**
	 * Add extension actions.
	 *
	 * @since 1.0
	 */
	protected function add_actions() {
		// Flush permalinks on activation/deactivation
		add_action( 'tml_activate_profiles', 'tml_flush_rewrite_rules' );
		add_action( 'tml_deactivate_profiles', 'tml_flush_rewrite_rules' );

		// Register the action
		add_action( 'init', 'tml_profiles_register_action', 0 );

		// Register the form
		add_action( 'init', 'tml_profiles_register_form', 0 );

		if ( is_admin() ) {
			// Themed profile check
			add_action( 'admin_init', 'tml_profiles_admin_themed_profile_check' );

			// Restricted admin check
			add_action( 'admin_init', 'tml_profiles_admin_restricted_admin_check' );
		} else {
			// Email change notification
			add_action( 'personal_options_update', 'send_confirmation_on_profile_email' );
		}
	}

	/**
	 * Add extension filters.
	 *
	 * @since 1.0
	 */
	protected function add_filters() {
		// Change the link to edit a user's profile
		add_filter( 'edit_profile_url', 'tml_profiles_filter_edit_profile_url', 10, 3 );

		// Hide the admin bar for users with restricted admin access
		add_filter( 'show_admin_bar', 'tml_profiles_filter_show_admin_bar' );

		// Change the link for wp_register() when logged in
		add_filter( 'register', 'tml_profiles_filter_register' );

		// Change the link to profiles generated by admin_url()
		add_filter( 'admin_url', 'tml_profiles_filter_admin_url', 10, 2 );
	}

	/**
	 * Get the extension settings page args.
	 *
	 * @since 1.0
	 *
	 * @return array The extension settings page args.
	 */
	public function get_settings_page_args() {
		return array(
			'page_title' => __( 'Theme My Login Profiles Settings', 'theme-my-login-profiles' ),
			'menu_title' => __( 'Profiles', 'theme-my-login-profiles' ),
			'menu_slug' => 'theme-my-login-profiles',
		);
	}

	/**
	 * Get the extension settings sections.
	 *
	 * @since 1.0
	 *
	 * @return array The extension settings sections.
	 */
	public function get_settings_sections() {
		return tml_profiles_admin_get_settings_sections();
	}

	/**
	 * Get the extension settings fields.
	 *
	 * @since 1.0
	 *
	 * @return array The extension settings fields.
	 */
	public function get_settings_fields() {
		return tml_profiles_admin_get_settings_fields();
	}

	/**
	 * Update the extension.
	 *
	 * @since 1.0
	 */
	protected function update() {
		$version = get_site_option( '_tml_profiles_version' );

		if ( version_compare( $version, $this->version, '>=' ) ) {
			return;
		}

		// Initial migration
		$options = get_option( 'theme_my_login_profiles', array() );
		if ( ! empty( $options ) ) {
			$themed_profile_roles = array();
			$restricted_admin_roles = array();
			foreach ( wp_roles()->get_names() as $role => $role_name ) {
				if ( ! empty( $options[ $role ]['theme_profile'] ) ) {
					$themed_profile_roles[] = $role;
				}
				if ( ! empty( $options[ $role ]['restrict_admin'] ) ) {
					$restricted_admin_roles[] = $role;
				}
			}
			update_site_option( 'tml_profiles_themed_profile_roles', $themed_profile_roles );
			update_site_option( 'tml_profiles_restricted_admin_roles', $restricted_admin_roles );

			delete_option( 'theme_my_login_profiles' );
		}

		update_site_option( '_tml_profiles_version', $this->version );

		tml_flush_rewrite_rules();
	}
}

tml_register_extension( new Theme_My_Login_Profiles( __FILE__ ) );
