/**
 * Admin Order Script
 *
 * Initializes intl-tel-input on billing/shipping phone fields in the
 * WooCommerce Add/Edit Order admin screens and performs validation before save.
 *
 * @package    Phone_Number_Validation
 */

(function () {
	'use strict';

	// Ensure localized data is available.
	if (typeof pnvAdminOrderData === 'undefined') {
		// Nothing to do if localization not available.
		return;
	}

	// Provide a global error map to match frontend expectations.
	// Some of the validation logic references window.wcPvPhoneErrorMap.
	try {
		window.wcPvPhoneErrorMap = pnvAdminOrderData.validationErrors || {};
	} catch (e) {
		window.wcPvPhoneErrorMap = {};
	}

	// Expose a couple of fallback messages to globals for convenience.
	window.wcPvPhoneUnknownErrorMsg = pnvAdminOrderData.phoneUnknownErrorMsg || 'Unknown number';
	window.wcPvPleaseCorrectPhone = pnvAdminOrderData.pleaseCorrectPhone || 'Please correct the phone number(s) before saving.';

	/**
	 * Helper: find the order edit/add form.
	 * WP post edit/add form has id "post".
	 */
	function getOrderForm() {
		// Prefer the standard post edit form, then common fallbacks used in WP admin.
		var form = document.getElementById('post') || document.querySelector('form#post') || document.querySelector('form[name="post"]');

		// Some admin pages render the order editor outside the usual post form.
		// Try to find a nearby form inside the order_data panel.
		if ( ! form ) {
			var orderPanel = document.getElementById('order_data') || document.querySelector('.woocommerce-order-data');
			if ( orderPanel ) {
				form = orderPanel.closest('form') || orderPanel.querySelector('form');
			}
		}

		// As a last resort, return the first form on the page (less ideal but workable).
		if ( ! form ) {
			form = document.querySelector('form');
		}

		return form;
	}

	/**
	 * Initialize intlTelInput on a given input element.
	 *
	 * @param {HTMLInputElement} input
	 * @returns {object|null} intl instance or null
	 */
	function initIntl(input, preferredCountry) {
		try {
			if (!input) {
				console.debug('pnv: initIntl called with no input element.');
				return null;
			}
			if (typeof window.intlTelInput !== 'function') {
				console.debug('pnv: intlTelInput not available on window.');
				return null;
			}

			// Determine initial country preference:
			// 1) preferredCountry passed in (from the admin select)
			// 2) wcDefaultCountry localized (store base country)
			// 3) plugin initialCountry setting (pnv initial country)
			// 4) fallback to 'US'
			var initialCountry = 'US';
			try {
				if ( typeof preferredCountry !== 'undefined' && preferredCountry ) {
					initialCountry = ('' + preferredCountry).toLowerCase();
				} else if ( typeof pnvAdminOrderData !== 'undefined' && pnvAdminOrderData.wcDefaultCountry ) {
					initialCountry = ('' + pnvAdminOrderData.wcDefaultCountry).toLowerCase();
				} else if ( typeof pnvAdminOrderData !== 'undefined' && pnvAdminOrderData.defaultCountry ) {
					initialCountry = ('' + pnvAdminOrderData.defaultCountry).toLowerCase();
				}
			} catch (e) {
				/* ignore and keep default */
			}

			// If the input itself contains an international number (starts with '+'),
			// derive the most specific country from the dial code and prefer that.
			try {
				if (input.value && input.value.trim().charAt(0) === '+' && typeof window.intlTelInputGlobals !== 'undefined' && window.intlTelInputGlobals.getCountryData) {
					var phone = input.value.substring(1);
					var countryData = window.intlTelInputGlobals.getCountryData();
					var best = { dialCode: '', iso2: null };
					countryData.forEach(function (c) {
						if (phone.indexOf(c.dialCode) === 0 && c.dialCode.length > best.dialCode.length) {
							best.dialCode = c.dialCode;
							best.iso2 = c.iso2;
						}
					});
					if (best.iso2) {
						initialCountry = best.iso2;
					}
				}
			} catch (e) {
				/* ignore and fall back to preferred/default */
			}

			var opts = {
				initialCountry: initialCountry,
				separateDialCode: !!pnvAdminOrderData.separateDialCode,
				preferredCountries: Array.isArray(pnvAdminOrderData.preferredCountries) ? pnvAdminOrderData.preferredCountries.map(function(c){ return (''+c).toUpperCase(); }) : [],
				utilsScript: pnvAdminOrderData.utilsScript || '',
				allowDropdown: !!pnvAdminOrderData.allowDropdown,
				excludeCountries: Array.isArray(pnvAdminOrderData.excludeCountries) ? pnvAdminOrderData.excludeCountries.map(function(c){ return (''+c).toUpperCase(); }) : [],
				nationalMode: true,
				formatOnDisplay: false,
				customPlaceholder: function (example, countryData) {
					try {
						if (pnvAdminOrderData.phonePlaceholder) {
							return pnvAdminOrderData.phonePlaceholder;
						}
					} catch (e) {
						/* ignore */
					}
					return example;
				}
			};

			console.debug('pnv: initializing intlTelInput on', input, 'with opts', opts);
			var inst = window.intlTelInput(input, opts);
			console.debug('pnv: intlTelInput instance created for', input, inst ? true : false);
			return inst;
		} catch (err) {
			console.error('pnv: initIntl error', err);
			return null;
		}
	}

	/**
	 * Create or return an existing hidden input in the form.
	 *
	 * @param {HTMLFormElement} form
	 * @param {string} name
	 * @returns {HTMLInputElement}
	 */
	function ensureHidden(form, name) {
		var existing = form.querySelector('input[name="' + name + '"]');
		if (existing) {
			return existing;
		}
		var input = document.createElement('input');
		input.type = 'hidden';
		input.name = name;
		form.appendChild(input);
		return input;
	}

	/**
	 * Display an admin notice at the top of the post form.
	 *
	 * @param {string} message
	 */
	function showAdminNotice(message) {
		// Remove any previous PNV notices.
		var prev = document.getElementById('pnv-admin-notice');
		if (prev) {
			prev.parentNode.removeChild(prev);
		}

		var notice = document.createElement('div');
		notice.id = 'pnv-admin-notice';
		notice.className = 'notice notice-error inline';
		notice.style.marginTop = '1em';

		var p = document.createElement('p');
		p.textContent = message;
		notice.appendChild(p);

		// Insert notice before the form (or at top of metabox container).
		var container = document.getElementById('wpbody-content') || document.getElementById('post-body');
		if (container && container.firstChild) {
			container.insertBefore(notice, container.firstChild);
		} else {
			var form = getOrderForm();
			if (form && form.parentNode) {
				form.parentNode.insertBefore(notice, form);
			} else {
				// Fallback to alert if DOM placement fails.
				alert(message);
			}
		}
	}

	/**
	 * Validate a single intl instance.
	 *
	 * @param {object|null} instance
	 * @param {string} fieldLabel
	 * @returns {{formatted:String|null, error:String|null}}
	 */
	function validateInstance(instance, fieldLabel) {
		var result = { formatted: '', error: '' };

		if (!instance) {
			return result;
		}

		try {
			if (typeof instance.isValidNumber === 'function' && instance.isValidNumber()) {
				// Use INTERNATIONAL format.
				result.formatted = instance.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL);
			} else if (typeof instance.getValidationError === 'function') {
				var err = instance.getValidationError();
				result.error = fieldLabel + ' phone ' + (window.wcPvPhoneErrorMap && window.wcPvPhoneErrorMap[err] ? window.wcPvPhoneErrorMap[err] : (pnvAdminOrderData.phoneUnknownErrorMsg || 'Unknown number'));
			}
		} catch (e) {
			/* ignore */
		}

		return result;
	}

	/**
	 * Main initializer executed on DOM ready.
	 */
	function init() {
		try {
			console.debug('pnv: admin-order init start. pnvAdminOrderData present:', typeof pnvAdminOrderData !== 'undefined');

			// Track whether we've successfully initialized inputs to avoid double-init.
			var initialized = false;

			// The main setup that finds inputs and wires intl + handlers.
			function setup() {
				if (initialized) {
					return;
				}
				var form = getOrderForm();

				// Locate billing and shipping phone fields in the admin order screen.
				var billingSelector = 'input[name="billing_phone"], input#billing_phone, input[name="_billing_phone"], input#_billing_phone';
				var shippingSelector = 'input[name="shipping_phone"], input#shipping_phone, input[name="_shipping_phone"], input#_shipping_phone';

				var billingInput = document.querySelector(billingSelector);
				var shippingInput = document.querySelector(shippingSelector);

				console.debug('pnv: selectors used:', billingSelector, shippingSelector);
				console.debug('pnv: billingInput found:', !!billingInput, billingInput);
				console.debug('pnv: shippingInput found:', !!shippingInput, shippingInput);

				// If neither input exists yet, don't mark initialized; wait for interaction.
				if (!billingInput && !shippingInput) {
					return;
				}

				// Locate billing/shipping country selects (support both _prefixed and non-prefixed).
				var billingCountrySelect = document.querySelector('select#_billing_country, select[name="_billing_country"], select#billing_country, select[name="billing_country"]');
				var shippingCountrySelect = document.querySelector('select#_shipping_country, select[name="_shipping_country"], select#shipping_country, select[name="shipping_country"]');

				// Determine initial country values from selects if available.
				var billingInitial = billingCountrySelect && billingCountrySelect.value ? billingCountrySelect.value : null;
				var shippingInitial = shippingCountrySelect && shippingCountrySelect.value ? shippingCountrySelect.value : null;

				console.debug('pnv: billingInitial (from select) =', billingInitial, 'shippingInitial =', shippingInitial);

				// Helper: try to read a Select2 rendered label for a select if its value is empty.
				function getSelect2Text(selectEl) {
					try {
						if (!selectEl || !selectEl.id) {
							return '';
						}
						// Select2 container id format is select2-{selectId}-container
						var container = document.getElementById('select2-' + selectEl.id + '-container');
						if (container) {
							return (container.getAttribute('title') || container.textContent || '').trim();
						}
						// Fallback: find nearest .select2-selection__rendered
						var selParent = selectEl.closest && selectEl.closest('.select2-container');
						if (selParent) {
							var rendered = selParent.querySelector('.select2-selection__rendered');
							if (rendered) {
								return (rendered.getAttribute('title') || rendered.textContent || '').trim();
							}
						}
					} catch (e) {
						/* ignore */
					}
					return '';
				}

				// Map a displayed country name to an ISO2 code by searching intl-tel-input country data.
				function countryNameToIso(name) {
					if (!name || typeof name !== 'string' || !window.intlTelInputGlobals || !window.intlTelInputGlobals.getCountryData) {
						return null;
					}
					var countryData = window.intlTelInputGlobals.getCountryData();
					name = name.toLowerCase();
					// Exact name match first, then contains.
					for (var i = 0; i < countryData.length; i++) {
						if (countryData[i].name && countryData[i].name.toLowerCase() === name) {
							return countryData[i].iso2;
						}
					}
					for (var j = 0; j < countryData.length; j++) {
						if (countryData[j].name && name.indexOf(countryData[j].name.toLowerCase()) !== -1) {
							return countryData[j].iso2;
						}
						// Also allow the reverse: country name contains displayed text (e.g., "United Kingdom (UK)").
						if (countryData[j].name && countryData[j].name.toLowerCase().indexOf(name) !== -1) {
							return countryData[j].iso2;
						}
					}
					return null;
				}

				// Helper to attempt setting country with retries (handles select2/async rendering).
				function attemptSetCountry(intlInstance, selectEl, attemptsLeft) {
					try {
						if (!intlInstance || !selectEl) {
							console.debug('pnv: attemptSetCountry - missing instance or select', !!intlInstance, !!selectEl);
							return;
						}
						var val = selectEl.value;
						console.debug('pnv: attemptSetCountry - attemptsLeft=', attemptsLeft, 'select value=', val);

						// If select has an explicit value, try that first.
						if (val) {
							try {
								intlInstance.setCountry(('' + val).toLowerCase());
								console.debug('pnv: set intl country to (from select value)', val);
								return;
							} catch (e) {
								console.debug('pnv: setCountry threw for value', val, e);
							}
						}

						// Try reading select2 rendered text (title/text) as a fallback.
						var label = getSelect2Text(selectEl);
						console.debug('pnv: attemptSetCountry - select2 label fallback =', label);
						if (label) {
							var iso = countryNameToIso(label);
							console.debug('pnv: countryNameToIso("' + label + '") ->', iso);
							if (iso) {
								try {
									intlInstance.setCountry(('' + iso).toLowerCase());
									console.debug('pnv: set intl country to (from select2 label)', iso, 'label=', label);
									return;
								} catch (e) {
									console.debug('pnv: setCountry threw for iso', iso, e);
								}
							}
						}

						// As a last resort: attempt to infer from the selected option's text content.
						try {
							var selectedOption = selectEl.options && selectEl.options[selectEl.selectedIndex];
							if (selectedOption && selectedOption.text) {
								var optIso = countryNameToIso(selectedOption.text);
								console.debug('pnv: attemptSetCountry - selectedOption.text ->', selectedOption.text, 'iso ->', optIso);
								if (optIso) {
									intlInstance.setCountry(optIso);
									console.debug('pnv: set intl country to (from selected option text)', optIso);
									return;
								}
							}
						} catch (e) {
							/* ignore */
						}

						// If no value/label yet and we have attempts left, retry shortly.
						if (attemptsLeft > 0) {
							setTimeout(function () {
								attemptSetCountry(intlInstance, selectEl, attemptsLeft - 1);
							}, 150);
						} else {
							console.debug('pnv: attemptSetCountry exhausted attempts without finding a country.');
						}
					} catch (e) {
						/* ignore */
					}
				}

				// Initialize intl instances where possible, passing preferred initial country when known.
				var billingIntl = billingInput ? initIntl(billingInput, billingInitial) : null;
				var shippingIntl = shippingInput ? initIntl(shippingInput, shippingInitial) : null;

				console.debug('pnv: billingIntl instance:', !!billingIntl);
				console.debug('pnv: shippingIntl instance:', !!shippingIntl);

				// Immediately attempt to set countries (in case init didn't pick up preferred country).
				try {
					if (billingCountrySelect) {
						console.debug('pnv: attempting to set billing country immediately');
						attemptSetCountry(billingIntl, billingCountrySelect, 6);
					}
					if (shippingCountrySelect) {
						console.debug('pnv: attempting to set shipping country immediately');
						attemptSetCountry(shippingIntl, shippingCountrySelect, 6);
					}
				} catch (e) {
					/* ignore */
				}

				// If country selects exist, wire change handlers to update intl instances when changed.
				try {
					// Helper to attempt setting country with retries (handles select2/async rendering).
					function attemptSetCountry(intlInstance, selectEl, attemptsLeft) {
						try {
							if (!intlInstance || !selectEl) {
								return;
							}
							var val = selectEl.value;
							if (val) {
								// intl-tel-input expects lower-case ISO2 codes.
								try {
									intlInstance.setCountry(('' + val).toLowerCase());
									console.debug('pnv: set intl country to', val);
									return;
								} catch (e) {
									/* ignore setCountry errors */
								}
							}
							// If no value yet and we have attempts left, retry shortly.
							if (attemptsLeft > 0) {
								setTimeout(function () {
									attemptSetCountry(intlInstance, selectEl, attemptsLeft - 1);
								}, 150);
							}
						} catch (e) {
							/* ignore */
						}
					}

					if (billingCountrySelect) {
						// Try to set immediately with retries.
						attemptSetCountry(billingIntl, billingCountrySelect, 6);

						billingCountrySelect.addEventListener('change', function () {
							try {
								if (billingIntl && typeof billingIntl.setCountry === 'function') {
									billingIntl.setCountry(('' + this.value).toLowerCase());
									console.debug('pnv: billing intl country updated to', this.value);
								}
							} catch (e) {
								/* ignore setCountry errors */
							}
						});
					}
					if (shippingCountrySelect) {
						// Try to set immediately with retries.
						attemptSetCountry(shippingIntl, shippingCountrySelect, 6);

						shippingCountrySelect.addEventListener('change', function () {
							try {
								if (shippingIntl && typeof shippingIntl.setCountry === 'function') {
									shippingIntl.setCountry(('' + this.value).toLowerCase());
									console.debug('pnv: shipping intl country updated to', this.value);
								}
							} catch (e) {
								/* ignore setCountry errors */
							}
						});
					}
				} catch (e) {
					/* ignore */
				}

				// Choose container for hidden fields and notices.
				var container = form || (document.getElementById('order_data') || document.querySelector('.woocommerce-order-data')) || document.body;

				// Ensure hidden inputs exist so code can place validated/err values.
				var billingValidName = (pnvAdminOrderData.phoneValidNamePrefix || 'pnv_phone_valid_') + 'billing';
				var billingErrName = (pnvAdminOrderData.phoneErrNamePrefix || 'pnv_phone_error_') + 'billing';
				var shippingValidName = (pnvAdminOrderData.phoneValidNamePrefix || 'pnv_phone_valid_') + 'shipping';
				var shippingErrName = (pnvAdminOrderData.phoneErrNamePrefix || 'pnv_phone_error_') + 'shipping';

				// Prefer appending hidden fields to the actual order form when available so they
				// are always included in the admin post submission. Fall back to the container.
				var hidContainer = form || container;
				var hidBillingValid = ensureHidden(hidContainer, billingValidName);
				var hidBillingErr = ensureHidden(hidContainer, billingErrName);
				var hidShippingValid = ensureHidden(hidContainer, shippingValidName);
				var hidShippingErr = ensureHidden(hidContainer, shippingErrName);

				// Helper to sync hidden fields based on intl instances.
				function syncHidden() {
					try {
						// Billing
						if (hidBillingValid) {
							if (billingIntl && typeof billingIntl.isValidNumber === 'function') {
								hidBillingValid.value = billingIntl.isValidNumber() ? billingIntl.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL) : '';
							} else if (billingInput) {
								hidBillingValid.value = billingInput.value || '';
							}
						}
						if (hidBillingErr) {
							if (billingIntl && typeof billingIntl.isValidNumber === 'function') {
								var v = billingIntl.isValidNumber() ? '' : (billingIntl.getValidationError ? billingIntl.getValidationError() : '');
								hidBillingErr.value = v ? (window.wcPvPhoneErrorMap && window.wcPvPhoneErrorMap[v] ? window.wcPvPhoneErrorMap[v] : (pnvAdminOrderData.phoneUnknownErrorMsg || 'Unknown number')) : '';
							} else {
								hidBillingErr.value = '';
							}
						}

						// Shipping
						if (hidShippingValid) {
							if (shippingIntl && typeof shippingIntl.isValidNumber === 'function') {
								hidShippingValid.value = shippingIntl.isValidNumber() ? shippingIntl.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL) : '';
							} else if (shippingInput) {
								hidShippingValid.value = shippingInput.value || '';
							}
						}
						if (hidShippingErr) {
							if (shippingIntl && typeof shippingIntl.isValidNumber === 'function') {
								var sv = shippingIntl.isValidNumber() ? '' : (shippingIntl.getValidationError ? shippingIntl.getValidationError() : '');
								hidShippingErr.value = sv ? (window.wcPvPhoneErrorMap && window.wcPvPhoneErrorMap[sv] ? window.wcPvPhoneErrorMap[sv] : (pnvAdminOrderData.phoneUnknownErrorMsg || 'Unknown number')) : '';
							} else {
								hidShippingErr.value = '';
							}
						}
					} catch (e) {
						/* ignore sync errors */
					}
				}

				// Bind input events to keep hidden fields up-to-date.
				if (billingInput) {
					billingInput.addEventListener('change', syncHidden);
					billingInput.addEventListener('input', syncHidden);
				}
				if (shippingInput) {
					shippingInput.addEventListener('change', syncHidden);
					shippingInput.addEventListener('input', syncHidden);
				}

				// Handler used when trying to save the order (either form submit or button click).
				function handleBeforeSave(event) {
					try {
						console.debug('pnv: before save handler triggered. Current values:', {
							billingInput: billingInput ? billingInput.value : null,
							shippingInput: shippingInput ? shippingInput.value : null
						});
					} catch (e) { /* ignore */ }

					// Sync hidden fields with current values.
					syncHidden();

					// Validate billing/shipping.
					var billingRes = billingIntl ? validateInstance(billingIntl, 'Billing') : { formatted: '', error: '' };
					var shippingRes = shippingIntl ? validateInstance(shippingIntl, 'Shipping') : { formatted: '', error: '' };

					console.debug('pnv: validation results (before save)', { billingRes: billingRes, shippingRes: shippingRes });

					// Update hidden fields again based on validation results.
					if (hidBillingValid) {
						hidBillingValid.value = billingRes.formatted || (billingInput ? billingInput.value || '' : '');
					}
					if (hidBillingErr) {
						hidBillingErr.value = billingRes.error || '';
					}
					if (hidShippingValid) {
						hidShippingValid.value = shippingRes.formatted || (shippingInput ? shippingInput.value || '' : '');
					}
					if (hidShippingErr) {
						hidShippingErr.value = shippingRes.error || '';
					}

					var hasError = false;
					var messageParts = [];

					if (billingRes.error) {
						hasError = true;
						messageParts.push(billingRes.error);
					}
					if (shippingRes.error) {
						hasError = true;
						messageParts.push(shippingRes.error);
					}

					// If errors and alwaysAllowCheckout is falsy, prevent save and show notice.
					if (hasError && !pnvAdminOrderData.alwaysAllowCheckout) {
						// If this was a click event on a button/link, prevent it.
						if (event && typeof event.preventDefault === 'function') {
							event.preventDefault();
							event.stopPropagation();
						}
						var msg = messageParts.length ? messageParts.join(' / ') : (pnvAdminOrderData.pleaseCorrectPhone || 'Please correct the phone number(s) before saving.');
						showAdminNotice(msg);
						// Scroll to top of admin area so notice is visible.
						window.scrollTo({ top: 0, behavior: 'smooth' });
						return false;
					}

					// Otherwise allow save to continue.
					return true;
				}

				// If a form exists, attach to its submit event.
				if (form) {
					form.addEventListener('submit', handleBeforeSave, true);
				} else {
					// Otherwise intercept clicks on common admin save buttons and attempt validation.
					var saveSelectors = ['#publish', '#save-post', 'button.save_order', '.save_order', 'button#save-order', 'button.save-post', 'input#save-post', 'button#publish'];
					document.addEventListener('click', function (e) {
						var t = e.target;
						if (!t) {
							return;
						}
						for (var i = 0; i < saveSelectors.length; i++) {
							var sel = saveSelectors[i];
							if (t.closest && t.closest(sel)) {
								var allowed = handleBeforeSave(e);
								if (!allowed) {
									// Prevent the original click from proceeding.
									return;
								}
								// If allowed, let the click continue.
								return;
							}
						}
					}, true);
				}

				// Mark initialized so we don't duplicate handlers.
				initialized = true;
				console.debug('pnv: admin-order setup complete.');
			} // end setup

			// Try an initial setup (in case fields are present immediately).
			setup();

			// Watch for clicks that open the address editor so we can initialize once it appears.
			document.addEventListener('click', function (e) {
				var target = e.target;
				if (target && (target.closest && (target.closest('.edit_address') || target.closest('.load_customer_billing') || target.closest('.load_customer_shipping') || target.closest('.billing-same-as-shipping')))) {
					// Delay slightly to allow the editor to become visible.
					setTimeout(setup, 100);
				}
			}, true);

			// Observe DOM changes inside the order data panel to initialize when the address block is toggled/inserted.
			var orderData = document.getElementById('order_data') || document.querySelector('.woocommerce-order-data');
			if (orderData && typeof MutationObserver !== 'undefined') {
				var mo = new MutationObserver(function (mutations) {
					// Attempt setup whenever subtree changes occur.
					setup();
				});
				mo.observe(orderData, { childList: true, subtree: true, attributes: true });
			}

		} catch (err) {
			console.error('pnv: init error', err);
			return;
		}
	}

	// Delay init until DOMContentLoaded.
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();
