/**
 * Block Checkout Script
 * Initializes validation for phone number fields in the WooCommerce checkout block.
 *
 * @package    Phone_Number_Validation
 */

document.addEventListener(
	'DOMContentLoaded',
	function () {
		'use strict';

		// Check if wcPvJson is defined.
		if (typeof wcPvJson === 'undefined') {
			console.error('wcPvJson is not defined.');
			return;
		}

		// Check if wcPvBlocksJson is defined.
		if (typeof wcPvBlocksJson === 'undefined') {
			console.error('wcPvBlocksJson is not defined.');
			return;
		}

		// --- Dynamic Default Country Setup for Blocks ---

		/**
		 * Function to dynamically determine default country from a phone number.
		 *
		 * @param {string} phone - The phone number (expected in international format).
		 * @returns {string|null} The ISO country code (e.g. "za") or null if not found.
		 */
		function getDefaultCountryFromPhone(phone) {
			// Ensure the phone number is in international format.
			if (!phone || phone.charAt(0) !== '+' || typeof window.intlTelInputGlobals === 'undefined' || !window.intlTelInputGlobals.getCountryData) {
				return null;
			}

			// Retrieve country data.
			var countryData = window.intlTelInputGlobals.getCountryData();
			var bestMatch = { dialCode: '', iso2: null };

			// Remove the '+' sign.
			phone = phone.substring(1);

			// Loop over each country to find the longest matching dial code.
			countryData.forEach(function(country) {
				var dialCode = country.dialCode;
				if (phone.indexOf(dialCode) === 0 && dialCode.length > bestMatch.dialCode.length) {
					bestMatch.dialCode = dialCode;
					bestMatch.iso2 = country.iso2;
				}
			});

			// For dial codes shared by multiple countries provide a preferred mapping.
			// This ensures ambiguous dial codes (e.g. +44) resolve to the expected main country flag.
			var preferredDialCountry = {
				'44': 'gb', // prefer Great Britain for +44 (over GG, JE, IM)
				'1': 'us',  // prefer United States for +1 (over CA, etc.)
				'7': 'ru'   // prefer Russia for +7 (over KZ)
			};

			// Merge server-provided mappings (from PHP localization).
			try {
				if (wcPvJson && wcPvJson.preferredDialCountry && typeof wcPvJson.preferredDialCountry === 'object') {
					for (var k in wcPvJson.preferredDialCountry) {
						if (Object.prototype.hasOwnProperty.call(wcPvJson.preferredDialCountry, k)) {
							preferredDialCountry[k] = ('' + wcPvJson.preferredDialCountry[k]).toLowerCase();
						}
					}
				}
				// Also allow blocks-specific localized JSON (wcPvBlocksJson) to provide mappings
				// in contexts where blocks-localization is used separately.
				if (typeof wcPvBlocksJson !== 'undefined' && wcPvBlocksJson && wcPvBlocksJson.preferredDialCountry && typeof wcPvBlocksJson.preferredDialCountry === 'object') {
					for (var kb in wcPvBlocksJson.preferredDialCountry) {
						if (Object.prototype.hasOwnProperty.call(wcPvBlocksJson.preferredDialCountry, kb)) {
							preferredDialCountry[kb] = ('' + wcPvBlocksJson.preferredDialCountry[kb]).toLowerCase();
						}
					}
				}
			} catch (e) {
				/* ignore */
			}

			// Merge any global JS-provided mappings (useful for runtime overrides without PHP changes).
			try {
				if (window.pnv_preferred_dial_country && typeof window.pnv_preferred_dial_country === 'object') {
					for (var g in window.pnv_preferred_dial_country) {
						if (Object.prototype.hasOwnProperty.call(window.pnv_preferred_dial_country, g)) {
							preferredDialCountry[g] = ('' + window.pnv_preferred_dial_country[g]).toLowerCase();
						}
					}
				}
			} catch (e) {
				/* ignore */
			}

			// Allow page scripts to programmatically modify the mapping via a global function.
			// If defined, the function should accept the current mapping object and return a (possibly modified) mapping object.
			try {
				if (typeof window.pnvModifyPreferredDialCountry === 'function') {
					var modified = window.pnvModifyPreferredDialCountry(Object.assign({}, preferredDialCountry));
					if (modified && typeof modified === 'object') {
						preferredDialCountry = modified;
					}
				}
			} catch (e) {
				/* ignore */
			}

			if (bestMatch.dialCode && preferredDialCountry[bestMatch.dialCode]) {
				return preferredDialCountry[bestMatch.dialCode];
			}

			return bestMatch.iso2;
		}

		// If a saved user phone exists, update wcPvJson.defaultCountry accordingly.
		if (wcPvJson.userPhone) {
			var userDefault = getDefaultCountryFromPhone(wcPvJson.userPhone);
			if (userDefault) {
				wcPvJson.defaultCountry = userDefault;
			}
		}

		// --- End Dynamic Default Country Setup ---

		// Store intlTelInput instances.
		let wcPvBillingPhoneIntl = null;
		let wcPvShippingPhoneIntl = null;

		// Flag to ensure the event listener is added only once.
		let placeOrderListenerAdded = false;

		// Touched flags to track user interaction.
		let billingPhoneTouched = false;
		let shippingPhoneTouched = false;

		// Blurred flags - used to ensure the global notice only appears after the field loses focus.
		let billingPhoneBlurred = false;
		let shippingPhoneBlurred = false;

		// Debounce timers so we only show field-level errors after user stops typing.
		let billingDebounceTimer = null;
		let shippingDebounceTimer = null;
		const PNV_DEBOUNCE_MS = 300;

		/**
		 * Function to validate phone number.
		 *
		 * @param {Object} phoneInstance - The intl-tel-input instance.
		 * @param {string} fieldType - 'Billing' or 'Shipping'.
		 * @param {boolean} isTouched - Whether the field has been interacted with.
		 * @param {string} inputValue - The current value of the phone input field.
		 * @returns {Object} - Contains formattedNumber and errorMessage.
		 */
		function wcPvValidatePhone(phoneInstance, fieldType, isTouched, inputValue) {
			let formattedNumber = false;
			let errorMessage = '';

			// If the input is empty, skip validation.
			if (inputValue.trim() === '') {
				return {
					formattedNumber: false,
					errorMessage: ''
				};
			}

			if (phoneInstance && phoneInstance.isValidNumber()) {
				// If separateDialCode is enabled we must NOT include the country dial code inside
				// the visible input. Use NATIONAL formatting for the visible input when separate
				// dial code is being used; otherwise keep the existing INTERNATIONAL format.
				if (wcPvJson && wcPvJson.separateDialCode) {
					formattedNumber = phoneInstance.getNumber(intlTelInputUtils.numberFormat.NATIONAL);
				} else {
					formattedNumber = phoneInstance.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL);
				}
			} else if (phoneInstance && isTouched) { // Only set error if touched.
				const validationError = phoneInstance.getValidationError();
				errorMessage = `${fieldType} phone ${
					wcPvJson.validationErrors[validationError] !== undefined
						? wcPvJson.validationErrors[validationError]
						: wcPvJson.phoneUnknownErrorMsg
				}`;
			}

			return {
				formattedNumber: formattedNumber,
				errorMessage: errorMessage
			};
		}

		/**
		 * Function to show a global error message styled like WooCommerce's error notices.
		 *
		 * @param {string} message - The error message to display.
		 */
		function showGlobalErrorMessage(message) {
			const noticesContainer = document.querySelector('.wc-block-components-notices');

			if (!noticesContainer) {
				console.error('Blocks: Notices container not found.');
				return;
			}

			let globalErrorNotice = noticesContainer.querySelector('.global-phone-validation-error');

			if (!globalErrorNotice) {
				globalErrorNotice = document.createElement('div');
				globalErrorNotice.className = 'wc-block-store-notice wc-block-components-notice-banner is-error is-dismissible global-phone-validation-error';

				const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				svg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
				svg.setAttribute('viewBox', '0 0 24 24');
				svg.setAttribute('width', '24');
				svg.setAttribute('height', '24');
				svg.setAttribute('aria-hidden', 'true');
				svg.setAttribute('focusable', 'false');

				const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				path.setAttribute('d', 'M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z');
				svg.appendChild(path);

				const contentDiv = document.createElement('div');
				contentDiv.className = 'wc-block-components-notice-banner__content';

				const messageDiv = document.createElement('div');
				messageDiv.textContent = message;

				contentDiv.appendChild(messageDiv);

				const dismissButton = document.createElement('button');
				dismissButton.className = 'wc-block-components-button wp-element-button wc-block-components-notice-banner__dismiss contained';
				dismissButton.setAttribute('aria-label', wcPvBlocksJson.dismissNotice);
				dismissButton.setAttribute('type', 'button');

				const dismissSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				dismissSvg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
				dismissSvg.setAttribute('viewBox', '0 0 24 24');
				dismissSvg.setAttribute('width', '24');
				dismissSvg.setAttribute('height', '24');
				dismissSvg.setAttribute('aria-hidden', 'true');
				dismissSvg.setAttribute('focusable', 'false');

				const dismissPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				dismissPath.setAttribute('d', 'M13 11.8l6.1-6.3-1-1-6.1 6.2-6.1-6.2-1 1 6.1 6.3-6.5 6.7 1 1 6.5-6.6 6.5 6.6 1-1z');
				dismissSvg.appendChild(dismissPath);

				dismissButton.appendChild(dismissSvg);

				globalErrorNotice.appendChild(svg);
				globalErrorNotice.appendChild(contentDiv);
				globalErrorNotice.appendChild(dismissButton);

				noticesContainer.appendChild(globalErrorNotice);

				globalErrorNotice.scrollIntoView({ behavior: 'smooth', block: 'start' });

				dismissButton.addEventListener('click', function () {
					globalErrorNotice.remove();
				});
			} else {
				const messageDiv = globalErrorNotice.querySelector('.wc-block-components-notice-banner__content div');
				if (messageDiv) {
					messageDiv.textContent = message;
				}
				globalErrorNotice.scrollIntoView({ behavior: 'smooth', block: 'start' });
			}
		}

		/**
		 * Function to update the state of the "Place Order" button.
		 */
		function updatePlaceOrderButton() {
			const placeOrderButton = document.querySelector('.wc-block-components-checkout-place-order-button');
			const billingError = document.querySelector('#billing .phone-validation-error');
			const shippingError = document.querySelector('#shipping .phone-validation-error');

			// Determine existence flags for inline errors.
			const billingErrorExists = !!billingError;
			const shippingErrorExists = !!shippingError;

			// Check the hidden error fields (these are only populated on blur).
			const billingHiddenError = document.querySelector('input[name="pnv_phone_error_billing"]');
			const shippingHiddenError = document.querySelector('input[name="pnv_phone_error_shipping"]');

			// Only show the global error notice when the hidden error field contains a message
			// and the user has blurred the relevant input. This prevents the global notice from
			// appearing while the user is still typing (debounced inline errors are still shown).
			const shouldShowGlobal =
				(billingHiddenError && billingHiddenError.value && billingPhoneBlurred) ||
				(shippingHiddenError && shippingHiddenError.value && shippingPhoneBlurred);

			if (placeOrderButton) {
				const alwaysAllowCheckout = wcPvJson.alwaysAllowCheckout !== undefined ? wcPvJson.alwaysAllowCheckout : wcPvBlocksJson.alwaysAllowCheckout;

				// If there are any field errors...
				if (billingErrorExists || shippingErrorExists) {
					// Only prevent checkout / show the global notice when the user has left the field (blur).
					if (!alwaysAllowCheckout && shouldShowGlobal) {
						placeOrderButton.setAttribute('disabled', 'disabled');
						placeOrderButton.style.opacity = '0.5';
						placeOrderButton.style.cursor = 'not-allowed';
						showGlobalErrorMessage(wcPvBlocksJson.pleaseCorrectPhone);
					} else {
						// While typing (errors present but no blur yet) keep button enabled so user can continue editing.
						placeOrderButton.removeAttribute('disabled');
						placeOrderButton.style.opacity = '1';
						placeOrderButton.style.cursor = 'pointer';
						removeGlobalErrorMessage();
					}
				} else {
					// No errors at all - enable button and remove any global notice.
					placeOrderButton.removeAttribute('disabled');
					placeOrderButton.style.opacity = '1';
					placeOrderButton.style.cursor = 'pointer';
					removeGlobalErrorMessage();
				}
			}
		}

		/**
		 * Function to remove the global error message.
		 */
		function removeGlobalErrorMessage() {
			const noticesContainer = document.querySelector('.wc-block-components-notices');
			if (noticesContainer) {
				const globalErrorNotice = noticesContainer.querySelector('.global-phone-validation-error');
				if (globalErrorNotice) {
					globalErrorNotice.remove();
				}
			}
		}

		/**
		 * Function to create and display error messages.
		 *
		 * @param {HTMLElement} phoneField - The phone input field.
		 * @param {string} errorMessage - The error message to display.
		 */
		function displayErrorMessage(phoneField, errorMessage) {
			phoneField.classList.add('has-error');
			phoneField.setAttribute('aria-invalid', 'true');

			let errorDiv = phoneField.parentNode.querySelector('.phone-validation-error');
			if (!errorDiv) {
				errorDiv = document.createElement('div');
				errorDiv.className = 'phone-validation-error';
				phoneField.parentNode.appendChild(errorDiv);

				let errorIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				errorIcon.setAttribute('width', '24');
				errorIcon.setAttribute('height', '24');
				errorIcon.setAttribute('viewBox', '-2 -2 24 24');
				errorIcon.setAttribute('fill', 'red');
				errorIcon.classList.add('error-icon');

				let path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				path.setAttribute('d', 'M10 2c4.42 0 8 3.58 8 8s-3.58 8-8 8-8-3.58-8-8 3.58-8 8-8zm1.13 9.38l.35-6.46H8.52l.35 6.46h2.26zm-.09 3.36c.24-.23.37-.55.37-.96 0-.42-.12-.74-.36-.97s-.59-.35-1.06-.35-.82.12-1.07.35-.37.55-.37.97c0 .41.13.73.38.96.26.23.61.34 1.06.34s.8-.11 1.05-.34z');
				errorIcon.appendChild(path);
				errorDiv.appendChild(errorIcon);

				let errorMessageNode = document.createElement('span');
				errorDiv.appendChild(errorMessageNode);
			}

			let errorMessageNode = errorDiv.querySelector('span');
			if (errorMessageNode) {
				errorMessageNode.textContent = errorMessage;
			}
		}

		/**
		 * Function to remove error messages.
		 *
		 * @param {HTMLElement} phoneField - The phone input field.
		 */
		function removeErrorMessage(phoneField) {
			phoneField.classList.remove('has-error');
			phoneField.setAttribute('aria-invalid', 'false');
			const errorDiv = phoneField.parentNode.querySelector('.phone-validation-error');
			if (errorDiv) {
				errorDiv.remove();
			}
		}

		/**
		 * Function to prevent checkout if there are validation errors.
		 */
		function preventInvalidCheckout() {
			if (placeOrderListenerAdded) {
				return;
			}

			const placeOrderButton = document.querySelector('.wc-block-components-checkout-place-order-button');
			if (placeOrderButton) {
				placeOrderButton.addEventListener(
					'click',
					function (event) {
						const billingError = document.querySelector('#billing .phone-validation-error');
						const shippingError = document.querySelector('#shipping .phone-validation-error');
						const alwaysAllowCheckout = wcPvJson.alwaysAllowCheckout !== undefined ? wcPvJson.alwaysAllowCheckout : wcPvBlocksJson.alwaysAllowCheckout;

						if ((billingError || shippingError) && !alwaysAllowCheckout) {
							event.preventDefault();
							event.stopPropagation();
							showGlobalErrorMessage(wcPvBlocksJson.pleaseCorrectPhone);
						} else {
							removeGlobalErrorMessage();
						}
					}
				);
				placeOrderListenerAdded = true;
			}
		}

		/**
		 * Function to handle country changes and update intl-tel-input instances.
		 */
		function handleCountryChange() {
			const billingCountrySelect = document.querySelector('.wc-block-checkout__billing-fields .wc-block-components-address-form__country select');
			const shippingCountrySelect = document.querySelector('.wc-block-checkout__shipping-fields .wc-block-components-address-form__country select');

			function updatePhoneCountry(selectElement, phoneInstance) {
				if (!selectElement || !phoneInstance) {
					return;
				}
				const selectedCountryCode = selectElement.value.toLowerCase();
				phoneInstance.setCountry(selectedCountryCode);
			}

			if (billingCountrySelect && !billingCountrySelect.classList.contains('pnv-country-listener-added')) {
				billingCountrySelect.classList.add('pnv-country-listener-added');
				billingCountrySelect.addEventListener(
					'change',
					function () {
						updatePhoneCountry(this, wcPvBillingPhoneIntl);
					}
				);
			}

			if (shippingCountrySelect && !shippingCountrySelect.classList.contains('pnv-country-listener-added')) {
				shippingCountrySelect.classList.add('pnv-country-listener-added');
				shippingCountrySelect.addEventListener(
					'change',
					function () {
						updatePhoneCountry(this, wcPvShippingPhoneIntl);
					}
				);
			}
		}

		/**
		 * Modified helper function to retrieve the selected country code.
		 * If a user phone exists, return the default country from wcPvJson.
		 *
		 * @param {string} formType - 'billing' or 'shipping'.
		 * @returns {string} - The selected country code in lowercase.
		 */
		function getSelectedCountryCode(formType) {
			// If a user phone exists, use the dynamic default.
			if (wcPvJson.userPhone) {
				return (wcPvJson.defaultCountry || 'us').toLowerCase();
			}

			const selector = formType === 'billing'
				? '.wc-block-checkout__billing-fields .wc-block-components-address-form__country select'
				: '.wc-block-checkout__shipping-fields .wc-block-components-address-form__country select';

			const countrySelect = document.querySelector(selector);
			if (countrySelect && countrySelect.value) {
				return countrySelect.value.toLowerCase();
			}

			// Fallback.
			return (wcPvJson.defaultCountry || 'us').toLowerCase();
		}

		/**
		 * Function to initialize phone fields in blocks.
		 */
		function initializeBlockPhoneFields() {
			const shippingPhone = document.querySelector('.wc-block-checkout__shipping-fields .wc-block-components-address-form__phone input[type="tel"]');
			const billingPhone = document.querySelector('.wc-block-checkout__billing-fields .wc-block-components-address-form__phone input[type="tel"]');

			// Initialize Billing Phone.
			if (billingPhone && !billingPhone.classList.contains('pnv-phone')) {
				billingPhone.classList.add('pnv-phone');
				const billingPhoneContainer = billingPhone.closest('.wc-block-components-address-form__phone');
				if (billingPhoneContainer) {
					billingPhoneContainer.classList.add('is-active');
				} else {
					console.warn('Blocks: Billing phone container not found.');
				}

				// Get initial country for billing.
				let billingInitialCountry = getSelectedCountryCode('billing');

				// If admin provided a blocks-specific placeholder, set it immediately so it's visible
				// before the utils script computes example numbers.
				try {
					if ( wcPvBlocksJson && wcPvBlocksJson.phonePlaceholder ) {
						billingPhone.setAttribute( 'placeholder', wcPvBlocksJson.phonePlaceholder );
					}
				} catch (e) {
					/* ignore */
				}

				wcPvBillingPhoneIntl = window.intlTelInput(
					billingPhone,
					{
						initialCountry: billingInitialCountry,
						separateDialCode: wcPvJson.separateDialCode,
						preferredCountries: wcPvJson.preferredCountries || [],
						utilsScript: wcPvJson.utilsScript,
						allowDropdown: wcPvJson.allowDropdown,
						excludeCountries: wcPvJson.excludeCountries || [],
						formatOnDisplay: false,
						customPlaceholder: function ( example, countryData ) {
							try {
								if ( wcPvBlocksJson && wcPvBlocksJson.phonePlaceholder ) {
									return wcPvBlocksJson.phonePlaceholder;
								}
								if ( typeof window.pnvCustomPhonePlaceholder === 'function' ) {
									var res = window.pnvCustomPhonePlaceholder( example, countryData );
									if ( typeof res === 'string' ) {
										return res;
									}
								}
							} catch (e) {
								/* ignore */
							}
							return example;
						}
					}
				);

				// If a logged-in user's phone was pre-populated in international format and
				// separateDialCode is enabled, convert the visible input to NATIONAL format so
				// the leading "+<countrycode>" is not shown inside the text field.
				try {
					if (wcPvJson && wcPvJson.separateDialCode && billingPhone && billingPhone.value && billingPhone.value.trim().charAt(0) === '+') {
						// Ensure the intl instance parses the current value.
						if (typeof wcPvBillingPhoneIntl.setNumber === 'function') {
							wcPvBillingPhoneIntl.setNumber(billingPhone.value);
						}
						if (typeof intlTelInputUtils !== 'undefined' && typeof wcPvBillingPhoneIntl.getNumber === 'function') {
							var billingNational = wcPvBillingPhoneIntl.getNumber(intlTelInputUtils.numberFormat.NATIONAL);
							if (billingNational && billingPhone.value !== billingNational) {
								try {
									const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
									nativeSetter.call(billingPhone, billingNational);
									billingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								} catch (e) {
									billingPhone.value = billingNational;
									billingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								}
							}
						}
					}
				} catch (e) {
					/* ignore */
				}

				const hiddenValidatorField = document.createElement('input');
				hiddenValidatorField.type = 'hidden';
				hiddenValidatorField.name = 'pnv_phone_valid_billing';
				billingPhone.parentNode.appendChild(hiddenValidatorField);

				const hiddenErrorField = document.createElement('input');
				hiddenErrorField.type = 'hidden';
				hiddenErrorField.name = 'pnv_phone_error_billing';
				billingPhone.parentNode.appendChild(hiddenErrorField);

				// Helper to run billing validation (used by debounce and on blur).
				// isBlur: boolean - when true we update hidden error field (used by blocks/global notices).
				function runBillingValidation(isBlur) {
					isBlur = !!isBlur;
					try {
						billingPhoneTouched = true;
						const inputValue = billingPhone.value;
						const validation = wcPvValidatePhone(wcPvBillingPhoneIntl, 'Billing', billingPhoneTouched, inputValue);

						// Always update valid hidden field (so formatted numbers are available).
						hiddenValidatorField.value = validation.formattedNumber || '';

						// Only update the hidden error field when this was triggered by blur,
						// otherwise keep it empty so blocks/global notice does not appear while typing.
						if (isBlur) {
							hiddenErrorField.value = validation.errorMessage;
						} else {
							// Clear any previously stored hidden error while the user is typing
							// to ensure the global notice is not triggered by other logic.
							hiddenErrorField.value = '';
							// Also proactively remove any global notice while the user is actively editing.
							removeGlobalErrorMessage();
						}

						// If we have a formatted international number, inject it into the visible input
						// and attempt to update React/WP Blocks internal state by using the native setter
						// and dispatching an input event (avoids React ignoring programmatic value changes).
						if (validation.formattedNumber) {
							// Only set/dispatch if the value actually differs to avoid infinite loops.
							if (billingPhone.value !== validation.formattedNumber) {
								try {
									const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
									nativeSetter.call(billingPhone, validation.formattedNumber);
									billingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								} catch (e) {
									// Fallback to simple assignment if native setter isn't available.
									billingPhone.value = validation.formattedNumber;
									billingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								}
							}
						}

						if (validation.errorMessage) {
							displayErrorMessage(billingPhone, validation.errorMessage);
						} else {
							removeErrorMessage(billingPhone);
						}
					} catch (e) {
						/* ignore */
					}
				}

				billingPhone.addEventListener('input', function () {
					billingPhoneTouched = true;
					if (billingDebounceTimer) {
						clearTimeout(billingDebounceTimer);
					}
					billingDebounceTimer = setTimeout(function () {
						// Run validation for inline feedback only (not as a blur).
						runBillingValidation(false);
						billingDebounceTimer = null;
					}, PNV_DEBOUNCE_MS);
				});

				billingPhone.addEventListener('blur', function () {
					billingPhoneTouched = true;
					billingPhoneBlurred = true;
					// If user was typing, run validation immediately and show notices now.
					if (billingDebounceTimer) {
						clearTimeout(billingDebounceTimer);
						billingDebounceTimer = null;
					}
					// Run validation as blur so hidden error is updated and global notices may appear.
					runBillingValidation(true);
					// Update global state after blur so the top notice only appears once focus is left.
					updatePlaceOrderButton();
				});
				billingPhone.addEventListener(
					'countrychange',
					function () {
						billingPhoneTouched = true;
						try {
							// If the input is empty, inject the selected country's dial code so it's visible/saved.
							// When separateDialCode is enabled we must NOT put the dial code inside the input field.
							if (billingPhone.value.trim() === '' && wcPvBillingPhoneIntl && typeof wcPvBillingPhoneIntl.getSelectedCountryData === 'function') {
								var data = wcPvBillingPhoneIntl.getSelectedCountryData();
								if (data && data.dialCode) {
									if (!wcPvJson || !wcPvJson.separateDialCode) {
										billingPhone.value = '+' + data.dialCode + ' ';
									} else {
										// Ensure the input remains empty when separate dial code is used.
										billingPhone.value = '';
									}
								}
							}
						} catch (e) {
							// Fail silently if intl-tel-input isn't available.
						}
						billingPhone.dispatchEvent(new Event('input'));
					}
				);
			}

			// Initialize Shipping Phone.
			if (shippingPhone && !shippingPhone.classList.contains('pnv-phone')) {
				shippingPhone.classList.add('pnv-phone');
				const shippingPhoneContainer = shippingPhone.closest('.wc-block-components-address-form__phone');
				if (shippingPhoneContainer) {
					shippingPhoneContainer.classList.add('is-active');
				} else {
					console.warn('Blocks: Shipping phone container not found.');
				}

				let shippingInitialCountry = getSelectedCountryCode('shipping');

				// If admin provided a blocks-specific placeholder, set it immediately so it's visible
				// before the utils script computes example numbers.
				try {
					if ( wcPvBlocksJson && wcPvBlocksJson.phonePlaceholder ) {
						shippingPhone.setAttribute( 'placeholder', wcPvBlocksJson.phonePlaceholder );
					}
				} catch (e) {
					/* ignore */
				}

				wcPvShippingPhoneIntl = window.intlTelInput(
					shippingPhone,
					{
						initialCountry: shippingInitialCountry,
						separateDialCode: wcPvJson.separateDialCode,
						preferredCountries: wcPvJson.preferredCountries || [],
						utilsScript: wcPvJson.utilsScript,
						allowDropdown: wcPvJson.allowDropdown,
						excludeCountries: wcPvJson.excludeCountries || [],
						formatOnDisplay: false,
						customPlaceholder: function ( example, countryData ) {
							try {
								if ( wcPvBlocksJson && wcPvBlocksJson.phonePlaceholder ) {
									return wcPvBlocksJson.phonePlaceholder;
								}
								if ( typeof window.pnvCustomPhonePlaceholder === 'function' ) {
									var res = window.pnvCustomPhonePlaceholder( example, countryData );
									if ( typeof res === 'string' ) {
										return res;
									}
								}
							} catch (e) {
								/* ignore */
							}
							return example;
						}
					}
				);

				// If a logged-in user's phone was pre-populated in international format and
				// separateDialCode is enabled, convert the visible input to NATIONAL format so
				// the leading "+<countrycode>" is not shown inside the text field.
				try {
					if (wcPvJson && wcPvJson.separateDialCode && shippingPhone && shippingPhone.value && shippingPhone.value.trim().charAt(0) === '+') {
						// Ensure the intl instance parses the current value.
						if (typeof wcPvShippingPhoneIntl.setNumber === 'function') {
							wcPvShippingPhoneIntl.setNumber(shippingPhone.value);
						}
						if (typeof intlTelInputUtils !== 'undefined' && typeof wcPvShippingPhoneIntl.getNumber === 'function') {
							var shippingNational = wcPvShippingPhoneIntl.getNumber(intlTelInputUtils.numberFormat.NATIONAL);
							if (shippingNational && shippingPhone.value !== shippingNational) {
								try {
									const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
									nativeSetter.call(shippingPhone, shippingNational);
									shippingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								} catch (e) {
									shippingPhone.value = shippingNational;
									shippingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								}
							}
						}
					}
				} catch (e) {
					/* ignore */
				}

				const hiddenValidatorField = document.createElement('input');
				hiddenValidatorField.type = 'hidden';
				hiddenValidatorField.name = 'pnv_phone_valid_shipping';
				shippingPhone.parentNode.appendChild(hiddenValidatorField);

				const hiddenErrorField = document.createElement('input');
				hiddenErrorField.type = 'hidden';
				hiddenErrorField.name = 'pnv_phone_error_shipping';
				shippingPhone.parentNode.appendChild(hiddenErrorField);

				// Helper to run shipping validation (used by debounce and on blur).
				// isBlur: boolean - when true we update hidden error field (used by blocks/global notices).
				function runShippingValidation(isBlur) {
					isBlur = !!isBlur;
					try {
						shippingPhoneTouched = true;
						const inputValue = shippingPhone.value;
						const validation = wcPvValidatePhone(wcPvShippingPhoneIntl, 'Shipping', shippingPhoneTouched, inputValue);

						// Always update valid hidden field (so formatted numbers are available).
						hiddenValidatorField.value = validation.formattedNumber || '';

						// Only update the hidden error field when this was triggered by blur,
						// otherwise keep it empty so blocks/global notice does not appear while typing.
						if (isBlur) {
							hiddenErrorField.value = validation.errorMessage;
						} else {
							// Clear any previously stored hidden error while the user is typing
							// to ensure the global notice is not triggered by other logic.
							hiddenErrorField.value = '';
							// Also proactively remove any global notice while the user is actively editing.
							removeGlobalErrorMessage();
						}

						// If we have a formatted international number, inject it into the visible input
						// and attempt to update React/WP Blocks internal state by using the native setter
						// and dispatching an input event (avoids React ignoring programmatic value changes).
						if (validation.formattedNumber) {
							// Only set/dispatch if the value actually differs to avoid infinite loops.
							if (shippingPhone.value !== validation.formattedNumber) {
								try {
									const nativeSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value').set;
									nativeSetter.call(shippingPhone, validation.formattedNumber);
									shippingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								} catch (e) {
									// Fallback to simple assignment if native setter isn't available.
									shippingPhone.value = validation.formattedNumber;
									shippingPhone.dispatchEvent(new Event('input', { bubbles: true }));
								}
							}
						}

						if (validation.errorMessage) {
							displayErrorMessage(shippingPhone, validation.errorMessage);
						} else {
							removeErrorMessage(shippingPhone);
						}
					} catch (e) {
						/* ignore */
					}
				}

				shippingPhone.addEventListener('input', function () {
					shippingPhoneTouched = true;
					if (shippingDebounceTimer) {
						clearTimeout(shippingDebounceTimer);
					}
					shippingDebounceTimer = setTimeout(function () {
						// Run validation for inline feedback only (not as a blur).
						runShippingValidation(false);
						shippingDebounceTimer = null;
					}, PNV_DEBOUNCE_MS);
				});

				shippingPhone.addEventListener('blur', function () {
					shippingPhoneTouched = true;
					shippingPhoneBlurred = true;
					// If user was typing, run validation immediately and show notices now.
					if (shippingDebounceTimer) {
						clearTimeout(shippingDebounceTimer);
						shippingDebounceTimer = null;
					}
					// Run validation as blur so hidden error is updated and global notices may appear.
					runShippingValidation(true);
					// Update global state after blur so the top notice only appears once focus is left.
					updatePlaceOrderButton();
				});
				shippingPhone.addEventListener(
					'countrychange',
					function () {
						shippingPhoneTouched = true;
						try {
							// If the input is empty, inject the selected country's dial code so it's visible/saved.
							// When separateDialCode is enabled we must NOT put the dial code inside the input field.
							if (shippingPhone.value.trim() === '' && wcPvShippingPhoneIntl && typeof wcPvShippingPhoneIntl.getSelectedCountryData === 'function') {
								var data = wcPvShippingPhoneIntl.getSelectedCountryData();
								if (data && data.dialCode) {
									if (!wcPvJson || !wcPvJson.separateDialCode) {
										shippingPhone.value = '+' + data.dialCode + ' ';
									} else {
										// Ensure the input remains empty when separate dial code is used.
										shippingPhone.value = '';
									}
								}
							}
						} catch (e) {
							// Fail silently if intl-tel-input isn't available.
						}
						shippingPhone.dispatchEvent(new Event('input'));
					}
				);
			}

			// Attach country change listeners.
			handleCountryChange();

			// Prevent invalid checkout.
			preventInvalidCheckout();
		}

		/**
		 * Function to create a mutation observer to watch for changes.
		 */
		function createMutationObserver() {
			const observer = new MutationObserver(
				(mutations) => {
					mutations.forEach(
						(mutation) => {
							if (mutation.addedNodes.length) {
								initializeBlockPhoneFields();
							}
						}
					);
				}
			);

			const checkoutForm = document.querySelector('.wp-block-woocommerce-checkout');
			if (checkoutForm) {
				observer.observe(
					checkoutForm,
					{
						childList: true,
						subtree: true
					}
				);
			}
		}

		// Initial check.
		initializeBlockPhoneFields();

		// Create mutation observer for dynamic changes.
		createMutationObserver();

		// Listen for checkout updates.
		document.addEventListener('wc-blocks-checkout-update-checkout', initializeBlockPhoneFields);
	}
);
