/**
 * Frontend Script
 *
 * @package    Phone_Number_Validation
 */

document.addEventListener('DOMContentLoaded', function () {

	'use strict';

	// Check if wcPvJson is defined.
	if (typeof wcPvJson === 'undefined') {
		console.error('wcPvJson is not defined.');
		return;
	}

	// Function to dynamically determine default country from a phone number.
	function getDefaultCountryFromPhone(phone) {
		// Ensure the phone number is in international format.
		if (!phone || phone.charAt(0) !== '+' || typeof window.intlTelInputGlobals === 'undefined' || !window.intlTelInputGlobals.getCountryData) {
			return null;
		}

		// Retrieve all country data from the intlTelInput library.
		var countryData = window.intlTelInputGlobals.getCountryData();
		var bestMatch = { dialCode: '', iso2: null };

		// Remove the '+' sign from the phone number.
		phone = phone.substring(1);

		// Loop over each country and find the one with the longest matching dial code.
		countryData.forEach(function(country) {
			var dialCode = country.dialCode;
			if (phone.indexOf(dialCode) === 0 && dialCode.length > bestMatch.dialCode.length) {
				bestMatch.dialCode = dialCode;
				bestMatch.iso2 = country.iso2;
			}
		});

		// For dial codes shared by multiple countries provide a preferred mapping.
		// This ensures ambiguous dial codes (e.g. +44) resolve to the expected main country flag.
		var preferredDialCountry = {
			'44': 'gb', // prefer Great Britain for +44 (over GG, JE, IM)
			'1': 'us',  // prefer United States for +1 (over CA, etc.)
			'7': 'ru'   // prefer Russia for +7 (over KZ)
		};

		// Merge server-provided mappings (from PHP localization).
		try {
			if (wcPvJson && wcPvJson.preferredDialCountry && typeof wcPvJson.preferredDialCountry === 'object') {
				for (var k in wcPvJson.preferredDialCountry) {
					if (Object.prototype.hasOwnProperty.call(wcPvJson.preferredDialCountry, k)) {
						preferredDialCountry[k] = ('' + wcPvJson.preferredDialCountry[k]).toLowerCase();
					}
				}
			}
			// Also allow blocks-specific localized JSON (wcPvBlocksJson) to provide mappings
			// in contexts where blocks-localization is used separately.
			if (typeof wcPvBlocksJson !== 'undefined' && wcPvBlocksJson && wcPvBlocksJson.preferredDialCountry && typeof wcPvBlocksJson.preferredDialCountry === 'object') {
				for (var kb in wcPvBlocksJson.preferredDialCountry) {
					if (Object.prototype.hasOwnProperty.call(wcPvBlocksJson.preferredDialCountry, kb)) {
						preferredDialCountry[kb] = ('' + wcPvBlocksJson.preferredDialCountry[kb]).toLowerCase();
					}
				}
			}
		} catch (e) {
			/* ignore */
		}

		// Merge any global JS-provided mappings (useful for runtime overrides without PHP changes).
		try {
			if (window.pnv_preferred_dial_country && typeof window.pnv_preferred_dial_country === 'object') {
				for (var g in window.pnv_preferred_dial_country) {
					if (Object.prototype.hasOwnProperty.call(window.pnv_preferred_dial_country, g)) {
						preferredDialCountry[g] = ('' + window.pnv_preferred_dial_country[g]).toLowerCase();
					}
				}
			}
		} catch (e) {
			/* ignore */
		}

		// Allow page scripts to programmatically modify the mapping via a global function.
		// If defined, the function should accept the current mapping object and return a (possibly modified) mapping object.
		try {
			if (typeof window.pnvModifyPreferredDialCountry === 'function') {
				var modified = window.pnvModifyPreferredDialCountry(Object.assign({}, preferredDialCountry));
				if (modified && typeof modified === 'object') {
					preferredDialCountry = modified;
				}
			}
		} catch (e) {
			/* ignore */
		}

		if (bestMatch.dialCode && preferredDialCountry[bestMatch.dialCode]) {
			return preferredDialCountry[bestMatch.dialCode];
		}

		return bestMatch.iso2;
	}

	// If a saved user phone exists, try to update the default country dynamically.
	if (wcPvJson.userPhone) {
		var userDefault = getDefaultCountryFromPhone(wcPvJson.userPhone);
		if (userDefault) {
			wcPvJson.defaultCountry = userDefault;
		}
	}

	// Retrieve validation error messages from localized JSON.
	const wcPvPhoneErrorMap = wcPvJson.validationErrors || {};

	// Select the billing and shipping phone fields.
	// Prefer the common ID used on checkout, but fall back to account/edit-address inputs
	// which may use name attributes rather than the same IDs.
	const billingPhoneField  = document.getElementById('billing_phone') || document.querySelector(`${wcPvJson.parentPage} input[name="billing_phone"]`);
	const shippingPhoneField = document.getElementById('shipping_phone') || document.querySelector(`${wcPvJson.parentPage} input[name="shipping_phone"]`);

	/**
	 * Initialize intlTelInput on a given phone field with provided options.
	 *
	 * @param {HTMLElement} phoneField - The phone input field element.
	 * @returns {Object|null} - Returns the intlTelInput instance or null if initialization fails.
	 */
	function initializeIntlTelInput(phoneField) {
		if (!phoneField) {
			return null;
		}

		// Store the original value before initializing intlTelInput.
		const originalValue = phoneField.value;

		// Determine initial country: prefer a country derived from the current input value
		// (useful when legacy shortcode populates the phone input but wcPvJson.userPhone is not set),
		// otherwise fall back to localized default.
		let initialCountry = (wcPvJson.defaultCountry === '' || typeof wcPvJson.defaultCountry === 'undefined') ? 'US' : wcPvJson.defaultCountry;
		try {
			if (originalValue && originalValue.trim().charAt(0) === '+') {
				const derived = getDefaultCountryFromPhone(originalValue);
				if (derived) {
					initialCountry = derived;
				}
			}
		} catch (e) {
			/* ignore */
		}

		// Optionally set a simple placeholder immediately from server-config so users
		// see it before intl-tel-input utils load and compute example numbers.
		try {
			if ( wcPvJson && wcPvJson.phonePlaceholder ) {
				phoneField.setAttribute( 'placeholder', wcPvJson.phonePlaceholder );
			}
		} catch (e) {
			/* ignore */
		}

		// Initialize or reuse existing intlTelInput instance.
		var intlInstance = null;
		try {
			// Try to detect an existing instance (modern intl-tel-input exposes getInstance on globals).
			if ( typeof window.intlTelInputGlobals !== 'undefined' && typeof window.intlTelInputGlobals.getInstance === 'function' ) {
				try {
					intlInstance = window.intlTelInputGlobals.getInstance( phoneField ) || null;
				} catch (e) {
					intlInstance = null;
				}
			}
		} catch (e) {
			intlInstance = null;
		}

		// If no existing instance, create a new one.
		if ( ! intlInstance ) {
			intlInstance = window.intlTelInput(
				phoneField,
				{
					initialCountry: initialCountry,
					separateDialCode: wcPvJson.separateDialCode,
					preferredCountries: wcPvJson.preferredCountries && wcPvJson.preferredCountries.length > 0 ? wcPvJson.preferredCountries.map(country => country.toUpperCase()) : [],
					utilsScript: wcPvJson.utilsScript,
					allowDropdown: wcPvJson.allowDropdown,
					excludeCountries: wcPvJson.excludeCountries && wcPvJson.excludeCountries.length > 0 ? wcPvJson.excludeCountries.map(country => country.toUpperCase()) : [],
					// Prevent auto-formatting.
					nationalMode: true,
					formatOnDisplay: false,
					/**
					 * Provide a customPlaceholder function so we can override the placeholder
					 * produced by intl-tel-input (example numbers). Priority:
					 * 1) wcPvJson.phonePlaceholder (server-side setting / filter)
					 * 2) window.pnvCustomPhonePlaceholder( examplePlaceholder, countryData ) if defined
					 * 3) default example placeholder provided by the library
					 */
					customPlaceholder: function ( example, countryData ) {
						try {
							if ( wcPvJson && wcPvJson.phonePlaceholder ) {
								return wcPvJson.phonePlaceholder;
							}
							if ( typeof window.pnvCustomPhonePlaceholder === 'function' ) {
								var res = window.pnvCustomPhonePlaceholder( example, countryData );
								if ( typeof res === 'string' ) {
									return res;
								}
							}
						} catch (e) {
							/* ignore */
						}
						return example;
					},
				}
			);
		}

		// Ensure derived initial country is applied after initialization (fixes some rendering edge-cases,
		// for example My Account edit-address where the options-based initialCountry may not be honored).
		try {
			if (originalValue && originalValue.trim().charAt(0) === '+') {
				var _derived = null;
				try {
					_derived = getDefaultCountryFromPhone(originalValue);
				} catch (e) {
					_derived = null;
				}
				if (_derived && intlInstance && typeof intlInstance.setCountry === 'function') {
					try {
						intlInstance.setCountry(_derived);
					} catch (e) {
						/* ignore */
					}
				}

				// Defensive: re-apply the derived country after short delays so it wins
				// over other scripts that might override the country shortly after init
				// (observed on some My Account edit-address render flows). Also attempt to
				// use the globals.getInstance path if available.
				(function() {
					function applyDerivedCountryToInstance(instance, country) {
						if (!country || !instance) {
							return false;
						}
						try {
							if (typeof instance.setCountry === 'function') {
								instance.setCountry(country);
								return true;
							}
						} catch (e) {
							/* ignore */
						}
						return false;
					}

					function getMaybeInstance() {
						try {
							if (typeof window.intlTelInputGlobals !== 'undefined' && typeof window.intlTelInputGlobals.getInstance === 'function') {
								try {
									return window.intlTelInputGlobals.getInstance(phoneField) || intlInstance;
								} catch (e) {
									return intlInstance;
								}
							}
						} catch (e) {
							/* ignore */
						}
						return intlInstance;
					}

					try {
						if (originalValue && originalValue.trim().charAt(0) === '+') {
							var attemptDelays = [250, 750, 1500];
							attemptDelays.forEach(function(delay) {
								try {
									setTimeout(function() {
										try {
											var mi = getMaybeInstance();
											if (_derived && mi) {
												applyDerivedCountryToInstance(mi, _derived);
											}
										} catch (e) {
											/* ignore */
										}
									}, delay);
								} catch (e) {
									/* ignore */
								}
							});

							// Final fallback: after a longer delay, if the selected flag still isn't the expected ISO,
							// update the intl-tel-input DOM selected flag as a last-resort visual fix.
							try {
								setTimeout(function () {
									try {
										var finalInstance = getMaybeInstance();
										var selected = finalInstance && typeof finalInstance.getSelectedCountryData === 'function' ? finalInstance.getSelectedCountryData() : null;
										// If instance exists and selected iso isn't our derived value, attempt to set it and then patch DOM.
										if (_derived && finalInstance && selected && selected.iso2 !== _derived) {
											applyDerivedCountryToInstance(finalInstance, _derived);
										}
										// DOM fallback: find the selected-flag element and adjust classes/title/dial code.
										try {
											var wrapper = phoneField.closest('.iti');
											if (wrapper) {
												var selectedFlag = wrapper.querySelector('.iti__selected-flag');
												if (selectedFlag) {
													// Update flag class.
													var flagElem = selectedFlag.querySelector('.iti__flag');
													if (flagElem) {
														// Remove any existing iti__xx classes and add derived one.
														// Class format: iti__<iso2>
														Array.prototype.slice.call(flagElem.classList).forEach(function(c) {
															if (c.indexOf('iti__') === 0 && c.length === 6) {
																try { flagElem.classList.remove(c); } catch (e) { /* ignore */ }
															}
														});
														try { flagElem.classList.add('iti__' + _derived); } catch (e) { /* ignore */ }
													}
													// Update title and selected dial code if present.
													try {
														var countryName = (function(iso){ try { var cd = window.intlTelInputGlobals && window.intlTelInputGlobals.getCountryData ? window.intlTelInputGlobals.getCountryData() : []; for (var i=0;i<cd.length;i++){ if (cd[i].iso2 === iso) return cd[i].name; } } catch (e){} return ''; })(_derived);
														var dialCode = (function(iso){ try { var cd = window.intlTelInputGlobals && window.intlTelInputGlobals.getCountryData ? window.intlTelInputGlobals.getCountryData() : []; for (var i=0;i<cd.length;i++){ if (cd[i].iso2 === iso) return cd[i].dialCode; } } catch (e){} return ''; })(_derived);
														if (countryName && dialCode) {
															try {
																selectedFlag.setAttribute('title', countryName + ': +' + dialCode);
															} catch (e) { /* ignore */ }
														}
														var dialCodeElem = wrapper.querySelector('.iti__selected-dial-code');
														if (dialCodeElem && dialCode) {
															try { dialCodeElem.textContent = '+' + dialCode; } catch (e) { /* ignore */ }
														}
													} catch (e) { /* ignore */ }
												}
											}
										} catch (e) {
											/* ignore DOM fallback errors */
										}
									} catch (e) {
										/* ignore */
									}
								}, 2000);
							} catch (e) {
								/* ignore */
							}
						}
					} catch (e) {
						/* ignore */
					}
				})();
			}
		} catch (e) {
			/* ignore */
		}

		// Restore the original value to prevent auto-formatting.
		setTimeout(() => {
			phoneField.value = originalValue;
		}, 0);

		return intlInstance;
	}

	// Initialize intlTelInput on billing_phone and shipping_phone.
	let wcPvBillingPhoneIntl  = initializeIntlTelInput(billingPhoneField);
	let wcPvShippingPhoneIntl = initializeIntlTelInput(shippingPhoneField);

	// Check if intlTelInput is loaded properly.
	if (wcPvBillingPhoneIntl && typeof wcPvBillingPhoneIntl.isValidNumber !== 'function') {
		console.error('intlTelInput is not fully loaded. utilsScript might be missing.');
	}
	if (wcPvShippingPhoneIntl && typeof wcPvShippingPhoneIntl.isValidNumber !== 'function') {
		console.error('intlTelInput is not fully loaded. utilsScript might be missing.');
	}

	// Separate variables to store phone error messages.
	let wcPvPhoneErrorMsgBilling  = '';
	let wcPvPhoneErrorMsgShipping = '';

	/**
	 * Validates the phone number using intlTelInput plugin.
	 *
	 * @param {Object} phoneInstance - The intlTelInput instance.
	 * @param {string} fieldType - 'Billing' or 'Shipping'.
	 * @returns {Object} - Returns an object with formattedNumber and errorMessage.
	 */
	function wcPvValidatePhone(phoneInstance, fieldType) {
		let formattedNumber = false;
		let errorMessage    = '';

		if (phoneInstance && phoneInstance.isValidNumber()) {
			// Use INTERNATIONAL format to include space between country code and number.
			formattedNumber = phoneInstance.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL);
		} else if (phoneInstance) {
			const validationError = phoneInstance.getValidationError();
			errorMessage          = `${fieldType} phone ${wcPvPhoneErrorMap[validationError] !== undefined ? wcPvPhoneErrorMap[validationError] : wcPvJson.phoneUnknownErrorMsg}`;
		}

		return {
			formattedNumber: formattedNumber,
			errorMessage: errorMessage
		};
	}

	/**
	 * Processes the phone validation and updates the form accordingly.
	 *
	 * @param {HTMLElement} formContainer - The form container element.
	 * @returns {Object} - Returns an object indicating if there was an error.
	 */
	function wcPvValidateProcess(formContainer) {
		let isShippingAddress = false;
		if (wcPvJson.currentPage === 'checkout') {
			const shipToDifferentAddressCheckbox = document.getElementById('ship-to-different-address-checkbox');
			isShippingAddress = shipToDifferentAddressCheckbox && shipToDifferentAddressCheckbox.checked;
		} else if (wcPvJson.currentPage === 'account') {
			// For the account page, always validate shipping phone if shipping field is present.
			isShippingAddress = (shippingPhoneField !== null);
		}

		const billingValidation  = wcPvBillingPhoneIntl ? wcPvValidatePhone(wcPvBillingPhoneIntl, 'Billing') : { formattedNumber: false, errorMessage: '' };
		const shippingValidation = wcPvShippingPhoneIntl && isShippingAddress ? wcPvValidatePhone(wcPvShippingPhoneIntl, 'Shipping') : { formattedNumber: false, errorMessage: '' };

		// Reset error messages.
		wcPvPhoneErrorMsgBilling  = '';
		wcPvPhoneErrorMsgShipping = '';

		// Handle billing phone validation.
		if (billingPhoneField) {
			// Remove existing billing error message if any.
			const existingBillingError = document.getElementById('wc-ls-phone-valid-field-err-msg-billing');
			if (existingBillingError) {
				existingBillingError.remove();
			}

			// Remove existing billing valid phone field if any.
			const existingBillingValid = document.getElementById('wc-ls-phone-valid-field-billing');
			if (existingBillingValid) {
				existingBillingValid.remove();
			}

			if (billingValidation.formattedNumber) {
				// Set the validated phone number in the billing phone input.
				billingPhoneField.value = billingValidation.formattedNumber;

				// Add a hidden input with the validated phone number if it doesn't exist.
				if (!document.getElementById('wc-ls-phone-valid-field-billing')) {
					const hiddenInput = document.createElement('input');
					hiddenInput.type  = 'hidden';
					hiddenInput.id    = 'wc-ls-phone-valid-field-billing';
					hiddenInput.name  = 'pnv_phone_valid_billing';
					hiddenInput.value = billingValidation.formattedNumber;
					formContainer.appendChild(hiddenInput);
				}
			} else {
				// Set the billing error message.
				wcPvPhoneErrorMsgBilling = billingValidation.errorMessage;

				// Append an error message as a hidden input.
				const errorInput = document.createElement('input');
				errorInput.type  = 'hidden';
				errorInput.id    = 'wc-ls-phone-valid-field-err-msg-billing';
				errorInput.name  = 'pnv_phone_error_billing';
				errorInput.value = wcPvPhoneErrorMsgBilling;
				formContainer.appendChild(errorInput);

				// Ensure valid phone field is removed if present.
				const validPhoneField = document.getElementById('wc-ls-phone-valid-field-billing');
				if (validPhoneField) {
					validPhoneField.remove();
				}
			}
		}

		// Handle shipping phone validation only if shipping address is used.
		if (shippingPhoneField && isShippingAddress) {
			// Remove existing shipping error message if any.
			const existingShippingError = document.getElementById('wc-ls-phone-valid-field-err-msg-shipping');
			if (existingShippingError) {
				existingShippingError.remove();
			}

			// Remove existing shipping valid phone field if any.
			const existingShippingValid = document.getElementById('wc-ls-phone-valid-field-shipping');
			if (existingShippingValid) {
				existingShippingValid.remove();
			}

			if (shippingValidation.formattedNumber) {
				// Set the validated phone number in the shipping phone input.
				shippingPhoneField.value = shippingValidation.formattedNumber;

				// Add a hidden input with the validated phone number if it doesn't exist.
				if (!document.getElementById('wc-ls-phone-valid-field-shipping')) {
					const hiddenInput = document.createElement('input');
					hiddenInput.type  = 'hidden';
					hiddenInput.id    = 'wc-ls-phone-valid-field-shipping';
					hiddenInput.name  = 'pnv_phone_valid_shipping';
					hiddenInput.value = shippingValidation.formattedNumber;
					formContainer.appendChild(hiddenInput);
				}
			} else {
				// Set the shipping error message.
				wcPvPhoneErrorMsgShipping = shippingValidation.errorMessage;

				// Append an error message as a hidden input.
				const errorInput = document.createElement('input');
				errorInput.type  = 'hidden';
				errorInput.id    = 'wc-ls-phone-valid-field-err-msg-shipping';
				errorInput.name  = 'pnv_phone_error_shipping';
				errorInput.value = wcPvPhoneErrorMsgShipping;
				formContainer.appendChild(errorInput);

				// Ensure valid phone field is removed if present.
				const validPhoneField = document.getElementById('wc-ls-phone-valid-field-shipping');
				if (validPhoneField) {
					validPhoneField.remove();
				}
			}
		}

		let hasError = false;
		if (wcPvPhoneErrorMsgBilling !== '' || wcPvPhoneErrorMsgShipping !== '') {
			hasError = true;
		}

		return { hasError: hasError };
	}

	/**
	 * Display inline error messages on My Account page if validation fails.
	 *
	 * @param {string} billingError - The billing error message (if any).
	 * @param {string} shippingError - The shipping error message (if any).
	 */
	function wcPvShowInlineErrors(billingError, shippingError) {
		// Remove any existing notices.
		const existingNotices = document.querySelectorAll('.woocommerce-NoticeGroup-checkout');
		existingNotices.forEach(function (notice) {
			notice.remove();
		});

		// Create notice container.
		const noticeGroup = document.createElement('div');
		noticeGroup.className = 'woocommerce-NoticeGroup woocommerce-NoticeGroup-checkout';

		const ul = document.createElement('ul');
		ul.className = 'woocommerce-error';
		ul.setAttribute('role', 'alert');

		// Add billing error if present.
		if (billingError !== '') {
			const li = document.createElement('li');
			li.textContent = billingError;
			ul.appendChild(li);
		}

		// Add shipping error if present.
		if (shippingError !== '') {
			const li = document.createElement('li');
			li.textContent = shippingError;
			ul.appendChild(li);
		}

		// Append only if there's at least one error message.
		if (ul.children.length > 0) {
			noticeGroup.appendChild(ul);

			// Insert the error messages at the top of the container.
			const accountContainer = document.querySelector(wcPvJson.parentPage);
			if (accountContainer) {
				accountContainer.insertBefore(noticeGroup, accountContainer.firstChild);
			}

			// Scroll to top to show the error notices.
			window.scrollTo({ top: 0, behavior: 'smooth' });
		}
	}

	/**
	 * Event listener for changes in the billing country select field.
	 */
	const billingCountrySelect = document.querySelector(`${wcPvJson.parentPage} #billing_country`);

	// If no saved user phone is present, then update the intlTelInput from the billing select on page load.
	if (!wcPvJson.userPhone && billingCountrySelect && wcPvBillingPhoneIntl) {
		wcPvBillingPhoneIntl.setCountry(billingCountrySelect.value);
	}

	if (billingCountrySelect) {
		billingCountrySelect.addEventListener('change', function () {
			const selectedCountry = this.value;
			// Only override if no saved user phone (i.e. user phone has priority).
			if (!wcPvJson.userPhone && (wcPvJson.onlyCountries.length === 0 || wcPvJson.onlyCountries.includes(selectedCountry.toUpperCase()))) {
				if (wcPvBillingPhoneIntl) {
					wcPvBillingPhoneIntl.setCountry(selectedCountry);
				}
			}
		});
	}

	// Similar logic for shipping country.
	const shippingCountrySelect = document.querySelector(`${wcPvJson.parentPage} #shipping_country`);

	// If no saved user phone is present, then update the intlTelInput from the shipping select on page load.
	if (!wcPvJson.userPhone && shippingCountrySelect && wcPvShippingPhoneIntl) {
		wcPvShippingPhoneIntl.setCountry(shippingCountrySelect.value);
	}

	if (shippingCountrySelect) {
		shippingCountrySelect.addEventListener('change', function () {
			const selectedCountry = this.value;
			// Only override if no saved user phone.
			if (!wcPvJson.userPhone && (wcPvJson.onlyCountries.length === 0 || wcPvJson.onlyCountries.includes(selectedCountry.toUpperCase()))) {
				if (wcPvShippingPhoneIntl) {
					wcPvShippingPhoneIntl.setCountry(selectedCountry);
				}
			}
		});
	}

	/**
	 * Adds a specific class to phone fields if separate dial code is enabled.
	 */
	if (wcPvJson.separateDialCode === true) {
		if (billingPhoneField) {
			billingPhoneField.classList.add('pnv-separate-dial-code');
		}
		if (shippingPhoneField) {
			shippingPhoneField.classList.add('pnv-separate-dial-code');
		}
	}

	/**
	 * Determines the current page and binds appropriate event listeners.
	 */
	if (wcPvJson.currentPage === 'checkout') {
		const checkoutForm = document.querySelector(`${wcPvJson.parentPage} form.checkout`);
		if (checkoutForm) {
			if (window.jQuery) {
				// Validate and rely on server-side for displaying errors.
				window.jQuery(checkoutForm).on('checkout_place_order', function () {
					wcPvValidateProcess(checkoutForm);
				});
			} else {
				checkoutForm.addEventListener('submit', function () {
					wcPvValidateProcess(checkoutForm);
				});
			}
		}
	} else if (wcPvJson.currentPage === 'account') {
		const accountForm = document.querySelector(`${wcPvJson.parentPage} form`);
		if (accountForm) {

			// Ensure hidden validator inputs exist inside the account form so they are submitted.
			function ensureHiddenAccountFields() {
				if ( ! accountForm.querySelector('input[name="pnv_phone_valid_billing"]') ) {
					const hid = document.createElement('input');
					hid.type = 'hidden';
					hid.name = 'pnv_phone_valid_billing';
					accountForm.appendChild(hid);
				}
				if ( ! accountForm.querySelector('input[name="pnv_phone_error_billing"]') ) {
					const hid = document.createElement('input');
					hid.type = 'hidden';
					hid.name = 'pnv_phone_error_billing';
					accountForm.appendChild(hid);
				}
				if ( ! accountForm.querySelector('input[name="pnv_phone_valid_shipping"]') ) {
					const hid = document.createElement('input');
					hid.type = 'hidden';
					hid.name = 'pnv_phone_valid_shipping';
					accountForm.appendChild(hid);
				}
				if ( ! accountForm.querySelector('input[name="pnv_phone_error_shipping"]') ) {
					const hid = document.createElement('input');
					hid.type = 'hidden';
					hid.name = 'pnv_phone_error_shipping';
					accountForm.appendChild(hid);
				}
			}

			// Update hidden fields from intl instances or input values.
			function syncAccountHiddenFields() {
				try {
					const vb = accountForm.querySelector('input[name="pnv_phone_valid_billing"]');
					const eb = accountForm.querySelector('input[name="pnv_phone_error_billing"]');
					const vs = accountForm.querySelector('input[name="pnv_phone_valid_shipping"]');
					const es = accountForm.querySelector('input[name="pnv_phone_error_shipping"]');

					if (vb) {
						if (wcPvBillingPhoneIntl && typeof wcPvBillingPhoneIntl.isValidNumber === 'function') {
							const v = wcPvBillingPhoneIntl.isValidNumber() ? wcPvBillingPhoneIntl.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL) : '';
							vb.value = v;
							if (eb) {
								const err = wcPvBillingPhoneIntl && !v ? wcPvBillingPhoneIntl.getValidationError() : '';
								eb.value = err ? wcPvPhoneErrorMap[err] || wcPvJson.phoneUnknownErrorMsg : '';
							}
						} else if (billingPhoneField) {
							vb.value = billingPhoneField.value || '';
						}
					}

					if (vs) {
						if (wcPvShippingPhoneIntl && typeof wcPvShippingPhoneIntl.isValidNumber === 'function') {
							const v = wcPvShippingPhoneIntl.isValidNumber() ? wcPvShippingPhoneIntl.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL) : '';
							vs.value = v;
							if (es) {
								const err = wcPvShippingPhoneIntl && !v ? wcPvShippingPhoneIntl.getValidationError() : '';
								es.value = err ? wcPvPhoneErrorMap[err] || wcPvJson.phoneUnknownErrorMsg : '';
							}
						} else if (shippingPhoneField) {
							vs.value = shippingPhoneField.value || '';
						}
					}

				} catch (e) {
					/* ignore */
				}
			}

			ensureHiddenAccountFields();

			// Keep hidden fields up-to-date while user edits (best-effort).
			if (billingPhoneField) {
				billingPhoneField.addEventListener('input', function () {
					// small debounce not needed here but keep simple sync
					syncAccountHiddenFields();
				});
				billingPhoneField.addEventListener('change', function () {
					syncAccountHiddenFields();
				});
			}
			if (shippingPhoneField) {
				shippingPhoneField.addEventListener('input', function () {
					syncAccountHiddenFields();
				});
				shippingPhoneField.addEventListener('change', function () {
					syncAccountHiddenFields();
				});
			}

			// On submit, run validation and ensure hidden fields have current values.
			accountForm.addEventListener('submit', function (event) {
				// run validation to populate inline errors / formatted values
				const result = wcPvValidateProcess(accountForm);

				// Always sync hidden fields just before submit so server receives them.
				syncAccountHiddenFields();

				// If errors are present and alwaysAllowCheckout is false, prevent submission.
				if (result.hasError && !wcPvJson.alwaysAllowCheckout) {
					event.preventDefault();
					// Show inline error messages on my-account page.
					wcPvShowInlineErrors(wcPvPhoneErrorMsgBilling, wcPvPhoneErrorMsgShipping);
				}
			});
		}
	}

});
