<?php
/**
 * Loading all plugin classes.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes
 */

namespace PNV\Phone_Number_Validation;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Loader
 *
 * Autoloads classes in the plugin.
 */
class Loader {

	/**
	 * Singleton instance.
	 *
	 * @var Loader|null
	 */
	private static $instance = null;

	/**
	 * Get the singleton instance.
	 *
	 * @return Loader
	 */
	public static function get_instance(): self {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialize the loader.
	 */
	public function init() {
		spl_autoload_register( array( $this, 'autoload' ) );

		// Initialize the main plugin class.
		Phone_Number_Validation::get_instance();
	}

	/**
	 * Autoload classes based on namespace and class name.
	 *
	 * @param string $classname Fully qualified class name.
	 */
	private function autoload( string $classname ): void {
		$prefix   = 'PNV\\Phone_Number_Validation\\';
		$base_dir = PNV_PLUGIN_DIR . 'includes/';

		$len = strlen( $prefix );
		if ( strncmp( $prefix, $classname, $len ) !== 0 ) {
			// Not a class from this plugin.
			return;
		}

		$relative_class = substr( $classname, $len );

		// Validate the class name format.
		if ( ! preg_match( '/^[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*$/', $relative_class ) ) {
			return;
		}

		// Handle sub-namespaces like Admin.
		if ( strpos( $relative_class, '\\' ) !== false ) {
			$parts = explode( '\\', $relative_class );
			// Validate each namespace part.
			foreach ( $parts as $part ) {
				if ( ! preg_match( '/^[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*$/', $part ) ) {
					return;
				}
			}
			$relative_class = end( $parts );
			$sub_folder     = strtolower( $parts[0] ) . '/';
		} else {
			$sub_folder = '';
		}

		// Create safe filename with WordPress sanitization.
		$file_name = sanitize_file_name( 'class-' . strtolower( str_replace( '_', '-', $relative_class ) ) . '.php' );

		// Use WordPress path sanitization and directory traversal prevention.
		$sub_folder = trailingslashit( sanitize_file_name( $sub_folder ) );

		// Construct and validate the path using WordPress functions.
		$file_path = wp_normalize_path( $base_dir . $sub_folder . $file_name );
		$base_path = wp_normalize_path( untrailingslashit( $base_dir ) );

		// Ensure the path is within our plugin directory.
		if ( 0 !== strpos( $file_path, $base_path ) ) {
			return;
		}

		// Final validation before including.
		if ( validate_file( str_replace( $base_path, '', $file_path ) ) === 0 && file_exists( $file_path ) ) {
			require_once wp_normalize_path( $file_path );
		}
	}
}
