<?php
/**
 * The main plugin logic.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes
 */

namespace PNV\Phone_Number_Validation;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use WC_Logger;
use Automattic\WooCommerce\Utilities\FeaturesUtil;

/**
 * Class Phone_Number_Validation
 *
 * Main plugin class.
 */
class Phone_Number_Validation {

	/**
	 * Singleton instance.
	 *
	 * @var Phone_Number_Validation|null
	 */
	private static $_instance = null; // phpcs:ignore

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $woo_custom_field_meta = array(
		'billing_hidden_phone_field'     => '_pnv_phone_validator',
		'billing_hidden_phone_err_field' => '_pnv_phone_validator_err',
	);

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	private $version = PNV_PLUGIN_VERSION;

	/**
	 * Settings instance.
	 *
	 * @var Admin\Settings
	 */
	private $settings;

	/**
	 * Constructor.
	 */
	private function __construct() {
		if ( Dependencies::is_woocommerce_active() ) {
			$this->define_constants();
			$this->includes();
			$this->init_hooks();
		} else {
			add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
		}
	}

	/**
	 * Get the singleton instance.
	 *
	 * @return Phone_Number_Validation
	 */
	public static function get_instance(): self {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Define additional constants.
	 */
	private function define_constants() {
		if ( ! defined( 'PNV_ABSPATH' ) ) {
			define( 'PNV_ABSPATH', plugin_dir_path( PNV_PLUGIN_FILE ) . '/' );
		}
		if ( ! defined( 'PNV_PLUGIN_BASE' ) ) {
			define( 'PNV_PLUGIN_BASE', plugin_basename( PNV_PLUGIN_FILE ) );
		}
		if ( ! defined( 'PNV_ASSETS_URL' ) ) {
			define( 'PNV_ASSETS_URL', plugins_url( 'assets/', PNV_PLUGIN_FILE ) );
		}
	}

	/**
	 * Include necessary files.
	 */
	private function includes() {
		// Frontend classes.
		require_once PNV_ABSPATH . 'public/class-checkout.php';
		require_once PNV_ABSPATH . 'public/class-my-account.php';

		// Admin classes.
		require_once PNV_ABSPATH . 'includes/admin/class-settings.php';

		// Initialize frontend classes.
		new Frontend\Checkout( $this );
		new Frontend\My_Account( $this );

		// Initialize admin classes.
		$this->settings = Admin\Settings::get_instance();
	}

	/**
	 * Initialize plugin hooks.
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'load_textdomain' ) );
		do_action( 'pnv_init' );

		// Declare HPOS and block compatibility before WooCommerce initializes.
		add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ), 10 );
		add_action( 'before_woocommerce_init', array( $this, 'declare_block_compatibility' ), 10 );

		// Use woocommerce_checkout_process for server-side validation.
		add_action( 'woocommerce_checkout_process', array( $this, 'checkout_validate' ) );

		// Validate admin-created orders when saving a shop_order in the admin area.
		// Use a later priority so our persistence runs after other save handlers and
		// can override the final saved billing/shipping phone values with the
		// validated international-format values produced by the admin JS.
		add_action( 'save_post_shop_order', array( $this, 'admin_order_validate' ), 20, 3 );

		// Also hook into WooCommerce's admin order processing action which runs during
		// the admin order edit flow. For legacy (postmeta) orders this ensures we
		// persist the validated international phone values after WooCommerce's own
		// processing; set a very late priority to act as the final override.
		add_action( 'woocommerce_process_shop_order_meta', array( $this, 'admin_order_persist_validated_phones' ), 999, 2 );

		// Append error information to redirect after saving so we can show an admin notice.
		add_filter( 'redirect_post_location', array( $this, 'admin_order_redirect_with_error' ), 10, 2 );

		// Display admin notices for order validation errors.
		add_action( 'admin_notices', array( $this, 'show_admin_order_error_notice' ), 15 );

		// Add plugin action links.
		add_filter( 'plugin_action_links_' . PNV_PLUGIN_BASE, array( $this, 'add_plugin_action_links' ) );

		// Add admin notice for block checkout phone field requirement.
		add_action( 'admin_notices', array( $this, 'check_phone_field_requirement' ) );
	}

	/**
	 * Declare HPOS compatibility.
	 */
	public function declare_hpos_compatibility() {
		if ( class_exists( FeaturesUtil::class ) ) {
			FeaturesUtil::declare_compatibility( 'custom_order_tables', 'phone-number-validation/phone-number-validation.php', true );
		}
	}

	/**
	 * Declare Cart/Checkout Block compatibility.
	 */
	public function declare_block_compatibility() {
		if ( class_exists( FeaturesUtil::class ) ) {
			FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', 'phone-number-validation/phone-number-validation.php', true );
		}
	}

	/**
	 * Load the plugin textdomain for translations.
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'phone-number-validation', false, dirname( plugin_basename( PNV_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Display admin notices if dependencies are missing.
	 */
	public function admin_notices() {
		echo '<div class="error"><p>';
		echo wp_kses_post(
			sprintf(
				/* translators: %s: WooCommerce plugin URL */
				__( 'The <strong>Phone Number Validation</strong> plugin requires <a href="%s" target="_blank">WooCommerce</a> to be active. Please ensure it is installed and activated.', 'phone-number-validation' ),
				'https://wordpress.org/plugins/woocommerce/'
			)
		);
		echo '</p></div>';
	}

	/**
	 * Check if we're on the checkout page.
	 *
	 * @return bool
	 */
	public function is_checkout(): bool {
		$id = get_option( 'woocommerce_checkout_page_id', false );
		return is_page( $id );
	}

	/**
	 * Check if we're on the my account page.
	 *
	 * @return bool
	 */
	public function is_account_page(): bool {
		$id = get_option( 'woocommerce_myaccount_page_id', false );
		return is_page( $id );
	}

	/**
	 * Get validation errors.
	 *
	 * @return array
	 */
	public function get_validation_errors(): array {
		$errors = array(
			1 => __( 'has an invalid country code.', 'phone-number-validation' ),
			2 => __( 'number is too short.', 'phone-number-validation' ),
			3 => __( 'number is too long.', 'phone-number-validation' ),
			4 => __( 'is an invalid number.', 'phone-number-validation' ),
		);

		return $errors;
	}

	/**
	 * Separate dial code filter.
	 *
	 * @param bool $value Whether to separate the dial code.
	 * @return bool
	 */
	public function separate_dial_code( bool $value = false ): bool {
		return apply_filters( 'pnv_separate_dial_code', $value );
	}

	/**
	 * Use default WooCommerce store country.
	 *
	 * @return bool
	 */
	public function use_wc_store_default_country(): bool {
		return apply_filters( 'pnv_use_wc_default_store_country', false );
	}

	/**
	 * Get default country.
	 *
	 * @return string
	 */
	public function get_default_country(): string {
		$default = '';

		if ( $this->use_wc_store_default_country() ) {
			$default = apply_filters( 'woocommerce_get_base_location', get_option( 'woocommerce_default_country' ) );

			// Remove sub-states.
			if ( strpos( $default, ':' ) !== false ) {
				list( $country, $state ) = explode( ':', $default );
				$default                 = $country;
			}
		}

		return apply_filters( 'pnv_set_default_country', $default );
	}

	/**
	 * Get preferred countries.
	 *
	 * @return array
	 */
	public function get_preferred_countries(): array {
		return apply_filters( 'pnv_preferred_countries', array() );
	}

	/**
	 * Check if Separate Dial Code is enabled.
	 *
	 * @return bool
	 */
	public function separate_dial_code_enabled(): bool {
		return 'yes' === get_option( 'pnv_separate_dial_code', 'yes' );
	}

	/**
	 * Check if Allow Dropdown is enabled.
	 *
	 * @return bool
	 */
	public function allow_dropdown(): bool {
		return 'yes' === get_option( 'pnv_allow_dropdown', 'yes' );
	}

	/**
	 * Get Exclude Countries.
	 *
	 * @return array
	 */
	public function get_exclude_countries(): array {
		$countries = get_option( 'pnv_exclude_countries', array( 'RU' ) );
		if ( empty( $countries ) || ! is_array( $countries ) ) {
			return array();
		}
		return array_map( 'strtoupper', array_map( 'trim', $countries ) );
	}

	/**
	 * Check if Debug is Enabled.
	 *
	 * @return bool
	 */
	public function is_debug_enabled(): bool {
		return 'yes' === get_option( 'pnv_enable_debug', 'no' );
	}

	/**
	 * Check if Always Allow Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_always_allow_checkout(): bool {
		return 'yes' === get_option( 'pnv_always_allow_checkout', 'yes' );
	}

	/**
	 * Determine if WooCommerce Block Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_block_checkout_enabled(): bool {
		// Use the Settings class to determine block checkout status.
		return $this->settings->is_block_checkout_enabled();
	}

	/**
	 * Check phone field status in WooCommerce Block Checkout and display admin notice if needed.
	 */
	public function check_phone_field_requirement() {
		// Only show notice to users who can manage WooCommerce settings.
		// phpcs:ignore WordPress.WP.Capabilities.Unknown -- Capability added by WooCommerce core.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		// Check if block checkout is enabled.
		if ( ! $this->is_block_checkout_enabled() ) {
			return; // Block checkout not in use; no need for this notice.
		}

		// Get the checkout page ID.
		$checkout_page_id = wc_get_page_id( 'checkout' );

		if ( ! $checkout_page_id || -1 === $checkout_page_id ) {
			// Checkout page not set or invalid.
			return;
		}

		// Get the post content of the checkout page.
		$post_content = get_post_field( 'post_content', $checkout_page_id );

		if ( ! $post_content ) {
			// No content found.
			return;
		}

		// Parse the blocks.
		$blocks = parse_blocks( $post_content );

		$phone_field_enabled = false;

		foreach ( $blocks as $block ) {
			if ( 'woocommerce/checkout' === $block['blockName'] ) {
				// Check block attributes for phone field settings.
				$attributes = isset( $block['attrs'] ) ? $block['attrs'] : array();

				// The actual attribute names depend on how WooCommerce defines them.
				// Adjust these keys based on the actual implementation.
				$show_phone = isset( $attributes['showPhoneField'] ) ? (bool) $attributes['showPhoneField'] : true;

				$phone_field_enabled = $show_phone;

				break; // Assuming only one checkout block is present.
			}
		}

		// If phone field is not enabled or not required, display admin notice.
		if ( ! $phone_field_enabled ) {
			// Get the edit post link for the checkout page.
			$checkout_edit_url = get_edit_post_link( $checkout_page_id );

			if ( ! $checkout_edit_url ) {
				// Fallback to admin URL if edit link is not available.
				$checkout_edit_url = admin_url( 'edit.php?post_type=page&page=wc-settings&tab=checkout' );
			}

			?>
			<div class="notice notice-error">
				<p>
					<strong><?php esc_html_e( 'Phone Number Validation: Phone Field Disabled', 'phone-number-validation' ); ?></strong>
				</p>
				<p>
					<?php
					esc_html_e( 'The Phone Number Validation plugin requires the phone number field to be enabled in the WooCommerce Checkout Block. Please edit your Checkout block to enable phone number validation or switch to classic (i.e. "Shortcode") checkout.', 'phone-number-validation' );
					?>
				</p>
				<p>
					<a href="<?php echo esc_url( $checkout_edit_url ); ?>" class="button button-primary">
						<?php esc_html_e( 'Edit Checkout Block', 'phone-number-validation' ); ?>
					</a>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Get current user's billing phone.
	 *
	 * @return string
	 */
	public function get_current_user_phone(): string {
		return get_user_meta( get_current_user_id(), 'billing_phone', true );
	}

	/**
	 * Get billing custom field meta.
	 *
	 * @return array
	 */
	public function get_billing_custom_field_meta(): array {
		return $this->woo_custom_field_meta;
	}

	/**
	 * Get plugin URL.
	 *
	 * @return string
	 */
	public function plugin_url(): string {
		return untrailingslashit( PNV_ASSETS_URL );
	}

	/**
	 * Get plugin text domain.
	 *
	 * @return string
	 */
	public function get_text_domain(): string {
		return 'phone-number-validation';
	}

	/**
	 * Get plugin version.
	 *
	 * @return string
	 */
	public function get_version(): string {
		return $this->version;
	}

	/**
	 * Validate the phone number on checkout.
	 */
	public function checkout_validate(): void {
		$phone_err_name  = $this->woo_custom_field_meta['billing_hidden_phone_err_field'];
		$phone_err_field = isset( $_POST[ $phone_err_name ] ) ? trim( sanitize_text_field( wp_unslash( $_POST[ $phone_err_name ] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.

		if ( ! empty( $phone_err_field ) ) {
			$phone_err_msg = __( 'Phone number validation - error: ', 'phone-number-validation' ) . esc_html( $phone_err_field );
			$this->log_info( $phone_err_msg );
			wc_add_notice( $phone_err_msg, 'error' );
		}
	}

	/**
	 * Server-side validation for shop_order saves in the admin (manual order creation).
	 *
	 * This acts as a safety net in case client-side validation is bypassed.
	 *
	 * @param int $post_id Post ID being saved.
	 */
	public function admin_order_validate( $post_id ) {
		// Only run in admin on shop_order saves.
		if ( ! is_admin() ) {
			return;
		}

		// Ignore revisions and autosaves.
		if ( wp_is_post_revision( $post_id ) || wp_is_post_autosave( $post_id ) ) {
			return;
		}

		if ( 'shop_order' !== get_post_type( $post_id ) ) {
			return;
		}

		// Capability check: ensure current user can edit this order.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// If the admin JS provided validated international phone values, persist them to the order.
		// This ensures the country code / international formatting is saved when admins create/edit orders.
		$validated_billing  = isset( $_POST['pnv_phone_valid_billing'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
		$validated_shipping = isset( $_POST['pnv_phone_valid_shipping'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.

		// Ensure core save handlers pick up the validated international values as well.
		// Set the $_POST billing/shipping phone to the validated value so any subsequent
		// processing that reads from $_POST will receive the international formatted number.
		if ( '' !== $validated_billing ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
			$_POST['billing_phone'] = $validated_billing; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
		}
		if ( '' !== $validated_shipping ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
			$_POST['shipping_phone'] = $validated_shipping; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
		}

		if ( '' !== $validated_billing ) {
			// Update post meta (billing phone) and, if possible, use the WC_Order CRUD to ensure consistency.
			update_post_meta( $post_id, '_billing_phone', $validated_billing );
			if ( function_exists( 'wc_get_order' ) ) {
				try {
					$order = wc_get_order( $post_id );
					if ( $order ) {
						$order->set_billing_phone( $validated_billing );
					}
				} catch ( \Exception $e ) {
					$this->log_error( '[Admin Order] Failed to save validated billing phone: ' . $e->getMessage() );
				}
			}
		}

		if ( '' !== $validated_shipping ) {
			update_post_meta( $post_id, '_shipping_phone', $validated_shipping );
			if ( function_exists( 'wc_get_order' ) ) {
				try {
					$order = isset( $order ) ? $order : wc_get_order( $post_id );
					if ( $order ) {
						$order->set_shipping_phone( $validated_shipping );
					}
				} catch ( \Exception $e ) {
					$this->log_error( '[Admin Order] Failed to save validated shipping phone: ' . $e->getMessage() );
				}
			}
		}

		// Pull potential validation error fields injected by the admin JS.
		$billing_err  = isset( $_POST['pnv_phone_error_billing'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_billing'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
		$shipping_err = isset( $_POST['pnv_phone_error_shipping'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_shipping'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.

		$messages = array();

		if ( '' !== $billing_err ) {
			$messages[] = sprintf( /* translators: %s: billing phone error */ __( 'Billing: %s', 'phone-number-validation' ), $billing_err );
		}
		if ( '' !== $shipping_err ) {
			$messages[] = sprintf( /* translators: %s: shipping phone error */ __( 'Shipping: %s', 'phone-number-validation' ), $shipping_err );
		}

		if ( empty( $messages ) ) {
			return;
		}

		// If Always Allow Checkout is enabled, do not block admin saves; just log.
		if ( $this->settings && $this->settings->is_always_allow_checkout() ) {
			$this->log_info( '[Admin Order] Phone validation warnings: ' . implode( ' / ', $messages ) );
			return;
		}

		// Combine messages and store transient for showing after redirect.
		$user_id = (int) get_current_user_id();
		$msg     = sprintf( /* translators: %s: combined errors */ __( 'Phone validation error(s): %s', 'phone-number-validation' ), implode( ' / ', $messages ) );
		set_transient( 'pnv_admin_order_error_' . $user_id, $msg, 30 );

		// Also log for debugging.
		$this->log_info( '[Admin Order] Blocking save due to phone validation errors: ' . $msg );

		// Note: we do not attempt to abort the post save here since WordPress has already
		// processed the save flow. Instead we append an indicator to the redirect URL
		// so an admin notice can be displayed and the admin can rectify the phone values.
	}

	/**
	 * Persist validated phone values for legacy (postmeta) order processing.
	 *
	 * WooCommerce has multiple admin save entry points. For legacy orders (non-HPOS)
	 * the woocommerce_process_shop_order_meta action is used during the admin save flow.
	 * This method runs on a very late priority and ensures the validated international
	 * phone values produced by our admin JS are persisted into postmeta so the country
	 * code is kept for legacy orders as well.
	 *
	 * @param int $post_id Post ID being saved.
	 */
	public function admin_order_persist_validated_phones( $post_id ) {
		// Only run in admin on shop_order saves.
		if ( ! is_admin() ) {
			return;
		}

		if ( 'shop_order' !== get_post_type( $post_id ) ) {
			return;
		}

		// Capability check.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		$validated_billing  = isset( $_POST['pnv_phone_valid_billing'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
		$validated_shipping = isset( $_POST['pnv_phone_valid_shipping'] ) ? trim( sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.

		// If validated values are present, ensure core handlers and legacy persistence see them.
		if ( '' !== $validated_billing ) {
			// Make sure any subsequent processing reading from $_POST gets the international value.
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
			$_POST['billing_phone'] = $validated_billing; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.

			// Persist into postmeta for legacy orders.
			update_post_meta( $post_id, '_billing_phone', $validated_billing );
		}

		if ( '' !== $validated_shipping ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
			$_POST['shipping_phone'] = $validated_shipping; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- admin post flow.
			update_post_meta( $post_id, '_shipping_phone', $validated_shipping );
		}
	}

	/**
	 * Append the admin order validation error message to the post-save redirect location.
	 *
	 * @param string $location Redirect URL.
	 * @param int    $post_id Post ID being saved.
	 * @return string Modified redirect URL.
	 */
	public function admin_order_redirect_with_error( $location, $post_id ) {
		// Avoid unused parameter warnings for backward-compatible action signature.
		unset( $post_id );
		if ( ! is_admin() ) {
			return $location;
		}

		$user_id = (int) get_current_user_id();
		$key     = 'pnv_admin_order_error_' . $user_id;
		$msg     = get_transient( $key );

		if ( $msg ) {
			delete_transient( $key );

			// Encode the message to safely include in URL.
			$encoded = rawurlencode( wp_kses_post( $msg ) );

			// Append the encoded message as a query arg so admin_notices can display it.
			$location = add_query_arg( 'pnv_phone_error', $encoded, $location );
		}

		return $location;
	}

	/**
	 * Display admin notice for order validation errors after redirect.
	 */
	public function show_admin_order_error_notice() {
		if ( ! is_admin() ) {
			return;
		}

		if ( ! isset( $_GET['pnv_phone_error'] ) || '' === $_GET['pnv_phone_error'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Value originates from our own redirect and is sanitized below.
			return;
		}

		$encoded = sanitize_text_field( wp_unslash( $_GET['pnv_phone_error'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- sanitized with sanitize_text_field()
		$msg     = rawurldecode( $encoded ); // phpcs:ignore WordPress.Security.NonceVerification.Missing

		if ( $msg ) {
			echo '<div class="notice notice-error is-dismissible">';
			echo '<p>' . esc_html( $msg ) . '</p>';
			echo '</div>';
		}
	}

	/**
	 * Add custom action links to the plugin entry on the Plugins page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_plugin_action_links( array $links ): array {
		$settings_url  = admin_url( 'admin.php?page=wc-settings&tab=shipping&section=pnv_settings' );
		$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . __( 'Settings', 'phone-number-validation' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Log an informational message.
	 *
	 * @param string $message The message to log.
	 */
	public function log_info( string $message ): void {
		if ( $this->settings->is_debug_enabled() ) {
			$logger  = wc_get_logger();
			$context = array( 'source' => 'phone-number-validation' );
			$logger->info( $message, $context );
		}
	}

	/**
	 * Log an error message.
	 *
	 * @param string $message The message to log.
	 */
	public function log_error( string $message ): void {
		if ( $this->settings->is_debug_enabled() ) {
			$logger  = wc_get_logger();
			$context = array( 'source' => 'phone-number-validation' );
			$logger->error( $message, $context );
		}
	}
}
