<?php
/**
 * Add phone number validation to WooCommerce checkout page.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes
 */

declare(strict_types=1);

namespace PNV\Phone_Number_Validation\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use PNV\Phone_Number_Validation\Phone_Number_Validation;
use PNV\Phone_Number_Validation\Admin\Settings;

/**
 * Class Checkout
 *
 * Handles checkout fields and validation.
 */
class Checkout {

	/**
	 * Plugin instance.
	 *
	 * @var Phone_Number_Validation
	 */
	private $plugin;

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $custom_field_meta;

	/**
	 * CDN URLs for intlTelInput assets.
	 *
	 * @var array
	 */
	private $intl_tel_input_urls = array(
		'css'   => PNV_ASSETS_URL . 'vendor/intl-tel-input/css/intlTelInput.css',
		'js'    => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/intlTelInput.min.js',
		'utils' => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/utils.js',
	);

	/**
	 * Constructor.
	 *
	 * @param Phone_Number_Validation $plugin Plugin instance.
	 */
	public function __construct( Phone_Number_Validation $plugin ) {
		$this->plugin            = $plugin;
		$this->custom_field_meta = $this->plugin->get_billing_custom_field_meta();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks(): void {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_filter( 'woocommerce_billing_fields', array( $this, 'add_billing_fields' ), 20, 1 );
		add_action( 'woocommerce_after_checkout_validation', array( $this, 'checkout_validate' ), 10, 1 );
		add_filter( 'woocommerce_checkout_fields', array( $this, 'add_shipping_phone_field' ), 20, 1 );
		add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'save_shipping_phone_field' ), 10, 1 );
		// Handle block/REST/HPOS order creation where the order object is provided.
		add_action( 'woocommerce_checkout_create_order', array( $this, 'save_shipping_phone_field' ), 10, 2 );
	}

	/**
	 * Enqueue necessary scripts and styles.
	 */
	public function enqueue_scripts(): void {
		// Only proceed when on an actual checkout page or when the
		// checkout shortcode/block is present on the current page.
		$is_checkout_request = is_checkout();
		// Detect classic shortcode used on non-checkout pages.
		$has_checkout_shortcode = false;
		if ( function_exists( 'has_shortcode' ) && is_singular() ) {
			$post_id = get_queried_object_id();
			if ( $post_id ) {
				$post_content = get_post_field( 'post_content', $post_id );
				if ( $post_content && has_shortcode( $post_content, 'woocommerce_checkout' ) ) {
					$has_checkout_shortcode = true;
				}
			}
		}
		if ( ! $is_checkout_request && ! $has_checkout_shortcode ) {
			return;
		}

		wp_enqueue_style( 'intlTelInput-css', $this->intl_tel_input_urls['css'], array(), '17.0.19' );
		wp_enqueue_style( 'pnv_css-style', $this->plugin->plugin_url() . '/css/frontend.css', array(), $this->plugin->get_version() );

		wp_enqueue_script( 'intlTelInput-js', $this->intl_tel_input_urls['js'], array(), '17.0.19', true );

		$script_dependencies = array( 'intlTelInput-js' );
		if ( $this->plugin->is_checkout() ) {
			$script_dependencies[] = 'wc-checkout';
		}

		wp_register_script( 'pnv_js-script', $this->plugin->plugin_url() . '/js/frontend.js', $script_dependencies, $this->plugin->get_version(), true );

		// Retrieve settings from the Settings class.
		$settings = Settings::get_instance();

		// Determine default country based on user's saved data.
		$default_country = $settings->get_initial_country();

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			// Fall back to the billing country saved with the address.
			$billing_country = get_user_meta( $user_id, 'billing_country', true );
			if ( ! empty( $billing_country ) ) {
				$default_country = sanitize_text_field( $billing_country );
			}
		}

		$pnv_json = array(
			'phoneValidatorName'    => $this->custom_field_meta['billing_hidden_phone_field'],
			'phoneValidatorErrName' => $this->custom_field_meta['billing_hidden_phone_err_field'],
			'phoneErrorTitle'       => __( 'Phone validation error: ', 'phone-number-validation' ),
			'phoneUnknownErrorMsg'  => __( 'Unknown number', 'phone-number-validation' ),
			'separateDialCode'      => $settings->separate_dial_code(),
			'validationErrors'      => $this->plugin->get_validation_errors(),
			'defaultCountry'        => $default_country,
			'preferredCountries'    => $settings->get_preferred_countries(),
			'utilsScript'           => $this->intl_tel_input_urls['utils'],
			'parentPage'            => '.woocommerce-checkout',
			'currentPage'           => 'checkout',
			'allowDropdown'         => $settings->allow_dropdown(),
			'excludeCountries'      => $settings->get_exclude_countries(),
			// Preferred dial-country mappings for ambiguous dial codes; used by frontend JS to prefer ISO flags.
			'preferredDialCountry'  => $settings->get_preferred_dial_country_mappings(),
			// Optional custom placeholder text exposed to JS. Only included when the admin has
			// enabled the custom placeholder setting.
			'phonePlaceholder'      => ( 'yes' === get_option( 'pnv_enable_custom_placeholder', 'no' ) ) ? apply_filters( 'pnv_phone_placeholder', sanitize_text_field( get_option( 'pnv_phone_placeholder', '' ) ) ) : '',
			'pleaseCorrectPhone'    => __( 'Please correct the phone number(s) before proceeding.', 'phone-number-validation' ),
			'dismissNotice'         => __( 'Dismiss this notice', 'phone-number-validation' ),
			'alwaysAllowCheckout'   => $settings->is_always_allow_checkout(),
		);

		// Optionally include user phone.
		$phone = $this->plugin->get_current_user_phone();
		if ( $phone ) {
			$pnv_json['userPhone'] = $phone;
		}

		wp_localize_script( 'pnv_js-script', 'wcPvJson', $pnv_json );
		wp_enqueue_script( 'pnv_js-script' );

		// Additionally enqueue blocks script if using block checkout.
		if ( wp_is_block_theme() || has_block( 'woocommerce/checkout' ) ) {
			wp_enqueue_script(
				'pnv_blocks-script',
				$this->plugin->plugin_url() . '/js/blocks.js',
				array( 'intlTelInput-js', 'pnv_js-script' ),
				$this->plugin->get_version(),
				true
			);

			// Localize Blocks Script.
			$blocks_json = array(
				'pleaseCorrectPhone'     => __( 'Please correct the phone number(s) before proceeding.', 'phone-number-validation' ),
				'dismissNotice'          => __( 'Dismiss this notice', 'phone-number-validation' ),
				// Preferred dial-country mappings so blocks can prefer specific flags when resolving ambiguous dial codes.
				'preferredDialCountry'   => $settings->get_preferred_dial_country_mappings(),
				// Only expose custom placeholder to blocks when admin has enabled it.
				'phonePlaceholder'       => ( 'yes' === get_option( 'pnv_enable_custom_placeholder', 'no' ) ) ? apply_filters( 'pnv_phone_placeholder', sanitize_text_field( get_option( 'pnv_phone_placeholder', '' ) ) ) : '',
				'alwaysAllowCheckout'    => $settings->is_always_allow_checkout(),
			);

			wp_localize_script( 'pnv_blocks-script', 'wcPvBlocksJson', $blocks_json );
		}
	}

	/**
	 * Add custom classes to the billing phone field.
	 *
	 * @param array $fields Existing billing fields.
	 * @return array Modified billing fields.
	 */
	public function add_billing_fields( array $fields ): array {
		if ( isset( $fields['billing_phone']['class'] ) && is_array( $fields['billing_phone']['class'] ) ) {
			$fields['billing_phone']['class'] = array_merge( $fields['billing_phone']['class'], array( 'pnv-phone' ) );
		}

		return $fields;
	}

	/**
	 * Add Shipping Phone Field to Checkout.
	 *
	 * @param array $fields Existing checkout fields.
	 * @return array Modified checkout fields.
	 */
	public function add_shipping_phone_field( array $fields ): array {
		// Get the Settings instance.
		$settings = Settings::get_instance();

		// Check if block checkout is NOT enabled, shipping phone is enabled, and ship to destination is not 'billing_only'.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {
			// Determine if the shipping phone should be required based on the setting.
			$is_required = $settings->is_make_shipping_phone_required();

			$fields['shipping']['shipping_phone'] = array(
				'label'       => __( 'Shipping Phone', 'phone-number-validation' ),
				'required'    => $is_required,
				'clear'       => false,
				'type'        => 'tel',
				'id'          => 'shipping_phone',
				'priority'    => 110,
				'class'       => array( 'form-row-wide', 'pnv-phone' ),
				'validate'    => $is_required ? array( 'required' ) : array(),
				'placeholder' => __( 'Enter your shipping phone number', 'phone-number-validation' ),
			);
		}

		return $fields;
	}

	/**
	 * Validate checkout fields.
	 *
	 * @param array $data Posted data from the checkout form.
	 */
	public function checkout_validate( array $data ): void { // phpcs:ignore
		// Retrieve error messages from hidden inputs.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_err_field_billing = sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_billing'] ?? '' ) );
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_err_field_shipping = sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_shipping'] ?? '' ) );

		// Retrieve the Settings instance.
		$settings = Settings::get_instance();

		// Check if Always Allow Checkout is disabled.
		$always_allow_checkout = $settings->is_always_allow_checkout();

		// Add error notices for billing phone.
		if ( ! empty( $phone_err_field_billing ) && ! $always_allow_checkout ) {
			// Use sprintf for better message construction.
			$phone_err_msg_billing = sprintf(
				/* translators: %s: Phone validation error message from client-side validation */
				__( 'Phone validation error: %s', 'phone-number-validation' ),
				esc_html( $phone_err_field_billing )
			);
			$this->plugin->log_info( '[Billing] Phone number validation - error: ' . $phone_err_msg_billing );
			wc_add_notice( $phone_err_msg_billing, 'error' );
		}

		// Add error notices for shipping phone.
		if ( ! empty( $phone_err_field_shipping ) && ! $always_allow_checkout ) {
			/* translators: %s: Phone validation error message from client-side validation */
			$phone_err_msg_shipping = sprintf(
				/* translators: %s: Phone validation error message from client-side validation */
				__( 'Phone validation error: %s', 'phone-number-validation' ),
				esc_html( $phone_err_field_shipping )
			);
			$this->plugin->log_info( '[Shipping] Phone number validation - error: ' . $phone_err_msg_shipping );
			wc_add_notice( $phone_err_msg_shipping, 'error' );
		}

		// Check if block checkout is NOT enabled and shipping phone is enabled, and ship to destination is not 'billing_only'.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {

			// Check if the user has opted to ship to a different address.
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$ship_to_different_address = isset( $_POST['ship_to_different_address'] ) && 1 === $_POST['ship_to_different_address'];

			if ( $ship_to_different_address ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
				$shipping_phone = sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ?? '' ) );

				// If the setting requires the shipping phone, validate it.
				if ( $settings->is_make_shipping_phone_required() && empty( $shipping_phone ) && ! $always_allow_checkout ) {
					$empty_shipping_err = __( 'Please enter a valid shipping phone number.', 'phone-number-validation' );
					$this->plugin->log_info( '[Shipping] Phone number validation - error: ' . $empty_shipping_err );
					wc_add_notice( $empty_shipping_err, 'error' );
				}
			}
		}
	}

	/**
	 * Save Shipping/Billing Phone Fields to Order Meta.
	 *
	 * Use validated international phone values when available (pnv_phone_valid_*).
	 *
	 * This method is used for both legacy (order ID) and modern (WC_Order object) hooks:
	 * - woocommerce_checkout_update_order_meta (passes order ID)
	 * - woocommerce_checkout_create_order (passes WC_Order object and checkout data)
	 *
	 * @param int|\WC_Order $order_or_id Order ID or WC_Order object.
	 * @param array|null    $data       Optional checkout data when called from create_order hook.
	 */
	public function save_shipping_phone_field( $order_or_id, $data = null ): void {
		/* phpcs:disable WordPress.Security.NonceVerification.Missing -- WooCommerce handles POST/nonce for checkout/save hooks */
		$settings = Settings::get_instance();

		// Normalize to a WC_Order object and order ID.
		if ( $order_or_id instanceof \WC_Order ) {
			$order    = $order_or_id;
			$order_id = $order->get_id();
		} else {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$order_id = (int) $order_or_id;
			$order    = wc_get_order( $order_id );
		}

		if ( ! $order ) {
			return;
		}

		// Prefer validated billing phone if present, otherwise fall back to submitted billing_phone.
		$billing_phone = '';

		// Helper to attempt extraction from the $data array in several common shapes used by block/REST checkout payloads.
		$extract_phone_from_data = function ( $data_array, $type ) {
			if ( ! is_array( $data_array ) ) {
				return '';
			}

			// Flat keys first.
			$flat_key = $type . '_phone';
			if ( ! empty( $data_array[ $flat_key ] ) ) {
				return $data_array[ $flat_key ];
			}

			// Nested structures.
			if ( ! empty( $data_array[ $type ] ) && is_array( $data_array[ $type ] ) ) {
				$possible_keys = array( 'phone', 'phone_number', 'telephone', 'phoneNumber', 'value' );
				foreach ( $possible_keys as $k ) {
					if ( isset( $data_array[ $type ][ $k ] ) && '' !== $data_array[ $type ][ $k ] ) {
						return $data_array[ $type ][ $k ];
					}
				}

				// Sometimes block checkout nests fields further; look for phone values in nested arrays.
				foreach ( $data_array[ $type ] as $sub ) {
					if ( is_array( $sub ) ) {
						foreach ( $possible_keys as $k ) {
							if ( isset( $sub[ $k ] ) && '' !== $sub[ $k ] ) {
								return $sub[ $k ];
							}
						}
					}
				}
			}

			return '';
		};

		// When called via REST/block checkout the $_POST may not be populated in the same way.
		// But the plugin's JS injects validated values into the visible inputs where possible.
		// We therefore check validated hidden fields, flat $_POST, and the $data payload.
		if ( ! empty( $_POST['pnv_phone_valid_billing'] ) ) {
			$billing_phone = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) );
		} elseif ( ! empty( $_POST['billing_phone'] ) ) {
			$billing_phone = sanitize_text_field( wp_unslash( $_POST['billing_phone'] ) );
		} else {
			$from_data = $extract_phone_from_data( $data, 'billing' );
			if ( '' !== $from_data ) {
				$billing_phone = sanitize_text_field( wp_unslash( $from_data ) );
			}
		}

		if ( '' !== $billing_phone ) {
			// Use the CRUD setter which is HPOS-compatible.
			$order->set_billing_phone( $billing_phone );
		}

		// Handle shipping phone only when shipping phone feature is enabled and shipping is used.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {
			$shipping_phone = '';

			if ( ! empty( $_POST['pnv_phone_valid_shipping'] ) ) {
				$shipping_phone = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) );
			} elseif ( ! empty( $_POST['shipping_phone'] ) ) {
				$shipping_phone = sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ) );
			} else {
				$from_data = $extract_phone_from_data( $data, 'shipping' );
				if ( '' !== $from_data ) {
					$shipping_phone = sanitize_text_field( wp_unslash( $from_data ) );
				}
			}

			if ( '' !== $shipping_phone ) {
				// Use the CRUD setter which is HPOS-compatible.
				$order->set_shipping_phone( $shipping_phone );
			}
		}

		// Save validated phone numbers as order meta using the CRUD method (HPOS-safe).
		$billing_valid_to_save  = '';
		$shipping_valid_to_save = '';

		if ( ! empty( $_POST['pnv_phone_valid_billing'] ) ) {
			$billing_valid_to_save = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) );
		} elseif ( is_array( $data ) && ! empty( $data['pnv_phone_valid_billing'] ) ) {
			$billing_valid_to_save = sanitize_text_field( wp_unslash( $data['pnv_phone_valid_billing'] ) );
		}

		if ( '' !== $billing_valid_to_save ) {
			$order->update_meta_data( '_billing_phone_valid', $billing_valid_to_save );
			// Force the order billing phone to the validated international number so admins see it.
			$order->set_billing_phone( $billing_valid_to_save );
		}

		if ( ! empty( $_POST['pnv_phone_valid_shipping'] ) ) {
			$shipping_valid_to_save = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) );
		} elseif ( is_array( $data ) && ! empty( $data['pnv_phone_valid_shipping'] ) ) {
			$shipping_valid_to_save = sanitize_text_field( wp_unslash( $data['pnv_phone_valid_shipping'] ) );
		}

		if ( '' !== $shipping_valid_to_save ) {
			$order->update_meta_data( '_shipping_phone_valid', $shipping_valid_to_save );
			// Force the order shipping phone to the validated international number so admins see it.
			$order->set_shipping_phone( $shipping_valid_to_save );
		}

		// Persist changes to the order (will save meta and properties; works for HPOS and legacy).
		$order->save();
		/* phpcs:enable WordPress.Security.NonceVerification.Missing */
	}

	/**
	 * Add plugin settings link on the Plugins page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_plugin_action_links( array $links ): array {
		$settings_url  = admin_url( 'admin.php?page=wc-settings&tab=shipping&section=pnv_settings' );
		$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . __( 'Settings', 'phone-number-validation' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}
}
