<?php
/**
 * Add phone number validation to WooCommerce my-account page.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/public
 */

namespace PNV\Phone_Number_Validation\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use PNV\Phone_Number_Validation\Phone_Number_Validation;
use PNV\Phone_Number_Validation\Admin\Settings;

/**
 * Class My_Account
 *
 * Handles account page phone validation.
 */
class My_Account {

	/**
	 * Plugin instance.
	 *
	 * @var Phone_Number_Validation
	 */
	private $plugin;

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $custom_field_meta;

	/**
	 * CDN URLs for intlTelInput assets.
	 *
	 * @var array
	 */
	private $intl_tel_input_urls = array(
		'css'   => PNV_ASSETS_URL . 'vendor/intl-tel-input/css/intlTelInput.css',
		'js'    => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/intlTelInput.min.js',
		'utils' => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/utils.js',
	);

	/**
	 * Constructor.
	 *
	 * @param Phone_Number_Validation $plugin Plugin instance.
	 */
	public function __construct( Phone_Number_Validation $plugin ) {
		$this->plugin            = $plugin;
		$this->custom_field_meta = $this->plugin->get_billing_custom_field_meta();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks(): void {
		// Preprocess validated phone values into $_POST before WooCommerce runs its validation.
		add_action( 'woocommerce_customer_save_address_validation', array( $this, 'preprocess_account_phone' ), 9, 3 );

		add_action( 'woocommerce_customer_save_address_validation', array( $this, 'account_page_validate' ), 10, 3 );
		add_filter( 'woocommerce_shipping_fields', array( $this, 'add_shipping_phone_field_to_account' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// Persist validated phone values when a user saves an address on My Account.
		// Handles both billing and shipping address saves.
		// Use a later priority so this runs after WooCommerce's own address save handlers.
		add_action( 'woocommerce_customer_save_address', array( $this, 'save_account_phone' ), 20, 2 );
	}

	/**
	 * Enqueue necessary scripts and styles for My Account page.
	 */
	public function enqueue_scripts(): void {
		// Only enqueue on My Account pages.
		if ( ! is_account_page() ) {
			return;
		}

		$endpoint = WC()->query->get_current_endpoint();
		if ( 'edit-address' !== $endpoint ) {
			return;
		}

		// Enqueue intlTelInput CSS and custom frontend CSS.
		wp_enqueue_style( 'intlTelInput-css', $this->intl_tel_input_urls['css'], array(), '17.0.19' );
		wp_enqueue_style( 'pnv_css-style', $this->plugin->plugin_url() . '/css/frontend.css', array(), $this->plugin->get_version() );

		// Enqueue intlTelInput JS.
		wp_enqueue_script( 'intlTelInput-js', $this->intl_tel_input_urls['js'], array(), '17.0.19', true );

		// Register and enqueue frontend.js with dependencies.
		wp_register_script( 'pnv_js-script', $this->plugin->plugin_url() . '/js/frontend.js', array( 'intlTelInput-js' ), $this->plugin->get_version(), true );

		// Retrieve settings from the Settings class.
		$settings = Settings::get_instance();

		// Determine default country based on user's saved data.
		$default_country = $settings->get_initial_country();
		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			// Fall back to the billing country saved with the address.
			$billing_country = get_user_meta( $user_id, 'billing_country', true );
			if ( ! empty( $billing_country ) ) {
				$default_country = sanitize_text_field( $billing_country );
			}
		}

		$pnv_json = array(
			'phoneValidatorName'    => $this->custom_field_meta['billing_hidden_phone_field'],
			'phoneValidatorErrName' => $this->custom_field_meta['billing_hidden_phone_err_field'],
			'phoneErrorTitle'       => __( 'Phone validation error:', 'phone-number-validation' ),
			'phoneUnknownErrorMsg'  => __( 'is an invalid number.', 'phone-number-validation' ),
			'separateDialCode'      => $settings->separate_dial_code(),
			'validationErrors'      => $this->plugin->get_validation_errors(),
			'defaultCountry'        => $default_country,
			'preferredCountries'    => $settings->get_preferred_countries(),
			'utilsScript'           => $this->intl_tel_input_urls['utils'],
			'parentPage'            => '.woocommerce-MyAccount-content',
			'currentPage'           => 'account',
			'allowDropdown'         => $settings->allow_dropdown(),
			'excludeCountries'      => $settings->get_exclude_countries(),
			// Preferred dial-country mappings for ambiguous dial codes (e.g. +44 => 'gb').
			'preferredDialCountry'  => $settings->get_preferred_dial_country_mappings(),
			// Optional custom placeholder text exposed to JS. Only included when the admin has
			// enabled the custom placeholder setting.
			'phonePlaceholder'      => ( 'yes' === get_option( 'pnv_enable_custom_placeholder', 'no' ) ) ? apply_filters( 'pnv_phone_placeholder', sanitize_text_field( get_option( 'pnv_phone_placeholder', '' ) ) ) : '',
			'alwaysAllowCheckout'   => $settings->is_always_allow_checkout(),
		);

		// Optionally include user phone.
		$phone = $this->plugin->get_current_user_phone();
		if ( $phone ) {
			$pnv_json['userPhone'] = $phone;
		}

		wp_localize_script( 'pnv_js-script', 'wcPvJson', $pnv_json );
		wp_enqueue_script( 'pnv_js-script' );
	}

	/**
	 * Add Shipping Phone Field to My Account shipping address form.
	 *
	 * @param array $fields Existing shipping fields.
	 * @return array Modified shipping fields.
	 */
	public function add_shipping_phone_field_to_account( array $fields ): array {
		if ( Settings::get_instance()->is_shipping_phone_enabled() ) {
			$fields['shipping_phone'] = array(
				'label'       => __( 'Shipping Phone', 'phone-number-validation' ),
				'placeholder' => _x( 'Enter your shipping phone number', 'placeholder', 'phone-number-validation' ),
				'required'    => false,
				'class'       => array( 'form-row-wide', 'pnv-phone', 'pnv-intl' ),
				'priority'    => 110,
			);
		}

		return $fields;
	}

	/**
	 * Preprocess validated phone values and inject them into $_POST before WooCommerce validation.
	 *
	 * Ensures the international formatted value produced by the client-side validator is
	 * placed into the standard billing_phone/shipping_phone POST keys so core validation and
	 * saving operate on the same value.
	 *
	 * @param array  $_args         Address fields.
	 * @param int    $_user_id      User ID being saved.
	 * @param string $_load_address Type of address e.g., billing or shipping.
	 */
	public function preprocess_account_phone( $_args, int $_user_id, string $_load_address ): void {
		// Intentionally ignore unused parameters; this hook manipulates the global $_POST.
		unset( $_args, $_user_id, $_load_address );
		/* phpcs:disable WordPress.Security.NonceVerification.Missing -- running before WooCommerce validation to massage POST */
		if ( ! isset( $_POST ) || ! is_array( $_POST ) ) {
			/* phpcs:enable WordPress.Security.NonceVerification.Missing */
			return;
		}

		/* phpcs:disable WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- preprocessing validated values */
		// Billing.
		if ( isset( $_POST['pnv_phone_valid_billing'] ) ) {
			$pnv_billing = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) );
			if ( '' !== trim( $pnv_billing ) ) {
				// Use validated international value as the canonical billing_phone for validation/save.
				$_POST['billing_phone'] = $pnv_billing;
			}
		}

		// Shipping.
		if ( isset( $_POST['pnv_phone_valid_shipping'] ) ) {
			$pnv_shipping = sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) );
			if ( '' !== trim( $pnv_shipping ) ) {
				$_POST['shipping_phone'] = $pnv_shipping;
			}
		}
		/* phpcs:enable WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized */
	}

	/**
	 * Validate the phone number on account page address save.
	 *
	 * @param array  $_args       Address fields.
	 * @param int    $_user_id    User ID being saved.
	 * @param string $load_address Type of address e.g., billing or shipping.
	 */
	public function account_page_validate( $_args, int $_user_id, string $load_address ): void {
		// Intentionally ignore unused parameters.
		unset( $_args, $_user_id );
		/* phpcs:disable WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce verification for account address saves */
		// Get the settings instance.
		$settings = Settings::get_instance();

		// If Always Allow Checkout is enabled, skip validation.
		if ( $settings->is_always_allow_checkout() ) {
			return;
		}

		// Determine which address is being validated.
		if ( 'billing' !== $load_address && 'shipping' !== $load_address ) {
			return; // Only validate billing and shipping addresses.
		}

		// Prefer the plugin's standardized POST field names (set by our JS) for validation.
		// Fall back to legacy/custom field meta names if present to remain backwards-compatible.
		if ( 'shipping' === $load_address ) {
			$post_valid_name = 'pnv_phone_valid_shipping';
			$post_err_name   = 'pnv_phone_error_shipping';
		} else {
			$post_valid_name = 'pnv_phone_valid_billing';
			$post_err_name   = 'pnv_phone_error_billing';
		}

		// Legacy/custom field names (kept for backward compatibility).
		$legacy_phone_name     = $this->custom_field_meta['billing_hidden_phone_field'];
		$legacy_phone_err_name = $this->custom_field_meta['billing_hidden_phone_err_field'];
		if ( 'shipping' === $load_address ) {
			$legacy_phone_name     = str_replace( 'billing', 'shipping', $legacy_phone_name );
			$legacy_phone_err_name = str_replace( 'billing', 'shipping', $legacy_phone_err_name );
		}

		// Retrieve and sanitize POST data. Prefer our pnv_ fields, fallback to legacy names.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_valid_field = '';
		if ( isset( $_POST[ $post_valid_name ] ) ) {
			$phone_valid_field = sanitize_text_field( wp_unslash( $_POST[ $post_valid_name ] ) );
		} elseif ( isset( $_POST[ $legacy_phone_name ] ) ) {
			$phone_valid_field = sanitize_text_field( wp_unslash( $_POST[ $legacy_phone_name ] ) );
		}
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_valid_err_field = '';
		if ( isset( $_POST[ $post_err_name ] ) ) {
			$phone_valid_err_field = sanitize_text_field( wp_unslash( $_POST[ $post_err_name ] ) );
		} elseif ( isset( $_POST[ $legacy_phone_err_name ] ) ) {
			$phone_valid_err_field = sanitize_text_field( wp_unslash( $_POST[ $legacy_phone_err_name ] ) );
		}

		if ( 'billing' === $load_address ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$billing_email = isset( $_POST['billing_email'] ) ? sanitize_email( wp_unslash( $_POST['billing_email'] ) ) : '';
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$billing_phone = isset( $_POST['billing_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['billing_phone'] ) ) : '';

			// If the client-side validator supplied an explicit error message, show it (unless always allow).
			if ( ! empty( $phone_valid_err_field ) && ! $settings->is_always_allow_checkout() ) {
				$ph = explode( ':', $phone_valid_err_field );
				if ( isset( $ph[0] ) ) {
					$ph[0] = '<strong>' . esc_html( $ph[0] ) . '</strong>';
				}
				$phone_err_msg = implode( ':', $ph );
				wc_add_notice( $phone_err_msg, 'error' );
				// Stop further processing so the user can correct it.
				return;
			}

			// If we have a formatted validated phone, accept it. Otherwise, allow WooCommerce to validate format.
		} elseif ( 'shipping' === $load_address ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$shipping_phone = isset( $_POST['shipping_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ) ) : '';

			// If the client-side validator supplied an explicit error message, show it (unless always allow).
			if ( ! empty( $phone_valid_err_field ) && ! $settings->is_always_allow_checkout() ) {
				$ph = explode( ':', $phone_valid_err_field );
				if ( isset( $ph[0] ) ) {
					$ph[0] = '<strong>' . esc_html( $ph[0] ) . '</strong>';
				}
				$phone_err_msg = implode( ':', $ph );
				wc_add_notice( $phone_err_msg, 'error' );
				// Stop further processing so the user can correct it.
				return;
			}

			// If we have a formatted validated phone, accept it. Otherwise, allow WooCommerce to validate format.
		}
		/* phpcs:enable WordPress.Security.NonceVerification.Missing */
	}

	/**
	 * Persist validated phone values into user meta when addresses are saved from My Account.
	 *
	 * This ensures the international formatted phone (including country code) set by the
	 * client-side validator is saved to the user's billing/shipping phone meta.
	 *
	 * Temporary logging added to help debug POST payloads when saving addresses.
	 *
	 * @param int    $user_id      User ID being saved.
	 * @param string $load_address 'billing' or 'shipping'.
	 */
	public function save_account_phone( int $user_id, string $load_address ): void {
		/* phpcs:disable WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce for account address saves */
		// Determine which meta keys to write.
		$billing_key  = 'billing_phone';
		$shipping_key = 'shipping_phone';

		// Normalise load_address.
		if ( 'billing' !== $load_address && 'shipping' !== $load_address ) {
			return;
		}

		// Log incoming POST keys and pnv fields when debug is enabled.
		if ( method_exists( $this->plugin, 'is_debug_enabled' ) && $this->plugin->is_debug_enabled() ) {
			$logger    = wc_get_logger();
			$context   = array( 'source' => 'phone-number-validation' );
			$post_keys = is_array( $_POST ) ? array_keys( $_POST ) : array();
			$logger->info( 'save_account_phone called for user_id=' . (int) $user_id . ' load_address=' . $load_address . ' | _POST keys: ' . implode( ',', $post_keys ), $context );

			$pvb  = isset( $_POST['pnv_phone_valid_billing'] ) ? sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) ) : '';
			$pveb = isset( $_POST['pnv_phone_error_billing'] ) ? sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_billing'] ) ) : '';
			$pvs  = isset( $_POST['pnv_phone_valid_shipping'] ) ? sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) ) : '';
			$pves = isset( $_POST['pnv_phone_error_shipping'] ) ? sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_shipping'] ) ) : '';
			$logger->info( 'pnv fields (billing_valid, billing_err, shipping_valid, shipping_err): ' . wp_json_encode( array( $pvb, $pveb, $pvs, $pves ) ), $context );
		}

		// Determine validated hidden field names used by our JS.
		$validated_billing_field  = 'pnv_phone_valid_billing';
		$validated_shipping_field = 'pnv_phone_valid_shipping';

		// Preference: validated hidden field (international format), then visible input.
		if ( 'billing' === $load_address ) {
			$phone = '';
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce.
			if ( isset( $_POST[ $validated_billing_field ] ) ) {
				$phone_candidate = sanitize_text_field( wp_unslash( $_POST[ $validated_billing_field ] ) );
				if ( '' !== trim( $phone_candidate ) ) {
					$phone = $phone_candidate;
				}
			} elseif ( isset( $_POST['billing_phone'] ) ) {
				$phone_candidate = sanitize_text_field( wp_unslash( $_POST['billing_phone'] ) );
				if ( '' !== trim( $phone_candidate ) ) {
					$phone = $phone_candidate;
				}
			}

			if ( '' !== $phone ) {
				update_user_meta( $user_id, $billing_key, $phone );
			}
		} elseif ( 'shipping' === $load_address ) {
			$phone = '';
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce.
			if ( isset( $_POST[ $validated_shipping_field ] ) ) {
				$phone_candidate = sanitize_text_field( wp_unslash( $_POST[ $validated_shipping_field ] ) );
				if ( '' !== trim( $phone_candidate ) ) {
					$phone = $phone_candidate;
				}
			} elseif ( isset( $_POST['shipping_phone'] ) ) {
				$phone_candidate = sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ) );
				if ( '' !== trim( $phone_candidate ) ) {
					$phone = $phone_candidate;
				}
			}

			if ( '' !== $phone ) {
				update_user_meta( $user_id, $shipping_key, $phone );
			}
		}
		/* phpcs:enable WordPress.Security.NonceVerification.Missing */
	}
}
