# Phone Number Validation for WooCommerce

## Overview

[Phone Number Validation](https://woocommerce.com/product/phone-number-validation) ensures your WooCommerce store collects valid, correctly formatted international phone numbers at checkout and in account pages. Built on the [intl-tel-input library](https://github.com/jackocnr/intl-tel-input) supported by [Twilio](https://twilio.com/), this plugin provides real-time phone number validation, country-aware formatting, and compatibility with both classic and block-based WooCommerce checkouts.

---

## Requirements

Before installing the plugin, ensure your environment meets the following minimum requirements:

- WordPress: 6.0 or higher
- WooCommerce: 9.0 or higher
- PHP 7.4 or higher

---

## Installation

1. Download the plugin .zip file from your WooCommerce account.
2. In WordPress admin, go to Plugins > Add New > Upload Plugin.
3. Upload the plugin .zip and click Install Now.
4. Click Activate Plugin after installation.
5. Configure plugin settings at WooCommerce > Settings > Shipping > Phone Validation (or via the Settings link after activation).

---

## Usage

The plugin exposes settings to control which phone fields appear, how they behave, and how validation runs. It works with both classic and block-based checkouts.

### Plugin Settings Overview

The settings panel (WooCommerce > Settings > Shipping > Phone Validation) includes:

- Add Shipping Phone Field - add an independent phone field to the shipping address.
- Make Shipping Phone Required - force customers to supply a shipping phone number.
- Initial Country - default country shown for customers without an account. Default: US.
- Supported Countries - list of allowed country codes; leave blank to allow all. If set, must include the Initial Country.
- Preferred Countries - countries shown first in the country selector (e.g. `US,GB`).
- Show Country Codes - show international dial codes alongside flags.
- Allow Dropdown - let customers choose their country from a dropdown.
- Exclude Countries - remove specific countries from the selector.
- Custom placeholder - enable a custom placeholder text for the phone input. When enabled, an additional "Phone Placeholder" field appears.
- Phone Placeholder (optional) - an optional custom placeholder text for the phone input field. Leave empty to use the plugin's generated placeholder (for example `555-201-1111`). When the custom placeholder is enabled, the value is shown on the checkout page and can be filtered via `pnv_phone_placeholder`.
- Enable Debug - when enabled, plugin debug logs are available at WooCommerce > Status > Logs.

---

### Classic Checkout

- Enhances the billing phone field with flag selectors, automatic country detection, formatting and validation.
- Optionally adds a separate shipping phone field (can be required).
- Validates phone numbers in real time as the customer types and shows inline errors.
- Prevents form submission until phone numbers pass validation (blocking or soft behavior depends on settings).
- Validated phone numbers are saved to the order metadata for later use.

---

### Block Checkout

- Detects WooCommerce block-based checkout pages and initializes phone field enhancements for block checkout UI components.
- Shows explicit error messages per field (invalid country code, too short, too long, invalid number).
- If validation fails, a checkout-level error message can be displayed and the Place Order button can be disabled until all phone fields are valid.
- Compatible with modern block-based checkout layouts and should work with standard WooCommerce blocks.

---

## FAQ

Q: How do I enable the shipping phone field?
A: Go to WooCommerce > Settings > Shipping > Phone Validation and check "Add Shipping Phone Field". Save settings. To make it required enable "Make Shipping Phone Required".

Q: Can I restrict validation to certain countries?
A: Yes. Use the Supported Countries setting to allow only specific country codes. You can also use Exclude Countries to remove particular countries.

Q: Where are validated phone numbers stored?
A: Validated phone numbers are saved in the order metadata and remain visible in the order admin details.

Q: Is debug logging available?
A: Yes. Enable "Enable Debug" in plugin settings. Logs appear under WooCommerce > Status > Logs.

Q: Does it work with HPOS (High Performance Order Storage)?
A: Yes - plugin stores phone data using WooCommerce order APIs and is compatible with both traditional and HPOS storage.

Q: What happens if a phone number is invalid?
A: The plugin displays a clear inline error. On block checkouts the Place Order button may be disabled until the input is corrected.

---

## Advanced

This section documents developer-facing filters and provides example snippets you can drop into your theme or plugin (for use in a site-specific plugin or mu-plugin).

Note: use these snippets in a plugin or via functions. of a child theme. Always follow WordPress best practices and sanitise inputs when needed.

### Available Filters

The plugin exposes several filters to customize behavior:

- `pnv_separate_dial_code` (bool)
  - Controls whether the plugin uses a separate dial code UI.
  - Default: false
  - Example: return true to show separate dial code input.

- `pnv_use_wc_default_store_country` (bool)
  - When true, plugin will use WooCommerce store default country for initial country detection.
  - Default: false

- `pnv_set_default_country` (string)
  - Filter the plugin's computed default country code.
  - Receives the current default country code, return a two-letter country code (e.g., 'US').

- `pnv_preferred_countries` (array)
  - Filter the list of preferred countries displayed at the top of the country selector.
  - Default: empty array (plugin settings may set defaults).

- `pnv_preferred_dial_country` (array)
  - Internal mapping used by admin settings; can be used to change admin defaults.
  - The mapping is exported to the frontend JavaScript as `preferredDialCountry` so the intl-tel-input instance can prefer a specific country flag when multiple countries share the same dial code (for example `+44` is shared by GB, GG, JE, IM). See assets/js/frontend.js for the client-side merge behavior.

- `pnv_phone_placeholder` (string)
  - Filter the phone placeholder text displayed inside inputs when custom placeholder is enabled.

- `pnv_exclude_countries` (array)
  - Filter the array of country codes to exclude from the country dropdown.

These filters are applied in admin and public classes to allow flexible customization.

### Example Code Snippets

1) Force separate dial code UI

```
add_filter( 'pnv_separate_dial_code', '__return_true' );
```

2) Use store default country as initial country

```
add_filter( 'pnv_use_wc_default_store_country', '__return_true' );
```

3) Override default country to France

```
add_filter( 'pnv_set_default_country', function( $default ) {
    // Force default country to France
    return 'FR';
} );
```

4) Add preferred countries

```
add_filter( 'pnv_preferred_countries', function( $countries ) {
    // Show France and United Kingdom at top of the list
    return array_merge( array( 'FR', 'GB' ), $countries );
} );
```

5) Change the placeholder text

```
add_filter( 'pnv_phone_placeholder', function( $placeholder ) {
    return '1 49 52 70 00';
} );
```

6) Preferred dial-country mappings (for ambiguous dial codes)

Use the `pnv_preferred_dial_country` filter to set server-side mappings that the frontend JavaScript will merge into its `preferredDialCountry` object. This lets you control which country flag is shown when multiple countries share the same international dial code.

Example:
```
add_filter( 'pnv_preferred_dial_country', function( $mappings ) {
    $mappings = is_array( $mappings ) ? $mappings : array();

    // prefer Guernsey for +44 (over GG, JE, IM)
    $mappings['44'] = 'gg';

    // prefer United States for +1 (over CA, etc.)
    $mappings['1'] = 'us';

    return $mappings;
} );
```

7) Exclude countries from selector

The plugin exposes a filter to remove specific country codes from the phone selector. This is useful both for officially supported WooCommerce countries and for country codes provided by the bundled validation library that are not part of WooCommerce's official country list.

Note about non-WooCommerce country codes (e.g. Kosovo, Ascension Island)
- The intl-tel-input library includes an extended set of country codes. Some of these (for example Kosovo `XK` and Ascension Island `AC`) are not present in WooCommerce's official country list because they either use temporary codes or are not separate ISO-2 countries.
- The plugin provides the `pnv_exclude_countries` filter so you can remove those codes from the frontend dropdown even when WooCommerce does not list them.


```
/**
 * Exclude Kosovo (XK) and Ascension Island (AC) from the phone number dropdown
 * in the Phone Number Validation plugin for WooCommerce.
 *
 * Add this code to your theme's functions. or as a custom plugin.
 */
add_filter( 'pnv_exclude_countries', function( $countries ) {
    $countries = is_array( $countries ) ? $countries : array();
    foreach ( array( 'XK', 'AC' ) as $code ) {
        if ( ! in_array( $code, $countries, true ) ) {
            $countries[] = $code;
        }
    }
    return $countries;
} );
```

---

## Troubleshooting & Tips

- If international formatting or flags do not show, ensure the bundled intl-tel-input scripts are loaded and no conflict is caused by other plugins that also load intl-tel-input.
- Enable debug logging in plugin settings to capture helpful messages (WooCommerce > Status > Logs).
- If checkout errors prevent order placement, test by temporarily disabling other checkout-related plugins to isolate conflicts.
- When customizing via filters, use a site-specific plugin or mu-plugin to avoid loss during theme updates.

