/**
 * Block Checkout Script
 * Initializes validation for phone number fields in the WooCommerce checkout block.
 *
 * @package    Phone_Number_Validation
 */

document.addEventListener(
	'DOMContentLoaded',
	function () {
		'use strict';

		// Check if wcPvJson is defined.
		if (typeof wcPvJson === 'undefined') {
			console.error('wcPvJson is not defined.');
			return;
		}

		// Check if wcPvBlocksJson is defined.
		if (typeof wcPvBlocksJson === 'undefined') {
			console.error('wcPvBlocksJson is not defined.');
			return;
		}

		// --- Dynamic Default Country Setup for Blocks ---

		/**
		 * Function to dynamically determine default country from a phone number.
		 *
		 * @param {string} phone - The phone number (expected in international format).
		 * @returns {string|null} The ISO country code (e.g. "za") or null if not found.
		 */
		function getDefaultCountryFromPhone(phone) {
			// Ensure the phone number is in international format.
			if (!phone || phone.charAt(0) !== '+' || typeof window.intlTelInputGlobals === 'undefined' || !window.intlTelInputGlobals.getCountryData) {
				return null;
			}

			// Retrieve country data.
			var countryData = window.intlTelInputGlobals.getCountryData();
			var bestMatch = { dialCode: '', iso2: null };

			// Remove the '+' sign.
			phone = phone.substring(1);

			// Loop over each country to find the longest matching dial code.
			countryData.forEach(function(country) {
				var dialCode = country.dialCode;
				if (phone.indexOf(dialCode) === 0 && dialCode.length > bestMatch.dialCode.length) {
					bestMatch.dialCode = dialCode;
					bestMatch.iso2 = country.iso2;
				}
			});

			return bestMatch.iso2;
		}

		// If a saved user phone exists, update wcPvJson.defaultCountry accordingly.
		if (wcPvJson.userPhone) {
			var userDefault = getDefaultCountryFromPhone(wcPvJson.userPhone);
			if (userDefault) {
				wcPvJson.defaultCountry = userDefault;
			}
		}

		// --- End Dynamic Default Country Setup ---

		// Store intlTelInput instances.
		let wcPvBillingPhoneIntl = null;
		let wcPvShippingPhoneIntl = null;

		// Flag to ensure the event listener is added only once.
		let placeOrderListenerAdded = false;

		// Touched flags to track user interaction.
		let billingPhoneTouched = false;
		let shippingPhoneTouched = false;

		/**
		 * Function to validate phone number.
		 *
		 * @param {Object} phoneInstance - The intl-tel-input instance.
		 * @param {string} fieldType - 'Billing' or 'Shipping'.
		 * @param {boolean} isTouched - Whether the field has been interacted with.
		 * @param {string} inputValue - The current value of the phone input field.
		 * @returns {Object} - Contains formattedNumber and errorMessage.
		 */
		function wcPvValidatePhone(phoneInstance, fieldType, isTouched, inputValue) {
			let formattedNumber = false;
			let errorMessage = '';

			// If the input is empty, skip validation.
			if (inputValue.trim() === '') {
				return {
					formattedNumber: false,
					errorMessage: ''
				};
			}

			if (phoneInstance && phoneInstance.isValidNumber()) {
				formattedNumber = phoneInstance.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL);
			} else if (phoneInstance && isTouched) { // Only set error if touched.
				const validationError = phoneInstance.getValidationError();
				errorMessage = `${fieldType} phone ${
					wcPvJson.validationErrors[validationError] !== undefined
						? wcPvJson.validationErrors[validationError]
						: wcPvJson.phoneUnknownErrorMsg
				}`;
			}

			return {
				formattedNumber: formattedNumber,
				errorMessage: errorMessage
			};
		}

		/**
		 * Function to show a global error message styled like WooCommerce's error notices.
		 *
		 * @param {string} message - The error message to display.
		 */
		function showGlobalErrorMessage(message) {
			const noticesContainer = document.querySelector('.wc-block-components-notices');

			if (!noticesContainer) {
				console.error('Blocks: Notices container not found.');
				return;
			}

			let globalErrorNotice = noticesContainer.querySelector('.global-phone-validation-error');

			if (!globalErrorNotice) {
				globalErrorNotice = document.createElement('div');
				globalErrorNotice.className = 'wc-block-store-notice wc-block-components-notice-banner is-error is-dismissible global-phone-validation-error';

				const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				svg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
				svg.setAttribute('viewBox', '0 0 24 24');
				svg.setAttribute('width', '24');
				svg.setAttribute('height', '24');
				svg.setAttribute('aria-hidden', 'true');
				svg.setAttribute('focusable', 'false');

				const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				path.setAttribute('d', 'M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z');
				svg.appendChild(path);

				const contentDiv = document.createElement('div');
				contentDiv.className = 'wc-block-components-notice-banner__content';

				const messageDiv = document.createElement('div');
				messageDiv.textContent = message;

				contentDiv.appendChild(messageDiv);

				const dismissButton = document.createElement('button');
				dismissButton.className = 'wc-block-components-button wp-element-button wc-block-components-notice-banner__dismiss contained';
				dismissButton.setAttribute('aria-label', wcPvBlocksJson.dismissNotice);
				dismissButton.setAttribute('type', 'button');

				const dismissSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				dismissSvg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
				dismissSvg.setAttribute('viewBox', '0 0 24 24');
				dismissSvg.setAttribute('width', '24');
				dismissSvg.setAttribute('height', '24');
				dismissSvg.setAttribute('aria-hidden', 'true');
				dismissSvg.setAttribute('focusable', 'false');

				const dismissPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				dismissPath.setAttribute('d', 'M13 11.8l6.1-6.3-1-1-6.1 6.2-6.1-6.2-1 1 6.1 6.3-6.5 6.7 1 1 6.5-6.6 6.5 6.6 1-1z');
				dismissSvg.appendChild(dismissPath);

				dismissButton.appendChild(dismissSvg);

				globalErrorNotice.appendChild(svg);
				globalErrorNotice.appendChild(contentDiv);
				globalErrorNotice.appendChild(dismissButton);

				noticesContainer.appendChild(globalErrorNotice);

				globalErrorNotice.scrollIntoView({ behavior: 'smooth', block: 'start' });

				dismissButton.addEventListener('click', function () {
					globalErrorNotice.remove();
				});
			} else {
				const messageDiv = globalErrorNotice.querySelector('.wc-block-components-notice-banner__content div');
				if (messageDiv) {
					messageDiv.textContent = message;
				}
				globalErrorNotice.scrollIntoView({ behavior: 'smooth', block: 'start' });
			}
		}

		/**
		 * Function to update the state of the "Place Order" button.
		 */
		function updatePlaceOrderButton() {
			const placeOrderButton = document.querySelector('.wc-block-components-checkout-place-order-button');
			const billingError = document.querySelector('#billing .phone-validation-error');
			const shippingError = document.querySelector('#shipping .phone-validation-error');

			if (placeOrderButton) {
				const alwaysAllowCheckout = wcPvJson.alwaysAllowCheckout !== undefined ? wcPvJson.alwaysAllowCheckout : wcPvBlocksJson.alwaysAllowCheckout;

				if (billingError || shippingError) {
					if (!alwaysAllowCheckout) {
						placeOrderButton.setAttribute('disabled', 'disabled');
						placeOrderButton.style.opacity = '0.5';
						placeOrderButton.style.cursor = 'not-allowed';
						showGlobalErrorMessage(wcPvBlocksJson.pleaseCorrectPhone);
					} else {
						placeOrderButton.removeAttribute('disabled');
						placeOrderButton.style.opacity = '1';
						placeOrderButton.style.cursor = 'pointer';
						removeGlobalErrorMessage();
					}
				} else {
					placeOrderButton.removeAttribute('disabled');
					placeOrderButton.style.opacity = '1';
					placeOrderButton.style.cursor = 'pointer';
					removeGlobalErrorMessage();
				}
			}
		}

		/**
		 * Function to remove the global error message.
		 */
		function removeGlobalErrorMessage() {
			const noticesContainer = document.querySelector('.wc-block-components-notices');
			if (noticesContainer) {
				const globalErrorNotice = noticesContainer.querySelector('.global-phone-validation-error');
				if (globalErrorNotice) {
					globalErrorNotice.remove();
				}
			}
		}

		/**
		 * Function to create and display error messages.
		 *
		 * @param {HTMLElement} phoneField - The phone input field.
		 * @param {string} errorMessage - The error message to display.
		 */
		function displayErrorMessage(phoneField, errorMessage) {
			phoneField.classList.add('has-error');
			phoneField.setAttribute('aria-invalid', 'true');

			let errorDiv = phoneField.parentNode.querySelector('.phone-validation-error');
			if (!errorDiv) {
				errorDiv = document.createElement('div');
				errorDiv.className = 'phone-validation-error';
				phoneField.parentNode.appendChild(errorDiv);

				let errorIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
				errorIcon.setAttribute('width', '24');
				errorIcon.setAttribute('height', '24');
				errorIcon.setAttribute('viewBox', '-2 -2 24 24');
				errorIcon.setAttribute('fill', 'red');
				errorIcon.classList.add('error-icon');

				let path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
				path.setAttribute('d', 'M10 2c4.42 0 8 3.58 8 8s-3.58 8-8 8-8-3.58-8-8 3.58-8 8-8zm1.13 9.38l.35-6.46H8.52l.35 6.46h2.26zm-.09 3.36c.24-.23.37-.55.37-.96 0-.42-.12-.74-.36-.97s-.59-.35-1.06-.35-.82.12-1.07.35-.37.55-.37.97c0 .41.13.73.38.96.26.23.61.34 1.06.34s.8-.11 1.05-.34z');
				errorIcon.appendChild(path);
				errorDiv.appendChild(errorIcon);

				let errorMessageNode = document.createElement('span');
				errorDiv.appendChild(errorMessageNode);
			}

			let errorMessageNode = errorDiv.querySelector('span');
			if (errorMessageNode) {
				errorMessageNode.textContent = errorMessage;
			}
		}

		/**
		 * Function to remove error messages.
		 *
		 * @param {HTMLElement} phoneField - The phone input field.
		 */
		function removeErrorMessage(phoneField) {
			phoneField.classList.remove('has-error');
			phoneField.setAttribute('aria-invalid', 'false');
			const errorDiv = phoneField.parentNode.querySelector('.phone-validation-error');
			if (errorDiv) {
				errorDiv.remove();
			}
		}

		/**
		 * Function to prevent checkout if there are validation errors.
		 */
		function preventInvalidCheckout() {
			if (placeOrderListenerAdded) {
				return;
			}

			const placeOrderButton = document.querySelector('.wc-block-components-checkout-place-order-button');
			if (placeOrderButton) {
				placeOrderButton.addEventListener(
					'click',
					function (event) {
						const billingError = document.querySelector('#billing .phone-validation-error');
						const shippingError = document.querySelector('#shipping .phone-validation-error');
						const alwaysAllowCheckout = wcPvJson.alwaysAllowCheckout !== undefined ? wcPvJson.alwaysAllowCheckout : wcPvBlocksJson.alwaysAllowCheckout;

						if ((billingError || shippingError) && !alwaysAllowCheckout) {
							event.preventDefault();
							event.stopPropagation();
							showGlobalErrorMessage(wcPvBlocksJson.pleaseCorrectPhone);
						} else {
							removeGlobalErrorMessage();
						}
					}
				);
				placeOrderListenerAdded = true;
			}
		}

		/**
		 * Function to handle country changes and update intl-tel-input instances.
		 */
		function handleCountryChange() {
			const billingCountrySelect = document.querySelector('.wc-block-checkout__billing-fields .wc-block-components-address-form__country select');
			const shippingCountrySelect = document.querySelector('.wc-block-checkout__shipping-fields .wc-block-components-address-form__country select');

			function updatePhoneCountry(selectElement, phoneInstance) {
				if (!selectElement || !phoneInstance) {
					return;
				}
				const selectedCountryCode = selectElement.value.toLowerCase();
				phoneInstance.setCountry(selectedCountryCode);
			}

			if (billingCountrySelect && !billingCountrySelect.classList.contains('pnv-country-listener-added')) {
				billingCountrySelect.classList.add('pnv-country-listener-added');
				billingCountrySelect.addEventListener(
					'change',
					function () {
						updatePhoneCountry(this, wcPvBillingPhoneIntl);
					}
				);
			}

			if (shippingCountrySelect && !shippingCountrySelect.classList.contains('pnv-country-listener-added')) {
				shippingCountrySelect.classList.add('pnv-country-listener-added');
				shippingCountrySelect.addEventListener(
					'change',
					function () {
						updatePhoneCountry(this, wcPvShippingPhoneIntl);
					}
				);
			}
		}

		/**
		 * Modified helper function to retrieve the selected country code.
		 * If a user phone exists, return the default country from wcPvJson.
		 *
		 * @param {string} formType - 'billing' or 'shipping'.
		 * @returns {string} - The selected country code in lowercase.
		 */
		function getSelectedCountryCode(formType) {
			// If a user phone exists, use the dynamic default.
			if (wcPvJson.userPhone) {
				return (wcPvJson.defaultCountry || 'us').toLowerCase();
			}

			const selector = formType === 'billing'
				? '.wc-block-checkout__billing-fields .wc-block-components-address-form__country select'
				: '.wc-block-checkout__shipping-fields .wc-block-components-address-form__country select';

			const countrySelect = document.querySelector(selector);
			if (countrySelect && countrySelect.value) {
				return countrySelect.value.toLowerCase();
			}

			// Fallback.
			return (wcPvJson.defaultCountry || 'us').toLowerCase();
		}

		/**
		 * Function to initialize phone fields in blocks.
		 */
		function initializeBlockPhoneFields() {
			const shippingPhone = document.querySelector('.wc-block-checkout__shipping-fields .wc-block-components-address-form__phone input[type="tel"]');
			const billingPhone = document.querySelector('.wc-block-checkout__billing-fields .wc-block-components-address-form__phone input[type="tel"]');

			// Initialize Billing Phone.
			if (billingPhone && !billingPhone.classList.contains('pnv-phone')) {
				billingPhone.classList.add('pnv-phone');
				const billingPhoneContainer = billingPhone.closest('.wc-block-components-address-form__phone');
				if (billingPhoneContainer) {
					billingPhoneContainer.classList.add('is-active');
				} else {
					console.warn('Blocks: Billing phone container not found.');
				}

				// Get initial country for billing.
				let billingInitialCountry = getSelectedCountryCode('billing');

				wcPvBillingPhoneIntl = window.intlTelInput(
					billingPhone,
					{
						initialCountry: billingInitialCountry,
						separateDialCode: wcPvJson.separateDialCode,
						preferredCountries: wcPvJson.preferredCountries || [],
						utilsScript: wcPvJson.utilsScript,
						allowDropdown: wcPvJson.allowDropdown,
						excludeCountries: wcPvJson.excludeCountries || [],
						formatOnDisplay: false
					}
				);

				const hiddenValidatorField = document.createElement('input');
				hiddenValidatorField.type = 'hidden';
				hiddenValidatorField.name = wcPvJson.phoneValidatorName;
				billingPhone.parentNode.appendChild(hiddenValidatorField);

				const hiddenErrorField = document.createElement('input');
				hiddenErrorField.type = 'hidden';
				hiddenErrorField.name = wcPvJson.phoneValidatorErrName;
				billingPhone.parentNode.appendChild(hiddenErrorField);

				billingPhone.addEventListener(
					'input',
					function () {
						billingPhoneTouched = true;
						const inputValue = billingPhone.value;
						const validation = wcPvValidatePhone(wcPvBillingPhoneIntl, 'Billing', billingPhoneTouched, inputValue);
						hiddenValidatorField.value = validation.formattedNumber || '';
						hiddenErrorField.value = validation.errorMessage;

						if (validation.errorMessage) {
							displayErrorMessage(billingPhone, validation.errorMessage);
						} else {
							removeErrorMessage(billingPhone);
						}

						updatePlaceOrderButton();
					}
				);

				billingPhone.addEventListener(
					'blur',
					function () {
						billingPhoneTouched = true;
						billingPhone.dispatchEvent(new Event('input'));
					}
				);
				billingPhone.addEventListener(
					'countrychange',
					function () {
						billingPhoneTouched = true;
						billingPhone.dispatchEvent(new Event('input'));
					}
				);
			}

			// Initialize Shipping Phone.
			if (shippingPhone && !shippingPhone.classList.contains('pnv-phone')) {
				shippingPhone.classList.add('pnv-phone');
				const shippingPhoneContainer = shippingPhone.closest('.wc-block-components-address-form__phone');
				if (shippingPhoneContainer) {
					shippingPhoneContainer.classList.add('is-active');
				} else {
					console.warn('Blocks: Shipping phone container not found.');
				}

				let shippingInitialCountry = getSelectedCountryCode('shipping');

				wcPvShippingPhoneIntl = window.intlTelInput(
					shippingPhone,
					{
						initialCountry: shippingInitialCountry,
						separateDialCode: wcPvJson.separateDialCode,
						preferredCountries: wcPvJson.preferredCountries || [],
						utilsScript: wcPvJson.utilsScript,
						allowDropdown: wcPvJson.allowDropdown,
						excludeCountries: wcPvJson.excludeCountries || [],
						formatOnDisplay: false
					}
				);

				const hiddenValidatorField = document.createElement('input');
				hiddenValidatorField.type = 'hidden';
				hiddenValidatorField.name = wcPvJson.phoneValidatorName + '_shipping';
				shippingPhone.parentNode.appendChild(hiddenValidatorField);

				const hiddenErrorField = document.createElement('input');
				hiddenErrorField.type = 'hidden';
				hiddenErrorField.name = wcPvJson.phoneValidatorErrName + '_shipping';
				shippingPhone.parentNode.appendChild(hiddenErrorField);

				shippingPhone.addEventListener(
					'input',
					function () {
						shippingPhoneTouched = true;
						const inputValue = shippingPhone.value;
						const validation = wcPvValidatePhone(wcPvShippingPhoneIntl, 'Shipping', shippingPhoneTouched, inputValue);
						hiddenValidatorField.value = validation.formattedNumber || '';
						hiddenErrorField.value = validation.errorMessage;

						if (validation.errorMessage) {
							displayErrorMessage(shippingPhone, validation.errorMessage);
						} else {
							removeErrorMessage(shippingPhone);
						}

						updatePlaceOrderButton();
					}
				);

				shippingPhone.addEventListener(
					'blur',
					function () {
						shippingPhoneTouched = true;
						shippingPhone.dispatchEvent(new Event('input'));
					}
				);
				shippingPhone.addEventListener(
					'countrychange',
					function () {
						shippingPhoneTouched = true;
						shippingPhone.dispatchEvent(new Event('input'));
					}
				);
			}

			// Attach country change listeners.
			handleCountryChange();

			// Prevent invalid checkout.
			preventInvalidCheckout();
		}

		/**
		 * Function to create a mutation observer to watch for changes.
		 */
		function createMutationObserver() {
			const observer = new MutationObserver(
				(mutations) => {
					mutations.forEach(
						(mutation) => {
							if (mutation.addedNodes.length) {
								initializeBlockPhoneFields();
							}
						}
					);
				}
			);

			const checkoutForm = document.querySelector('.wp-block-woocommerce-checkout');
			if (checkoutForm) {
				observer.observe(
					checkoutForm,
					{
						childList: true,
						subtree: true
					}
				);
			}
		}

		// Initial check.
		initializeBlockPhoneFields();

		// Create mutation observer for dynamic changes.
		createMutationObserver();

		// Listen for checkout updates.
		document.addEventListener('wc-blocks-checkout-update-checkout', initializeBlockPhoneFields);
	}
);
