/**
 * Frontend Script
 *
 * @package    Phone_Number_Validation
 */

document.addEventListener('DOMContentLoaded', function () {

	'use strict';

	// Check if wcPvJson is defined.
	if (typeof wcPvJson === 'undefined') {
		console.error('wcPvJson is not defined.');
		return;
	}

	// Function to dynamically determine default country from a phone number.
	function getDefaultCountryFromPhone(phone) {
		// Ensure the phone number is in international format.
		if (!phone || phone.charAt(0) !== '+' || typeof window.intlTelInputGlobals === 'undefined' || !window.intlTelInputGlobals.getCountryData) {
			return null;
		}

		// Retrieve all country data from the intlTelInput library.
		var countryData = window.intlTelInputGlobals.getCountryData();
		var bestMatch = { dialCode: '', iso2: null };

		// Remove the '+' sign from the phone number.
		phone = phone.substring(1);

		// Loop over each country and find the one with the longest matching dial code.
		countryData.forEach(function(country) {
			var dialCode = country.dialCode;
			if (phone.indexOf(dialCode) === 0 && dialCode.length > bestMatch.dialCode.length) {
				bestMatch.dialCode = dialCode;
				bestMatch.iso2 = country.iso2;
			}
		});

		return bestMatch.iso2;
	}

	// If a saved user phone exists, try to update the default country dynamically.
	if (wcPvJson.userPhone) {
		var userDefault = getDefaultCountryFromPhone(wcPvJson.userPhone);
		if (userDefault) {
			wcPvJson.defaultCountry = userDefault;
		}
	}

	// Retrieve validation error messages from localized JSON.
	const wcPvPhoneErrorMap = wcPvJson.validationErrors || {};

	// Select the billing and shipping phone fields by their IDs.
	const billingPhoneField  = document.getElementById('billing_phone');
	const shippingPhoneField = document.getElementById('shipping_phone');

	/**
	 * Initialize intlTelInput on a given phone field with provided options.
	 *
	 * @param {HTMLElement} phoneField - The phone input field element.
	 * @returns {Object|null} - Returns the intlTelInput instance or null if initialization fails.
	 */
	function initializeIntlTelInput(phoneField) {
		if (!phoneField) {
			return null;
		}

		// Store the original value before initializing intlTelInput.
		const originalValue = phoneField.value;

		// Initialize intlTelInput with desired options.
		const intlInstance = window.intlTelInput(
			phoneField,
			{
				initialCountry: (wcPvJson.defaultCountry === '' || typeof wcPvJson.defaultCountry === 'undefined') ? 'US' : wcPvJson.defaultCountry,
				separateDialCode: wcPvJson.separateDialCode,
				preferredCountries: wcPvJson.preferredCountries && wcPvJson.preferredCountries.length > 0 ? wcPvJson.preferredCountries.map(country => country.toUpperCase()) : [],
				utilsScript: wcPvJson.utilsScript,
				allowDropdown: wcPvJson.allowDropdown,
				excludeCountries: wcPvJson.excludeCountries && wcPvJson.excludeCountries.length > 0 ? wcPvJson.excludeCountries.map(country => country.toUpperCase()) : [],
				// Prevent auto-formatting.
				nationalMode: true,
				formatOnDisplay: false,
			}
		);

		// Restore the original value to prevent auto-formatting.
		setTimeout(() => {
			phoneField.value = originalValue;
		}, 0);

		return intlInstance;
	}

	// Initialize intlTelInput on billing_phone and shipping_phone.
	let wcPvBillingPhoneIntl  = initializeIntlTelInput(billingPhoneField);
	let wcPvShippingPhoneIntl = initializeIntlTelInput(shippingPhoneField);

	// Check if intlTelInput is loaded properly.
	if (wcPvBillingPhoneIntl && typeof wcPvBillingPhoneIntl.isValidNumber !== 'function') {
		console.error('intlTelInput is not fully loaded. utilsScript might be missing.');
	}
	if (wcPvShippingPhoneIntl && typeof wcPvShippingPhoneIntl.isValidNumber !== 'function') {
		console.error('intlTelInput is not fully loaded. utilsScript might be missing.');
	}

	// Separate variables to store phone error messages.
	let wcPvPhoneErrorMsgBilling  = '';
	let wcPvPhoneErrorMsgShipping = '';

	/**
	 * Validates the phone number using intlTelInput plugin.
	 *
	 * @param {Object} phoneInstance - The intlTelInput instance.
	 * @param {string} fieldType - 'Billing' or 'Shipping'.
	 * @returns {Object} - Returns an object with formattedNumber and errorMessage.
	 */
	function wcPvValidatePhone(phoneInstance, fieldType) {
		let formattedNumber = false;
		let errorMessage    = '';

		if (phoneInstance && phoneInstance.isValidNumber()) {
			// Use INTERNATIONAL format to include space between country code and number.
			formattedNumber = phoneInstance.getNumber(intlTelInputUtils.numberFormat.INTERNATIONAL);
		} else if (phoneInstance) {
			const validationError = phoneInstance.getValidationError();
			errorMessage          = `${fieldType} phone ${wcPvPhoneErrorMap[validationError] !== undefined ? wcPvPhoneErrorMap[validationError] : wcPvJson.phoneUnknownErrorMsg}`;
		}

		return {
			formattedNumber: formattedNumber,
			errorMessage: errorMessage
		};
	}

	/**
	 * Processes the phone validation and updates the form accordingly.
	 *
	 * @param {HTMLElement} formContainer - The form container element.
	 * @returns {Object} - Returns an object indicating if there was an error.
	 */
	function wcPvValidateProcess(formContainer) {
		let isShippingAddress = false;
		if (wcPvJson.currentPage === 'checkout') {
			const shipToDifferentAddressCheckbox = document.getElementById('ship-to-different-address-checkbox');
			isShippingAddress = shipToDifferentAddressCheckbox && shipToDifferentAddressCheckbox.checked;
		} else if (wcPvJson.currentPage === 'account') {
			// For the account page, always validate shipping phone if shipping field is present.
			isShippingAddress = (shippingPhoneField !== null);
		}

		const billingValidation  = wcPvBillingPhoneIntl ? wcPvValidatePhone(wcPvBillingPhoneIntl, 'Billing') : { formattedNumber: false, errorMessage: '' };
		const shippingValidation = wcPvShippingPhoneIntl && isShippingAddress ? wcPvValidatePhone(wcPvShippingPhoneIntl, 'Shipping') : { formattedNumber: false, errorMessage: '' };

		// Reset error messages.
		wcPvPhoneErrorMsgBilling  = '';
		wcPvPhoneErrorMsgShipping = '';

		// Handle billing phone validation.
		if (billingPhoneField) {
			// Remove existing billing error message if any.
			const existingBillingError = document.getElementById('wc-ls-phone-valid-field-err-msg-billing');
			if (existingBillingError) {
				existingBillingError.remove();
			}

			// Remove existing billing valid phone field if any.
			const existingBillingValid = document.getElementById('wc-ls-phone-valid-field-billing');
			if (existingBillingValid) {
				existingBillingValid.remove();
			}

			if (billingValidation.formattedNumber) {
				// Set the validated phone number in the billing phone input.
				billingPhoneField.value = billingValidation.formattedNumber;

				// Add a hidden input with the validated phone number if it doesn't exist.
				if (!document.getElementById('wc-ls-phone-valid-field-billing')) {
					const hiddenInput = document.createElement('input');
					hiddenInput.type  = 'hidden';
					hiddenInput.id    = 'wc-ls-phone-valid-field-billing';
					hiddenInput.name  = 'pnv_phone_valid_billing';
					hiddenInput.value = billingValidation.formattedNumber;
					formContainer.appendChild(hiddenInput);
				}
			} else {
				// Set the billing error message.
				wcPvPhoneErrorMsgBilling = billingValidation.errorMessage;

				// Append an error message as a hidden input.
				const errorInput = document.createElement('input');
				errorInput.type  = 'hidden';
				errorInput.id    = 'wc-ls-phone-valid-field-err-msg-billing';
				errorInput.name  = 'pnv_phone_error_billing';
				errorInput.value = wcPvPhoneErrorMsgBilling;
				formContainer.appendChild(errorInput);

				// Ensure valid phone field is removed if present.
				const validPhoneField = document.getElementById('wc-ls-phone-valid-field-billing');
				if (validPhoneField) {
					validPhoneField.remove();
				}
			}
		}

		// Handle shipping phone validation only if shipping address is used.
		if (shippingPhoneField && isShippingAddress) {
			// Remove existing shipping error message if any.
			const existingShippingError = document.getElementById('wc-ls-phone-valid-field-err-msg-shipping');
			if (existingShippingError) {
				existingShippingError.remove();
			}

			// Remove existing shipping valid phone field if any.
			const existingShippingValid = document.getElementById('wc-ls-phone-valid-field-shipping');
			if (existingShippingValid) {
				existingShippingValid.remove();
			}

			if (shippingValidation.formattedNumber) {
				// Set the validated phone number in the shipping phone input.
				shippingPhoneField.value = shippingValidation.formattedNumber;

				// Add a hidden input with the validated phone number if it doesn't exist.
				if (!document.getElementById('wc-ls-phone-valid-field-shipping')) {
					const hiddenInput = document.createElement('input');
					hiddenInput.type  = 'hidden';
					hiddenInput.id    = 'wc-ls-phone-valid-field-shipping';
					hiddenInput.name  = 'pnv_phone_valid_shipping';
					hiddenInput.value = shippingValidation.formattedNumber;
					formContainer.appendChild(hiddenInput);
				}
			} else {
				// Set the shipping error message.
				wcPvPhoneErrorMsgShipping = shippingValidation.errorMessage;

				// Append an error message as a hidden input.
				const errorInput = document.createElement('input');
				errorInput.type  = 'hidden';
				errorInput.id    = 'wc-ls-phone-valid-field-err-msg-shipping';
				errorInput.name  = 'pnv_phone_error_shipping';
				errorInput.value = wcPvPhoneErrorMsgShipping;
				formContainer.appendChild(errorInput);

				// Ensure valid phone field is removed if present.
				const validPhoneField = document.getElementById('wc-ls-phone-valid-field-shipping');
				if (validPhoneField) {
					validPhoneField.remove();
				}
			}
		}

		let hasError = false;
		if (wcPvPhoneErrorMsgBilling !== '' || wcPvPhoneErrorMsgShipping !== '') {
			hasError = true;
		}

		return { hasError: hasError };
	}

	/**
	 * Display inline error messages on My Account page if validation fails.
	 *
	 * @param {string} billingError - The billing error message (if any).
	 * @param {string} shippingError - The shipping error message (if any).
	 */
	function wcPvShowInlineErrors(billingError, shippingError) {
		// Remove any existing notices.
		const existingNotices = document.querySelectorAll('.woocommerce-NoticeGroup-checkout');
		existingNotices.forEach(function (notice) {
			notice.remove();
		});

		// Create notice container.
		const noticeGroup = document.createElement('div');
		noticeGroup.className = 'woocommerce-NoticeGroup woocommerce-NoticeGroup-checkout';

		const ul = document.createElement('ul');
		ul.className = 'woocommerce-error';
		ul.setAttribute('role', 'alert');

		// Add billing error if present.
		if (billingError !== '') {
			const li = document.createElement('li');
			li.textContent = billingError;
			ul.appendChild(li);
		}

		// Add shipping error if present.
		if (shippingError !== '') {
			const li = document.createElement('li');
			li.textContent = shippingError;
			ul.appendChild(li);
		}

		// Append only if there's at least one error message.
		if (ul.children.length > 0) {
			noticeGroup.appendChild(ul);

			// Insert the error messages at the top of the container.
			const accountContainer = document.querySelector(wcPvJson.parentPage);
			if (accountContainer) {
				accountContainer.insertBefore(noticeGroup, accountContainer.firstChild);
			}

			// Scroll to top to show the error notices.
			window.scrollTo({ top: 0, behavior: 'smooth' });
		}
	}

	/**
	 * Event listener for changes in the billing country select field.
	 */
	const billingCountrySelect = document.querySelector(`${wcPvJson.parentPage} #billing_country`);

	// If no saved user phone is present, then update the intlTelInput from the billing select on page load.
	if (!wcPvJson.userPhone && billingCountrySelect && wcPvBillingPhoneIntl) {
		wcPvBillingPhoneIntl.setCountry(billingCountrySelect.value);
	}

	if (billingCountrySelect) {
		billingCountrySelect.addEventListener('change', function () {
			const selectedCountry = this.value;
			// Only override if no saved user phone (i.e. user phone has priority).
			if (!wcPvJson.userPhone && (wcPvJson.onlyCountries.length === 0 || wcPvJson.onlyCountries.includes(selectedCountry.toUpperCase()))) {
				if (wcPvBillingPhoneIntl) {
					wcPvBillingPhoneIntl.setCountry(selectedCountry);
				}
			}
		});
	}

	// Similar logic for shipping country.
	const shippingCountrySelect = document.querySelector(`${wcPvJson.parentPage} #shipping_country`);

	// If no saved user phone is present, then update the intlTelInput from the shipping select on page load.
	if (!wcPvJson.userPhone && shippingCountrySelect && wcPvShippingPhoneIntl) {
		wcPvShippingPhoneIntl.setCountry(shippingCountrySelect.value);
	}

	if (shippingCountrySelect) {
		shippingCountrySelect.addEventListener('change', function () {
			const selectedCountry = this.value;
			// Only override if no saved user phone.
			if (!wcPvJson.userPhone && (wcPvJson.onlyCountries.length === 0 || wcPvJson.onlyCountries.includes(selectedCountry.toUpperCase()))) {
				if (wcPvShippingPhoneIntl) {
					wcPvShippingPhoneIntl.setCountry(selectedCountry);
				}
			}
		});
	}

	/**
	 * Adds a specific class to phone fields if separate dial code is enabled.
	 */
	if (wcPvJson.separateDialCode === true) {
		if (billingPhoneField) {
			billingPhoneField.classList.add('pnv-separate-dial-code');
		}
		if (shippingPhoneField) {
			shippingPhoneField.classList.add('pnv-separate-dial-code');
		}
	}

	/**
	 * Determines the current page and binds appropriate event listeners.
	 */
	if (wcPvJson.currentPage === 'checkout') {
		const checkoutForm = document.querySelector(`${wcPvJson.parentPage} form.checkout`);
		if (checkoutForm) {
			if (window.jQuery) {
				// Validate and rely on server-side for displaying errors.
				window.jQuery(checkoutForm).on('checkout_place_order', function () {
					wcPvValidateProcess(checkoutForm);
				});
			} else {
				checkoutForm.addEventListener('submit', function () {
					wcPvValidateProcess(checkoutForm);
				});
			}
		}
	} else if (wcPvJson.currentPage === 'account') {
		const accountForm = document.querySelector(`${wcPvJson.parentPage} form`);
		if (accountForm) {
			accountForm.addEventListener('submit', function (event) {
				const result = wcPvValidateProcess(accountForm);
				// If errors are present and alwaysAllowCheckout is false, prevent submission.
				if (result.hasError && !wcPvJson.alwaysAllowCheckout) {
					event.preventDefault();
					// Show inline error messages on my-account page.
					wcPvShowInlineErrors(wcPvPhoneErrorMsgBilling, wcPvPhoneErrorMsgShipping);
				}
			});
		}
	}

});
