<?php
/**
 * The plugin settings.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes/admin
 */

namespace PNV\Phone_Number_Validation\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use PNV\Phone_Number_Validation\Phone_Number_Validation;

/**
 * Class Settings
 *
 * Handles the Phone Validation settings in WooCommerce.
 */
class Settings {

	/**
	 * Singleton instance.
	 *
	 * @var Settings|null
	 */
	private static $_instance = null; // phpcs:ignore

	/**
	 * Cached result of block checkout check.
	 *
	 * @var bool|null
	 */
	private $block_checkout_enabled = null;

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_filter( 'woocommerce_get_sections_shipping', array( $this, 'add_settings_section' ) );
		add_filter( 'woocommerce_get_settings_shipping', array( $this, 'add_settings_fields' ), 10, 2 );

		// Enqueue admin scripts for enhanced select fields.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}

	/**
	 * Get the singleton instance.
	 *
	 * @return Settings
	 */
	public static function get_instance(): self {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Add a new section to WooCommerce Shipping settings.
	 *
	 * @param array $sections Existing sections.
	 * @return array Modified sections.
	 */
	public function add_settings_section( array $sections ): array {
		$sections['pnv_settings'] = __( 'Phone Validation', 'phone-number-validation' );
		return $sections;
	}

	/**
	 * Add settings fields to the new section.
	 *
	 * @param array  $settings Existing settings.
	 * @param string $current_section Current section ID.
	 * @return array Modified settings.
	 */
	public function add_settings_fields( array $settings, string $current_section ): array {
		if ( 'pnv_settings' !== $current_section ) {
			return $settings;
		}

		$updated_settings = array();

		// Add a title.
		$updated_settings[] = array(
			'title' => __( 'Phone Validation Settings', 'phone-number-validation' ),
			'type'  => 'title',
			'id'    => 'pnv_settings_general',
		);

		/**
		 * Conditionally add the "Add Shipping Phone" and related settings
		 * only if the WooCommerce Checkout Block is NOT enabled.
		 */
		if ( ! $this->is_block_checkout_enabled() ) {
			// Add the "Add Shipping Phone" checkbox.
			$updated_settings[] = array(
				'title'    => __( 'Add shipping phone field', 'phone-number-validation' ),
				'desc'     => __( 'Add a shipping phone number input field to the shipping form.', 'phone-number-validation' ),
				'id'       => 'pnv_add_shipping_phone',
				'default'  => 'no',
				'type'     => 'checkbox',
				'desc_tip' => false,
			);

			// "Make Shipping Phone Required" Checkbox.
			$updated_settings[] = array(
				'title'    => __( 'Make shipping phone required', 'phone-number-validation' ),
				'desc'     => __( 'Require customers to enter a shipping phone number. Otherwise this field will display as optional.', 'phone-number-validation' ),
				'id'       => 'pnv_make_shipping_phone_required',
				'default'  => 'no',
				'type'     => 'checkbox',
				'desc_tip' => false,
				'css'      => 'margin-left:20px;',
			);
		}

		// Add Initial Country setting.
		$updated_settings[] = array(
			'title'             => __( 'Default country code', 'phone-number-validation' ),
			'desc'              => __( 'The fallback initial country code/flag shown when billing/shipping country isn\'t set during checkout.', 'phone-number-validation' ),
			'id'                => 'pnv_initial_country',
			'default'           => 'US',
			'type'              => 'select',
			'options'           => $this->get_country_options(),
			'desc_tip'          => false,
			'custom_attributes' => array(
				'data-placeholder' => __( 'Select a country', 'phone-number-validation' ),
			),
			'class'             => 'wc-enhanced-select',
		);

		// Add Preferred Countries setting (Multiselect).
		$updated_settings[] = array(
			'title'             => __( 'Preferred countries', 'phone-number-validation' ),
			'desc'              => __( 'The preferred countries to appear at the top of the dropdown list during checkout.', 'phone-number-validation' ),
			'id'                => 'pnv_preferred_countries',
			'default'           => array( 'US', 'GB' ),
			'type'              => 'multiselect',
			'options'           => $this->get_country_options(),
			'desc_tip'          => false,
			'css'               => 'width:400px;',
			'custom_attributes' => array(
				'data-placeholder' => __( 'Select preferred countries', 'phone-number-validation' ),
			),
			'class'             => 'wc-enhanced-select',
		);

		// Add Separate Dial Code setting.
		$updated_settings[] = array(
			'title'    => __( 'Show country codes', 'phone-number-validation' ),
			'desc'     => __( 'Show country codes/flags next to each other on the checkout phone field.', 'phone-number-validation' ),
			'id'       => 'pnv_separate_dial_code',
			'default'  => 'yes',
			'type'     => 'checkbox',
			'desc_tip' => false,
		);

		// Add Allow Dropdown setting.
		$updated_settings[] = array(
			'title'    => __( 'Allow dropdown', 'phone-number-validation' ),
			'desc'     => __( 'Allow users to select country from a dropdown.', 'phone-number-validation' ),
			'id'       => 'pnv_allow_dropdown',
			'default'  => 'yes',
			'type'     => 'checkbox',
			'desc_tip' => false,
		);

		// Add Exclude Countries setting (Multiselect).
		$updated_settings[] = array(
			'title'             => __( 'Exclude countries', 'phone-number-validation' ),
			'desc'              => __( 'Select countries to exclude from the country dropdown.', 'phone-number-validation' ),
			'id'                => 'pnv_exclude_countries',
			'default'           => array( 'RU' ),
			'type'              => 'multiselect',
			'options'           => $this->get_country_options(),
			'desc_tip'          => false,
			'css'               => 'width:400px;',
			'custom_attributes' => array(
				'data-placeholder' => __( 'Select countries to exclude', 'phone-number-validation' ),
			),
			'class'             => 'wc-enhanced-select',
		);

		/**
		 * Always Allow Checkout.
		 */
		$updated_settings[] = array(
			'title'    => __( 'Always allow checkout', 'phone-number-validation' ),
			'desc'     => __( 'Allow checkout to proceed even when the phone number is incorrect.', 'phone-number-validation' ),
			'id'       => 'pnv_always_allow_checkout',
			'default'  => 'yes',
			'type'     => 'checkbox',
			'desc_tip' => false,
		);

		// Adds a checkbox to enable or disable debug logging.
		$updated_settings[] = array(
			'title'    => __( 'Enable debug', 'phone-number-validation' ),
			'desc'     => __( 'Enable debug logging for phone number validation.', 'phone-number-validation' ),
			'id'       => 'pnv_enable_debug',
			'default'  => 'no',
			'type'     => 'checkbox',
			'desc_tip' => false,
		);

		// Add a section end.
		$updated_settings[] = array(
			'type' => 'sectionend',
			'id'   => 'pnv_settings_general',
		);

		return $updated_settings;
	}

	/**
	 * Enqueue admin scripts for enhanced select fields.
	 */
	public function enqueue_admin_scripts() {
		// Only enqueue on WooCommerce settings pages.
		$screen = get_current_screen();
		if ( isset( $screen->id ) && 'woocommerce_page_wc-settings' === $screen->id ) {
			// Enqueue custom admin JS for initializing enhanced selects if needed.
			wp_enqueue_script(
				'pnv_admin_js',
				\PNV_ASSETS_URL . 'js/admin.js',
				array( 'jquery', 'wc-enhanced-select' ),
				\PNV_PLUGIN_VERSION,
				true
			);

			// Pass all countries to JS via wp_localize_script.
			wp_localize_script(
				'pnv_admin_js',
				'pnvAdminData',
				array(
					'allCountries' => $this->get_country_options(),
				)
			);
		}
	}

	/**
	 * Get country options for select fields.
	 *
	 * @return array
	 */
	private function get_country_options(): array {
		$countries = WC()->countries->get_countries();
		$options   = array();

		foreach ( $countries as $code => $name ) {
			$options[ $code ] = $name;
		}

		return $options;
	}

	/**
	 * Check if Shipping Phone is enabled.
	 *
	 * @return bool
	 */
	public function is_shipping_phone_enabled(): bool {
		return 'yes' === get_option( 'pnv_add_shipping_phone', 'no' );
	}

	/**
	 * Check if Making Shipping Phone Required is enabled.
	 *
	 * @return bool
	 */
	public function is_make_shipping_phone_required(): bool {
		return 'yes' === get_option( 'pnv_make_shipping_phone_required', 'no' );
	}

	/**
	 * Get Initial Country.
	 *
	 * @return string
	 */
	public function get_initial_country(): string {
		return sanitize_text_field( get_option( 'pnv_initial_country', 'US' ) );
	}

	/**
	 * Get Preferred Countries.
	 *
	 * @return array
	 */
	public function get_preferred_countries(): array {
		$countries = get_option( 'pnv_preferred_countries', array( 'US', 'GB' ) );
		if ( empty( $countries ) || ! is_array( $countries ) ) {
			return array();
		}
		return array_map( 'strtoupper', array_map( 'trim', $countries ) );
	}

	/**
	 * Check if Separate Dial Code is enabled.
	 *
	 * @return bool
	 */
	public function separate_dial_code(): bool {
		return 'yes' === get_option( 'pnv_separate_dial_code', 'yes' );
	}

	/**
	 * Check if Allow Dropdown is enabled.
	 *
	 * @return bool
	 */
	public function allow_dropdown(): bool {
		return 'yes' === get_option( 'pnv_allow_dropdown', 'yes' );
	}

	/**
	 * Get Exclude Countries.
	 *
	 * @return array
	 */
	public function get_exclude_countries(): array {
		$countries = get_option( 'pnv_exclude_countries', array( 'RU' ) );
		if ( empty( $countries ) || ! is_array( $countries ) ) {
			$countries = array();
		}
		$countries = array_map( 'strtoupper', array_map( 'trim', $countries ) );
		
		// Make the list of excluded countries filterable for the phone number dropdown.
		$countries = apply_filters( 'pnv_exclude_countries', $countries );
		return $countries;
	}

	/**
	 * Check if Debug is Enabled.
	 *
	 * @return bool
	 */
	public function is_debug_enabled(): bool {
		return 'yes' === get_option( 'pnv_enable_debug', 'no' );
	}

	/**
	 * Check if Always Allow Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_always_allow_checkout(): bool {
		return 'yes' === get_option( 'pnv_always_allow_checkout', 'yes' );
	}

	/**
	 * Determine if WooCommerce Block Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_block_checkout_enabled(): bool {
		if ( null !== $this->block_checkout_enabled ) {
			return $this->block_checkout_enabled;
		}

		$checkout_page_id = wc_get_page_id( 'checkout' );

		if ( ! $checkout_page_id || -1 === $checkout_page_id ) {
			// Checkout page not set or invalid.
			$this->block_checkout_enabled = false;
			return false;
		}

		// Get the post content of the checkout page.
		$post_content = get_post_field( 'post_content', $checkout_page_id );

		if ( ! $post_content ) {
			// No content found.
			$this->block_checkout_enabled = false;
			return false;
		}

		// Check for the presence of the WooCommerce checkout block.
		// The block is usually represented by <!-- wp:woocommerce/checkout -->.
		if ( strpos( $post_content, '<!-- wp:woocommerce/checkout' ) !== false ) {
			$this->block_checkout_enabled = true;
			return true;
		}

		$this->block_checkout_enabled = false;
		return false;
	}
}
