<?php
/**
 * The main plugin logic.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes
 */

namespace PNV\Phone_Number_Validation;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use WC_Logger;
use Automattic\WooCommerce\Utilities\FeaturesUtil;

/**
 * Class Phone_Number_Validation
 *
 * Main plugin class.
 */
class Phone_Number_Validation {

	/**
	 * Singleton instance.
	 *
	 * @var Phone_Number_Validation|null
	 */
	private static $_instance = null; // phpcs:ignore

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $woo_custom_field_meta = array(
		'billing_hidden_phone_field'     => '_pnv_phone_validator',
		'billing_hidden_phone_err_field' => '_pnv_phone_validator_err',
	);

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	private $version = PNV_PLUGIN_VERSION;

	/**
	 * Settings instance.
	 *
	 * @var Admin\Settings
	 */
	private $settings;

	/**
	 * Constructor.
	 */
	private function __construct() {
		if ( Dependencies::is_woocommerce_active() ) {
			$this->define_constants();
			$this->includes();
			$this->init_hooks();
		} else {
			add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
		}
	}

	/**
	 * Get the singleton instance.
	 *
	 * @return Phone_Number_Validation
	 */
	public static function get_instance(): self {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Define additional constants.
	 */
	private function define_constants() {
		if ( ! defined( 'PNV_ABSPATH' ) ) {
			define( 'PNV_ABSPATH', plugin_dir_path( PNV_PLUGIN_FILE ) . '/' );
		}
		if ( ! defined( 'PNV_PLUGIN_BASE' ) ) {
			define( 'PNV_PLUGIN_BASE', plugin_basename( PNV_PLUGIN_FILE ) );
		}
		if ( ! defined( 'PNV_ASSETS_URL' ) ) {
			define( 'PNV_ASSETS_URL', plugins_url( 'assets/', PNV_PLUGIN_FILE ) );
		}
	}

	/**
	 * Include necessary files.
	 */
	private function includes() {
		// Frontend classes.
		require_once PNV_ABSPATH . 'public/class-checkout.php';
		require_once PNV_ABSPATH . 'public/class-my-account.php';

		// Admin classes.
		require_once PNV_ABSPATH . 'includes/admin/class-settings.php';

		// Initialize frontend classes.
		new Frontend\Checkout( $this );
		new Frontend\My_Account( $this );

		// Initialize admin classes.
		$this->settings = Admin\Settings::get_instance();
	}

	/**
	 * Initialize plugin hooks.
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'load_textdomain' ) );
		do_action( 'pnv_init' );

		// Declare HPOS and block compatibility before WooCommerce initializes.
		add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ), 10 );
		add_action( 'before_woocommerce_init', array( $this, 'declare_block_compatibility' ), 10 );

		// Use woocommerce_checkout_process for server-side validation.
		add_action( 'woocommerce_checkout_process', array( $this, 'checkout_validate' ) );

		// Add plugin action links.
		add_filter( 'plugin_action_links_' . PNV_PLUGIN_BASE, array( $this, 'add_plugin_action_links' ) );

		// Add admin notice for block checkout phone field requirement.
		add_action( 'admin_notices', array( $this, 'check_phone_field_requirement' ) );
	}

	/**
	 * Declare HPOS compatibility.
	 */
	public function declare_hpos_compatibility() {
		if ( class_exists( FeaturesUtil::class ) ) {
			FeaturesUtil::declare_compatibility( 'custom_order_tables', 'phone-number-validation/phone-number-validation.php', true );
		}
	}

	/**
	 * Declare Cart/Checkout Block compatibility.
	 */
	public function declare_block_compatibility() {
		if ( class_exists( FeaturesUtil::class ) ) {
			FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', 'phone-number-validation/phone-number-validation.php', true );
		}
	}

	/**
	 * Load the plugin textdomain for translations.
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'phone-number-validation', false, dirname( plugin_basename( PNV_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * Display admin notices if dependencies are missing.
	 */
	public function admin_notices() {
		echo '<div class="error"><p>';
		echo wp_kses_post(
			sprintf(
				/* translators: %s: WooCommerce plugin URL */
				__( 'The <strong>Phone Number Validation</strong> plugin requires <a href="%s" target="_blank">WooCommerce</a> to be active. Please ensure it is installed and activated.', 'phone-number-validation' ),
				'https://wordpress.org/plugins/woocommerce/'
			)
		);
		echo '</p></div>';
	}

	/**
	 * Check if we're on the checkout page.
	 *
	 * @return bool
	 */
	public function is_checkout(): bool {
		$id = get_option( 'woocommerce_checkout_page_id', false );
		return is_page( $id );
	}

	/**
	 * Check if we're on the my account page.
	 *
	 * @return bool
	 */
	public function is_account_page(): bool {
		$id = get_option( 'woocommerce_myaccount_page_id', false );
		return is_page( $id );
	}

	/**
	 * Get validation errors.
	 *
	 * @return array
	 */
	public function get_validation_errors(): array {
		$errors = array(
			1 => __( 'has an invalid country code.', 'phone-number-validation' ),
			2 => __( 'number is too short.', 'phone-number-validation' ),
			3 => __( 'number is too long.', 'phone-number-validation' ),
			4 => __( 'is an invalid number.', 'phone-number-validation' ),
		);

		return $errors;
	}

	/**
	 * Separate dial code filter.
	 *
	 * @param bool $value Whether to separate the dial code.
	 * @return bool
	 */
	public function separate_dial_code( bool $value = false ): bool {
		return apply_filters( 'pnv_separate_dial_code', $value );
	}

	/**
	 * Use default WooCommerce store country.
	 *
	 * @return bool
	 */
	public function use_wc_store_default_country(): bool {
		return apply_filters( 'pnv_use_wc_default_store_country', false );
	}

	/**
	 * Get default country.
	 *
	 * @return string
	 */
	public function get_default_country(): string {
		$default = '';

		if ( $this->use_wc_store_default_country() ) {
			$default = apply_filters( 'woocommerce_get_base_location', get_option( 'woocommerce_default_country' ) );

			// Remove sub-states.
			if ( strpos( $default, ':' ) !== false ) {
				list( $country, $state ) = explode( ':', $default );
				$default                 = $country;
			}
		}

		return apply_filters( 'pnv_set_default_country', $default );
	}

	/**
	 * Get preferred countries.
	 *
	 * @return array
	 */
	public function get_preferred_countries(): array {
		return apply_filters( 'pnv_preferred_countries', array() );
	}

	/**
	 * Check if Separate Dial Code is enabled.
	 *
	 * @return bool
	 */
	public function separate_dial_code_enabled(): bool {
		return 'yes' === get_option( 'pnv_separate_dial_code', 'yes' );
	}

	/**
	 * Check if Allow Dropdown is enabled.
	 *
	 * @return bool
	 */
	public function allow_dropdown(): bool {
		return 'yes' === get_option( 'pnv_allow_dropdown', 'yes' );
	}

	/**
	 * Get Exclude Countries.
	 *
	 * @return array
	 */
	public function get_exclude_countries(): array {
		$countries = get_option( 'pnv_exclude_countries', array( 'RU' ) );
		if ( empty( $countries ) || ! is_array( $countries ) ) {
			return array();
		}
		return array_map( 'strtoupper', array_map( 'trim', $countries ) );
	}

	/**
	 * Check if Debug is Enabled.
	 *
	 * @return bool
	 */
	public function is_debug_enabled(): bool {
		return 'yes' === get_option( 'pnv_enable_debug', 'no' );
	}

	/**
	 * Check if Always Allow Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_always_allow_checkout(): bool {
		return 'yes' === get_option( 'pnv_always_allow_checkout', 'yes' );
	}

	/**
	 * Determine if WooCommerce Block Checkout is enabled.
	 *
	 * @return bool
	 */
	public function is_block_checkout_enabled(): bool {
		// Use the Settings class to determine block checkout status.
		return $this->settings->is_block_checkout_enabled();
	}

	/**
	 * Check phone field status in WooCommerce Block Checkout and display admin notice if needed.
	 */
	public function check_phone_field_requirement() {
		// Only show notice to users who can manage WooCommerce settings.
		// phpcs:ignore WordPress.WP.Capabilities.Unknown -- Capability added by WooCommerce core.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		// Check if block checkout is enabled.
		if ( ! $this->is_block_checkout_enabled() ) {
			return; // Block checkout not in use; no need for this notice.
		}

		// Get the checkout page ID.
		$checkout_page_id = wc_get_page_id( 'checkout' );

		if ( ! $checkout_page_id || -1 === $checkout_page_id ) {
			// Checkout page not set or invalid.
			return;
		}

		// Get the post content of the checkout page.
		$post_content = get_post_field( 'post_content', $checkout_page_id );

		if ( ! $post_content ) {
			// No content found.
			return;
		}

		// Parse the blocks.
		$blocks = parse_blocks( $post_content );

		$phone_field_enabled = false;

		foreach ( $blocks as $block ) {
			if ( 'woocommerce/checkout' === $block['blockName'] ) {
				// Check block attributes for phone field settings.
				$attributes = isset( $block['attrs'] ) ? $block['attrs'] : array();

				// The actual attribute names depend on how WooCommerce defines them.
				// Adjust these keys based on the actual implementation.
				$show_phone = isset( $attributes['showPhoneField'] ) ? (bool) $attributes['showPhoneField'] : true;

				$phone_field_enabled = $show_phone;

				break; // Assuming only one checkout block is present.
			}
		}

		// If phone field is not enabled or not required, display admin notice.
		if ( ! $phone_field_enabled ) {
			// Get the edit post link for the checkout page.
			$checkout_edit_url = get_edit_post_link( $checkout_page_id );

			if ( ! $checkout_edit_url ) {
				// Fallback to admin URL if edit link is not available.
				$checkout_edit_url = admin_url( 'edit.php?post_type=page&page=wc-settings&tab=checkout' );
			}

			?>
			<div class="notice notice-error">
				<p>
					<strong><?php esc_html_e( 'Phone Number Validation: Phone Field Disabled', 'phone-number-validation' ); ?></strong>
				</p>
				<p>
					<?php
					esc_html_e( 'The Phone Number Validation plugin requires the phone number field to be enabled in the WooCommerce Checkout Block. Please edit your Checkout block to enable phone number validation or switch to classic (i.e. "Shortcode") checkout.', 'phone-number-validation' );
					?>
				</p>
				<p>
					<a href="<?php echo esc_url( $checkout_edit_url ); ?>" class="button button-primary">
						<?php esc_html_e( 'Edit Checkout Block', 'phone-number-validation' ); ?>
					</a>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Get current user's billing phone.
	 *
	 * @return string
	 */
	public function get_current_user_phone(): string {
		return get_user_meta( get_current_user_id(), 'billing_phone', true );
	}

	/**
	 * Get billing custom field meta.
	 *
	 * @return array
	 */
	public function get_billing_custom_field_meta(): array {
		return $this->woo_custom_field_meta;
	}

	/**
	 * Get plugin URL.
	 *
	 * @return string
	 */
	public function plugin_url(): string {
		return untrailingslashit( PNV_ASSETS_URL );
	}

	/**
	 * Get plugin text domain.
	 *
	 * @return string
	 */
	public function get_text_domain(): string {
		return 'phone-number-validation';
	}

	/**
	 * Get plugin version.
	 *
	 * @return string
	 */
	public function get_version(): string {
		return $this->version;
	}

	/**
	 * Validate the phone number on checkout.
	 */
	public function checkout_validate(): void {
		$phone_err_name = $this->woo_custom_field_meta['billing_hidden_phone_err_field'];
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_err_field = isset( $_POST[ $phone_err_name ] ) ? trim( sanitize_text_field( wp_unslash( $_POST[ $phone_err_name ] ) ) ) : '';

		if ( ! empty( $phone_err_field ) ) {
			$phone_err_msg = __( 'Phone number validation - error: ', 'phone-number-validation' ) . esc_html( $phone_err_field );
			$this->log_info( $phone_err_msg );
			wc_add_notice( $phone_err_msg, 'error' );
		}
	}

	/**
	 * Add custom action links to the plugin entry on the Plugins page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_plugin_action_links( array $links ): array {
		$settings_url  = admin_url( 'admin.php?page=wc-settings&tab=shipping&section=pnv_settings' );
		$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . __( 'Settings', 'phone-number-validation' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Log an informational message.
	 *
	 * @param string $message The message to log.
	 */
	public function log_info( string $message ): void {
		if ( $this->settings->is_debug_enabled() ) {
			$logger  = wc_get_logger();
			$context = array( 'source' => 'phone-number-validation' );
			$logger->info( $message, $context );
		}
	}

	/**
	 * Log an error message.
	 *
	 * @param string $message The message to log.
	 */
	public function log_error( string $message ): void {
		if ( $this->settings->is_debug_enabled() ) {
			$logger  = wc_get_logger();
			$context = array( 'source' => 'phone-number-validation' );
			$logger->error( $message, $context );
		}
	}
}
