<?php
/**
 * Add phone number validation to WooCommerce checkout page.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/includes
 */

declare(strict_types=1);

namespace PNV\Phone_Number_Validation\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use PNV\Phone_Number_Validation\Phone_Number_Validation;
use PNV\Phone_Number_Validation\Admin\Settings;

/**
 * Class Checkout
 *
 * Handles checkout fields and validation.
 */
class Checkout {

	/**
	 * Plugin instance.
	 *
	 * @var Phone_Number_Validation
	 */
	private $plugin;

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $custom_field_meta;

	/**
	 * CDN URLs for intlTelInput assets.
	 *
	 * @var array
	 */
	private $intl_tel_input_urls = array(
		'css'   => PNV_ASSETS_URL . 'vendor/intl-tel-input/css/intlTelInput.css',
		'js'    => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/intlTelInput.min.js',
		'utils' => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/utils.js',
	);

	/**
	 * Constructor.
	 *
	 * @param Phone_Number_Validation $plugin Plugin instance.
	 */
	public function __construct( Phone_Number_Validation $plugin ) {
		$this->plugin            = $plugin;
		$this->custom_field_meta = $this->plugin->get_billing_custom_field_meta();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks(): void {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_filter( 'woocommerce_billing_fields', array( $this, 'add_billing_fields' ), 20, 1 );
		add_action( 'woocommerce_after_checkout_validation', array( $this, 'checkout_validate' ), 10, 1 );
		add_filter( 'woocommerce_checkout_fields', array( $this, 'add_shipping_phone_field' ), 20, 1 );
		add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'save_shipping_phone_field' ), 10, 1 );
	}

	/**
	 * Enqueue necessary scripts and styles.
	 */
	public function enqueue_scripts(): void {
		if ( ! is_checkout() ) {
			return;
		}

		wp_enqueue_style( 'intlTelInput-css', $this->intl_tel_input_urls['css'], array(), '17.0.19' );
		wp_enqueue_style( 'pnv_css-style', $this->plugin->plugin_url() . '/css/frontend.css', array(), $this->plugin->get_version() );

		wp_enqueue_script( 'intlTelInput-js', $this->intl_tel_input_urls['js'], array(), '17.0.19', true );

		$script_dependencies = array( 'intlTelInput-js' );
		if ( $this->plugin->is_checkout() ) {
			$script_dependencies[] = 'wc-checkout';
		}

		wp_register_script( 'pnv_js-script', $this->plugin->plugin_url() . '/js/frontend.js', $script_dependencies, $this->plugin->get_version(), true );

		// Retrieve settings from the Settings class.
		$settings = Settings::get_instance();

		// Determine default country based on user's saved data.
		$default_country = $settings->get_initial_country();

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			// Fall back to the billing country saved with the address.
			$billing_country = get_user_meta( $user_id, 'billing_country', true );
			if ( ! empty( $billing_country ) ) {
				$default_country = sanitize_text_field( $billing_country );
			}
		}

		$pnv_json = array(
			'phoneValidatorName'    => $this->custom_field_meta['billing_hidden_phone_field'],
			'phoneValidatorErrName' => $this->custom_field_meta['billing_hidden_phone_err_field'],
			'phoneErrorTitle'       => __( 'Phone validation error: ', 'phone-number-validation' ),
			'phoneUnknownErrorMsg'  => __( 'Unknown number', 'phone-number-validation' ),
			'separateDialCode'      => $settings->separate_dial_code(),
			'validationErrors'      => $this->plugin->get_validation_errors(),
			'defaultCountry'        => $default_country,
			'preferredCountries'    => $settings->get_preferred_countries(),
			'utilsScript'           => $this->intl_tel_input_urls['utils'],
			'parentPage'            => '.woocommerce-checkout',
			'currentPage'           => 'checkout',
			'allowDropdown'         => $settings->allow_dropdown(),
			'excludeCountries'      => $settings->get_exclude_countries(),
			'pleaseCorrectPhone'    => __( 'Please correct the phone number(s) before proceeding.', 'phone-number-validation' ),
			'dismissNotice'         => __( 'Dismiss this notice', 'phone-number-validation' ),
			'alwaysAllowCheckout'   => $settings->is_always_allow_checkout(),
		);

		// Optionally include user phone.
		$phone = $this->plugin->get_current_user_phone();
		if ( $phone ) {
			$pnv_json['userPhone'] = $phone;
		}

		wp_localize_script( 'pnv_js-script', 'wcPvJson', $pnv_json );
		wp_enqueue_script( 'pnv_js-script' );

		// Additionally enqueue blocks script if using block checkout.
		if ( wp_is_block_theme() || has_block( 'woocommerce/checkout' ) ) {
			wp_enqueue_script(
				'pnv_blocks-script',
				$this->plugin->plugin_url() . '/js/blocks.js',
				array( 'intlTelInput-js', 'pnv_js-script' ),
				$this->plugin->get_version(),
				true
			);

			// Localize Blocks Script.
			$blocks_json = array(
				'pleaseCorrectPhone'  => __( 'Please correct the phone number(s) before proceeding.', 'phone-number-validation' ),
				'dismissNotice'       => __( 'Dismiss this notice', 'phone-number-validation' ),
				'alwaysAllowCheckout' => $settings->is_always_allow_checkout(),
			);

			wp_localize_script( 'pnv_blocks-script', 'wcPvBlocksJson', $blocks_json );
		}
	}

	/**
	 * Add custom classes to the billing phone field.
	 *
	 * @param array $fields Existing billing fields.
	 * @return array Modified billing fields.
	 */
	public function add_billing_fields( array $fields ): array {
		if ( isset( $fields['billing_phone']['class'] ) && is_array( $fields['billing_phone']['class'] ) ) {
			$fields['billing_phone']['class'] = array_merge( $fields['billing_phone']['class'], array( 'pnv-phone' ) );
		}

		return $fields;
	}

	/**
	 * Add Shipping Phone Field to Checkout.
	 *
	 * @param array $fields Existing checkout fields.
	 * @return array Modified checkout fields.
	 */
	public function add_shipping_phone_field( array $fields ): array {
		// Get the Settings instance.
		$settings = Settings::get_instance();

		// Check if block checkout is NOT enabled, shipping phone is enabled, and ship to destination is not 'billing_only'.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {
			// Determine if the shipping phone should be required based on the setting.
			$is_required = $settings->is_make_shipping_phone_required();

			$fields['shipping']['shipping_phone'] = array(
				'label'       => __( 'Shipping Phone', 'phone-number-validation' ),
				'required'    => $is_required,
				'clear'       => false,
				'type'        => 'tel',
				'id'          => 'shipping_phone',
				'priority'    => 110,
				'class'       => array( 'form-row-wide', 'pnv-phone' ),
				'validate'    => $is_required ? array( 'required' ) : array(),
				'placeholder' => __( 'Enter your shipping phone number', 'phone-number-validation' ),
			);
		}

		return $fields;
	}

	/**
	 * Validate checkout fields.
	 *
	 * @param array $data Posted data from the checkout form.
	 */
	public function checkout_validate( array $data ): void { // phpcs:ignore
		// Retrieve error messages from hidden inputs.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_err_field_billing = sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_billing'] ?? '' ) );
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_err_field_shipping = sanitize_text_field( wp_unslash( $_POST['pnv_phone_error_shipping'] ?? '' ) );

		// Retrieve the Settings instance.
		$settings = Settings::get_instance();

		// Check if Always Allow Checkout is disabled.
		$always_allow_checkout = $settings->is_always_allow_checkout();

		// Add error notices for billing phone.
		if ( ! empty( $phone_err_field_billing ) && ! $always_allow_checkout ) {
			// Use sprintf for better message construction.
			$phone_err_msg_billing = sprintf(
				/* translators: %s: Phone validation error message from client-side validation */
				__( 'Phone validation error: %s', 'phone-number-validation' ),
				esc_html( $phone_err_field_billing )
			);
			$this->plugin->log_info( '[Billing] Phone number validation - error: ' . $phone_err_msg_billing );
			wc_add_notice( $phone_err_msg_billing, 'error' );
		}

		// Add error notices for shipping phone.
		if ( ! empty( $phone_err_field_shipping ) && ! $always_allow_checkout ) {
			/* translators: %s: Phone validation error message from client-side validation */
			$phone_err_msg_shipping = sprintf(
				/* translators: %s: Phone validation error message from client-side validation */
				__( 'Phone validation error: %s', 'phone-number-validation' ),
				esc_html( $phone_err_field_shipping )
			);
			$this->plugin->log_info( '[Shipping] Phone number validation - error: ' . $phone_err_msg_shipping );
			wc_add_notice( $phone_err_msg_shipping, 'error' );
		}

		// Check if block checkout is NOT enabled and shipping phone is enabled, and ship to destination is not 'billing_only'.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {

			// Check if the user has opted to ship to a different address.
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$ship_to_different_address = isset( $_POST['ship_to_different_address'] ) && 1 === $_POST['ship_to_different_address'];

			if ( $ship_to_different_address ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
				$shipping_phone = sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ?? '' ) );

				// If the setting requires the shipping phone, validate it.
				if ( $settings->is_make_shipping_phone_required() && empty( $shipping_phone ) && ! $always_allow_checkout ) {
					$empty_shipping_err = __( 'Please enter a valid shipping phone number.', 'phone-number-validation' );
					$this->plugin->log_info( '[Shipping] Phone number validation - error: ' . $empty_shipping_err );
					wc_add_notice( $empty_shipping_err, 'error' );
				}
			}
		}
	}

	/**
	 * Save Shipping Phone Field to Order Meta.
	 *
	 * @param int $order_id Order ID.
	 */
	public function save_shipping_phone_field( int $order_id ): void {
		$settings = Settings::get_instance();

		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) && ! empty( $_POST['shipping_phone'] ) ) {
			$order = wc_get_order( $order_id );
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$order->set_shipping_phone( sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ) ) );
			$order->save();
		}

		// Optionally, save the validated phone numbers if needed.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		if ( ! empty( $_POST['pnv_phone_valid_billing'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			update_post_meta( $order_id, '_billing_phone_valid', sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_billing'] ) ) );
		}

		if ( ! $settings->is_block_checkout_enabled() && $settings->is_shipping_phone_enabled() && 'billing_only' !== get_option( 'woocommerce_ship_to_destination' ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			if ( ! empty( $_POST['pnv_phone_valid_shipping'] ) ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
				update_post_meta( $order_id, '_shipping_phone_valid', sanitize_text_field( wp_unslash( $_POST['pnv_phone_valid_shipping'] ) ) );
			}
		}
	}

	/**
	 * Add plugin settings link on the Plugins page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_plugin_action_links( array $links ): array {
		$settings_url  = admin_url( 'admin.php?page=wc-settings&tab=shipping&section=pnv_settings' );
		$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . __( 'Settings', 'phone-number-validation' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}
}
