<?php
/**
 * Add phone number validation to WooCommerce my-account page.
 *
 * @since      1.0.0
 *
 * @package    Phone_Number_Validation
 * @subpackage Phone_Number_Validation/public
 */

namespace PNV\Phone_Number_Validation\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use PNV\Phone_Number_Validation\Phone_Number_Validation;
use PNV\Phone_Number_Validation\Admin\Settings;

/**
 * Class My_Account
 *
 * Handles account page phone validation.
 */
class My_Account {

	/**
	 * Plugin instance.
	 *
	 * @var Phone_Number_Validation
	 */
	private $plugin;

	/**
	 * Custom field meta keys.
	 *
	 * @var array
	 */
	private $custom_field_meta;

	/**
	 * CDN URLs for intlTelInput assets.
	 *
	 * @var array
	 */
	private $intl_tel_input_urls = array(
		'css'   => PNV_ASSETS_URL . 'vendor/intl-tel-input/css/intlTelInput.css',
		'js'    => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/intlTelInput.min.js',
		'utils' => PNV_ASSETS_URL . 'vendor/intl-tel-input/js/utils.js',
	);

	/**
	 * Constructor.
	 *
	 * @param Phone_Number_Validation $plugin Plugin instance.
	 */
	public function __construct( Phone_Number_Validation $plugin ) {
		$this->plugin            = $plugin;
		$this->custom_field_meta = $this->plugin->get_billing_custom_field_meta();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks(): void {
		add_action( 'woocommerce_customer_save_address_validation', array( $this, 'account_page_validate' ), 10, 3 );
		add_filter( 'woocommerce_shipping_fields', array( $this, 'add_shipping_phone_field_to_account' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Enqueue necessary scripts and styles for My Account page.
	 */
	public function enqueue_scripts(): void {
		// Only enqueue on My Account pages.
		if ( ! is_account_page() ) {
			return;
		}

		$endpoint = WC()->query->get_current_endpoint();
		if ( 'edit-address' !== $endpoint ) {
			return;
		}

		// Enqueue intlTelInput CSS and custom frontend CSS.
		wp_enqueue_style( 'intlTelInput-css', $this->intl_tel_input_urls['css'], array(), '17.0.19' );
		wp_enqueue_style( 'pnv_css-style', $this->plugin->plugin_url() . '/css/frontend.css', array(), $this->plugin->get_version() );

		// Enqueue intlTelInput JS.
		wp_enqueue_script( 'intlTelInput-js', $this->intl_tel_input_urls['js'], array(), '17.0.19', true );

		// Register and enqueue frontend.js with dependencies.
		wp_register_script( 'pnv_js-script', $this->plugin->plugin_url() . '/js/frontend.js', array( 'intlTelInput-js' ), $this->plugin->get_version(), true );

		// Retrieve settings from the Settings class.
		$settings = Settings::get_instance();

		// Determine default country based on user's saved data.
		$default_country = $settings->get_initial_country();
		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			// Fall back to the billing country saved with the address.
			$billing_country = get_user_meta( $user_id, 'billing_country', true );
			if ( ! empty( $billing_country ) ) {
				$default_country = sanitize_text_field( $billing_country );
			}
		}

		$pnv_json = array(
			'phoneValidatorName'    => $this->custom_field_meta['billing_hidden_phone_field'],
			'phoneValidatorErrName' => $this->custom_field_meta['billing_hidden_phone_err_field'],
			'phoneErrorTitle'       => __( 'Phone validation error:', 'phone-number-validation' ),
			'phoneUnknownErrorMsg'  => __( 'is an invalid number.', 'phone-number-validation' ),
			'separateDialCode'      => $settings->separate_dial_code(),
			'validationErrors'      => $this->plugin->get_validation_errors(),
			'defaultCountry'        => $default_country,
			'preferredCountries'    => $settings->get_preferred_countries(),
			'utilsScript'           => $this->intl_tel_input_urls['utils'],
			'parentPage'            => '.woocommerce-MyAccount-content',
			'currentPage'           => 'account',
			'allowDropdown'         => $settings->allow_dropdown(),
			'excludeCountries'      => $settings->get_exclude_countries(),
			'alwaysAllowCheckout'   => $settings->is_always_allow_checkout(),
		);

		// Optionally include user phone.
		$phone = $this->plugin->get_current_user_phone();
		if ( $phone ) {
			$pnv_json['userPhone'] = $phone;
		}

		wp_localize_script( 'pnv_js-script', 'wcPvJson', $pnv_json );
		wp_enqueue_script( 'pnv_js-script' );
	}

	/**
	 * Add Shipping Phone Field to My Account shipping address form.
	 *
	 * @param array $fields Existing shipping fields.
	 * @return array Modified shipping fields.
	 */
	public function add_shipping_phone_field_to_account( array $fields ): array {
		if ( Settings::get_instance()->is_shipping_phone_enabled() ) {
			$fields['shipping_phone'] = array(
				'label'       => __( 'Shipping Phone', 'phone-number-validation' ),
				'placeholder' => _x( 'Enter your shipping phone number', 'placeholder', 'phone-number-validation' ),
				'required'    => false,
				'class'       => array( 'form-row-wide', 'pnv-phone', 'pnv-intl' ),
				'priority'    => 110,
			);
		}

		return $fields;
	}

	/**
	 * Validate the phone number on account page address save.
	 *
	 * @param array  $args         Address fields.
	 * @param int    $user_id      User ID being saved.
	 * @param string $load_address Type of address e.g., billing or shipping.
	 */
	public function account_page_validate( $args, int $user_id, string $load_address ): void {
		// Get the settings instance.
		$settings = Settings::get_instance();

		// If Always Allow Checkout is enabled, skip validation.
		if ( $settings->is_always_allow_checkout() ) {
			return;
		}

		// Determine which address is being validated.
		if ( 'billing' !== $load_address && 'shipping' !== $load_address ) {
			return; // Only validate billing and shipping addresses.
		}

		// Define meta keys based on address type.
		$phone_name     = $this->custom_field_meta['billing_hidden_phone_field'];
		$phone_err_name = $this->custom_field_meta['billing_hidden_phone_err_field'];

		// Adjust meta keys for shipping if needed.
		if ( 'shipping' === $load_address ) {
			$phone_name     = str_replace( 'billing', 'shipping', $phone_name );
			$phone_err_name = str_replace( 'billing', 'shipping', $phone_err_name );
		}

		// Retrieve and sanitize POST data.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_valid_field = isset( $_POST[ $phone_name ] ) ? strtolower( sanitize_text_field( wp_unslash( $_POST[ $phone_name ] ) ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
		$phone_valid_err_field = isset( $_POST[ $phone_err_name ] ) ? trim( sanitize_text_field( wp_unslash( $_POST[ $phone_err_name ] ) ) ) : '';

		if ( 'billing' === $load_address ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$billing_email = isset( $_POST['billing_email'] ) ? sanitize_email( wp_unslash( $_POST['billing_email'] ) ) : '';
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$billing_phone = isset( $_POST['billing_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['billing_phone'] ) ) : '';

			if ( ! empty( $billing_email ) && ! empty( $billing_phone ) && ( empty( $phone_valid_field ) || ! is_numeric( $phone_valid_field ) ) ) {

				if ( ! is_numeric( str_replace( ' ', '', $billing_phone ) ) ) {
					// WooCommerce will handle invalid formats.
					return;
				}

				$ph = explode( ':', $phone_valid_err_field );

				if ( isset( $ph[0] ) ) {
					$ph[0] = '<strong>' . esc_html( $ph[0] ) . '</strong>';
				}

				$phone_err_msg = implode( ':', $ph );

				wc_add_notice( $phone_err_msg, 'error' );
			}
		} elseif ( 'shipping' === $load_address ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles it already.
			$shipping_phone = isset( $_POST['shipping_phone'] ) ? sanitize_text_field( wp_unslash( $_POST['shipping_phone'] ) ) : '';

			if ( ! empty( $shipping_phone ) && ( empty( $phone_valid_field ) || ! is_numeric( $phone_valid_field ) ) ) {

				if ( ! is_numeric( str_replace( ' ', '', $shipping_phone ) ) ) {
					// WooCommerce will handle invalid formats.
					return;
				}

				$ph = explode( ':', $phone_valid_err_field );

				if ( isset( $ph[0] ) ) {
					$ph[0] = '<strong>' . esc_html( $ph[0] ) . '</strong>';
				}

				$phone_err_msg = implode( ':', $ph );

				wc_add_notice( $phone_err_msg, 'error' );
			}
		}
	}
}
