<?php
/**
 * WooCommerce Customer/Order/Coupon Export
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Customer/Order/Coupon Export to newer
 * versions in the future. If you wish to customize WooCommerce Customer/Order/Coupon Export for your
 * needs please refer to http://docs.woocommerce.com/document/ordercustomer-csv-exporter/
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2015-2021, SkyVerge, Inc. (info@skyverge.com)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginFramework\v5_10_2 as Framework;

/**
 * Customer/Order CSV Export Background Export
 *
 * Class that handles performing exports in the background
 *
 * @since 4.0.0
 */
class WC_Customer_Order_CSV_Export_Background_Export extends Framework\SV_WP_Background_Job_Handler {


	/**
	 * Initialize background export handler
	 *
	 * @since 4.0.0
	 */
	public function __construct() {

		$this->prefix   = 'wc_customer_order_export';
		$this->action   = 'background_export';
		$this->data_key = 'object_ids';

		parent::__construct();

		add_action( "{$this->identifier}_job_complete", [ $this, 'finish_export' ] );
		add_action( "{$this->identifier}_job_failed",   [ $this, 'failed_export' ] );
		add_action( "{$this->identifier}_job_deleted",  [ $this, 'delete_export_data' ] );
	}


	/**
	 * Finish the export job
	 *
	 * @since 4.0.0
	 * @param object $job Job instance
	 */
	public function finish_export( $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->finish_export( $job );
	}


	/**
	 * Process a single item from the job
	 *
	 * @since 4.0.0
	 * @param mixed $item
	 * @param object $job related job instance
	 */
	public function process_item( $item, $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->export_item( $item, $job );
	}


	/**
	 * Handle failed  export jobs
	 *
	 * @since 4.0.0
	 * @param object $job Job instance
	 */
	public function failed_export( $job ) {

		wc_customer_order_csv_export()->get_export_handler_instance()->failed_export( $job );
	}


	/**
	 * Delete data & files related to the export
	 *
	 * @since 4.0.0
	 * @param stdClass $job
	 */
	public function delete_export_data( $job ) {

		// Delete exported file generated by the job that was deleted
		// suppress errors as they're not really helpful here
		@unlink( $job->file_path );

		// delete user notices related to the export
		if ( $job->created_by ) {
			wc_customer_order_csv_export()->get_export_handler_instance()->remove_export_finished_notice( $job->id, $job->created_by );
		}
	}


	/**
	 * Gets an export status message.
	 *
	 * @since 4.4.0
	 *
	 * @param string $status status slug
	 * @param stdClass|string|null $export export job ID or object
	 * @return string status message
	 */
	public function get_export_status_message( $status, $export = null ) {

		$message      = '';
		$logs_message = sprintf( __( 'Additional details may be found in the CSV Export %1$slogs%2$s.', 'woocommerce-customer-order-csv-export' ), '<a href="' . admin_url( 'admin.php?page=wc-status&tab=logs' ) . '">', '</a>' );

		switch ( $status ) {

			case 'not-found':
				$message = sprintf( __( 'No export found with id %s. It may have been cancelled during export.', 'woocommerce-customer-order-csv-export' ), $export );
			break;

			case 'failed':
				$message = __( 'Unfortunately, your export failed.', 'woocommerce-customer-order-csv-export' ) . ' ' . $logs_message;
			break;

			case 'transfer-failed':

				if ( is_object( $export ) && $export->method ) {

					$label = wc_customer_order_csv_export()->get_methods_instance()->get_export_method_label( $export->method );

					/* translators: Placeholders: %s - a transfer method name such as Email or FTP */
					$message = sprintf( __( 'Export completed successfully, but the transfer %s failed.', 'woocommerce-customer-order-csv-export' ), $label ) . ' ';

				} else {

					$message = __( 'Export completed successfully, but the transfer failed.', 'woocommerce-customer-order-csv-export' ) . ' ';
				}

				/* translators: Placeholders: %1$s - <a> tag, %2$s - </a> tag */
				$message .= sprintf( __( 'Exported file is available under %1$sExport List%2$s.', 'woocommerce-customer-order-csv-export' ), '<a href="' . admin_url( 'admin.php?page=wc_customer_order_csv_export&tab=export_list' ) . '">', '</a>' ) . ' ' . $logs_message;

			break;
		}

		return $message;
	}

}
